
##
## Copyright (c) 1998-2002, Statistics Research, Bell Labs, Lucent Technologies.
##   All rights reserved.
## 
## This program is a part of the S-Net Project: a distributed data
## analysis computing environment for Internet traffic data.
##
## http://cm.bell-labs.com/stat/InternetTraffic
##

adjustPacSize <- function(size, pacSize.type, target="wirelen") {
  ## target:     wirelen, which includes link-layer (such as ethernet) header
  ##             iplen, which includes upto ip header 
  ## AIX:        "POS/PPP", observed len is always iplen (ip header + tcp header + payload)
  ##             [40,1500] bytes
  ## BL:         "ETH.PAYLOAD", observed len is tcp payload length
  ##             [0,1460] bytes
  ## NLANR:      "ATM/AAL5"
  ##             [40,1500] bytes
  ## COS:        "ATM/AAL5/COS", for nlanr traces + ATM/AAL5 link, normally there are 8 bytes for VC
  ##             multiplex, but none for COS.
  ##             [40,1500] bytes
  ## NZIX2:      "ETH.ETH", DAG ETHER format, from the ether.wlen field
  ##             [64,1518] bytes
  ## OC48(MFN):  "POS" DAG POS format, from the pos. wlen field (with 4 bytes HDLC added to iplen)
  ##             then add 2 or 4 bytes frame checksum and 1 idle byte of 0x7E
  ##             [44,1500] bytes
  ## UNC:        (no pac analysis)
  ## No Change:  "NONE", return the same size as input regardless of target
  
  if(F) {
    for(type in c("ETH.ETH", "ETH.IP", "ETH.PAYLOAD", "ATM/AAL5", "ATM/AAL5/COS", 
                  "POS/PPP", "SIMU")) {
      print(getMinPacSize(type))
      for(target in c("wirelen", "iplen")) {
        print(adjustPacSize(seq(0,1500,by=300), type, target))
      }
    }
  }
  ## browser()
  ## if(is.na(match(pacSize.type, c("ETH.ETH", "ETH.IP", "ETH.PAYLOAD", 
  ##                                "ATM/AAL5", "ATM/AAL5/COS", 
  ##                                "POS/PPP", "SIMU")))){
  ##   stop("wrong in argument pacSize.type \n")
  ## }
  if(is.na(match(target, c("wirelen","iplen")))){
    cat(">>> warning in adjustPacSize(): unknown target", target, "\n")
    stop("wrong in argument target\n")
  }
  if(pacSize.type == "ETH.IP") { ## NOT USED so far
    x <- size
    if(target == "wirelen") {
      size[size<=46] <- rep(46, sum(size<=46))
      x <- size + 38
    }
  } else if(pacSize.type == "ETH.ETH") {
    ## for NZIX2, size has already contain part of the ether header
    x <- size - 18 - 6
    ## the 6 byte padding is not quite right, but just to make the
    ## minimum iplen 40 bytes.
    if(target == "wirelen") {
      x <- size + 20;
    }
  } else if(pacSize.type == "ETH.PAYLOAD") {
    ## BL trace
    size <- size + 40 ## 40 bytes for IP header
    x _ size
    if(target == "wirelen") {
      size[size<=46] <- rep(46, sum(size<=46))
      x <- size + 38
    }
  } else if(pacSize.type == "ATM/AAL5") {
    x _ size
    if(target == "wirelen") {
      x <- ceiling((size + 8)/48)*53
    }
  } else if(pacSize.type == "ATM/AAL5/COS") {
    x _ size
    if(target == "wirelen") {
      x <- ceiling(size/48)*53
    }
    ##for nlanr traces + ATM/AAL5 link, normally there are 8 bytes for VC
    ##multiplex, but none for COS.
  } else if(pacSize.type == "POS/PPP") {
    x _ size
    if(target == "wirelen") {
      size <- size+7
      size[size<=55] <- rep(55,sum(size<=55))
      x <- size+16
    }
  } else if(pacSize.type == "SIMU") {
    ## after nov21, 2001,
    ## simulation has 14 extra bytes in addition to the 40-byte IP/TCP header
    size <- size + 40
    x _ size
    if(target == "wirelen") {
      ## this should be removed once michelle implements an option
      ##   of not adding the 14 byte for running simulation
      x <- size + 14
    }
  } else if(pacSize.type == "POS") {
    ## the result from dagdump is wlen (4-byte-HDLC + iplen)
    ## 
    ## 4-byte-HDLC + iplen + (2 or 4 frame check sum) + (1 idle byte of 0x7E)
    ## 
    x _ size - 4 ## iplen 
    if(target == "wirelen") {
      x _ size + 2 + 1 
    }
  } else if(pacSize.type == "NONE") {
    x _ size ## regardless of wirelen or iplen
  } else {
    cat(">>> warning in adjustPacSize(): unknown pacSize.type", pacSize.type, "\n")
  }
  x
}

getMinPacSize <- function(pacSize.type) {
  ## this used ONLY for getting the 3 proportions on packet sizes
  ## not used very often.
  ## if(is.na(match(pacSize.type, c("ETH.ETH", "ETH.IP", "ETH.PAYLOAD", 
  ##                                "ATM/AAL5", "ATM/AAL5/COS", 
  ##                                "POS/PPP", "SIMU")))){
  ##   stop("wrong in argument pacSize.type \n")
  ## }
  if(pacSize.type=="ETH.ETH" || pacSize.type=="ETH.IP" || pacSize.type=="ETH.PAYLOAD") {
    x <- 84
  } else if(pacSize.type == "ATM/AAL5") {
    x <- 53
  } else if(pacSize.type == "ATM/AAL5/COS") {
    x <- 53
  } else if(pacSize.type == "POS/PPP") {
    x <- 55+16
  } else if(pacSize.type == "SIMU") {
    x <- 40
  } else if(pacSize.type == "POS") {
    x <- 40 ## not important, loosely defined.
  } else if(pacSize.type == "NONE") {
    x <- 40 ## not important, loosely defined.
  } else {
    cat(">>> warning in adjustPacSize(): unknown pacSize.type", pacSize.type, "\n")
  }
  x
}

