
##
## Copyright (c) 1998-2002, Statistics Research, Bell Labs, Lucent Technologies.
##   All rights reserved.
## 
## This program is a part of the S-Net Project: a distributed data
## analysis computing environment for Internet traffic data.
##
## http://cm.bell-labs.com/stat/InternetTraffic
##

##
## $Id: discreteWeibull.R,v 1.1 2002/02/01 23:00:16 dxsun Exp $
## $Source: /packet/CVSRoot/SNet_R/SNet/R/discreteWeibull.R,v $

## Routines that deals with discrete weibull distribution
## Discrete Weibull is defined as the ceiling of the Weibull distribution.
rDiscreteWeibull <- function(n, shape=0.5, scale=1) {
  ## generate random samples
  if(F) {
    rDiscreteWeibull(10, shape=0.5, scale=1)
  }
  ceiling(rweibull(n, shape, scale))
}

dDiscreteWeibull <- function(x, shape=0.5, scale=1) {
  ## prob. distribution of the ceiling of a weibull distribution
  ## scale=scale temter, shape=shape temter
  if(F) {
    dDiscreteWeibull(2, shape=0.5, scale=1)
  }
  ## exp(-((x-1)/scale)^shape) - exp(-(x/scale)^shape)
  pweibull(x, shape, scale) - pweibull(x-1, shape, scale)
}

dDiscreteWeibull.deriv <- function(x, shape=0.5, scale=1) {
  ## derivative of prob. density of the ceiling of a weibull distribution
  ## scale=scale temter, shape=shape temter
  
  if(F) {
    dDiscreteWeibull.deriv(c(2,3), shape=0.5, scale=1)
  }
  deriv.scale <- function(x, shape, scale){
    id <- (x==0)
    d <- rep(0,length(x))
    d[id] <- 0
    d[!id] <- exp(-(x[!id]/scale)^shape) * (x[!id]/scale)^shape *
	(shape/scale)
    d
  }
  deriv.shape <- function(x, shape, scale){
    id <- (x==0)
    d <- rep(0,length(x))
    d[id] <- 0
    d[!id] <- exp(-(x[!id]/scale)^shape) * (-(x[!id]/scale)^shape) *
	log(x[!id]/scale)
    d
  }
  
  if(any(x<=0)) warnings("check value of x, (x>=1) \n")
  n <- length(x)
  d <- matrix(0,n,2)
  d[,1] <- deriv.scale(x-1, shape, scale) - deriv.scale(x, shape, scale)
  d[,2] <- deriv.shape(x-1, shape, scale) - deriv.shape(x, shape, scale)
  drop(d)
}

fitDiscreteWeibull.MLE <- function(x) {
  if(F) {
    x <- rDiscreteWeibull(5000, shape=0.5, scale=2)
    x _ x + round(x*0.1)
    out <- fitDiscreteWeibull.MLE(x)
    out <- weibull.MLE(x)
  }
  obj <- function(te){
    ## -log(likelihood) for observations x from Discrete Weibull.
    shape <- te[1]; scale <- te[2]
    ## x <- get("x", frame=sys.parent()-2)
    ## x <- get("x", inherit=T)
    nx <- table(x)
    x <- as.numeric(names(nx)); names(nx) <- NULL
    px <- dDiscreteWeibull(x, shape, scale)
    out <- -sum(nx*log(px))
    if(out>1e300) out _ 1e300 ## just to avoid Inf
    ## cat("shape scale val: ", shape, scale, out, "\n")
    out
  }
  obj.derivative <- function(te){
    ## derivative of -log(lik) for observation x from discrete weibull.
    shape <- te[1]; scale <- te[2]
    ## x <- get("x", inherit=T)
    ## x <- get("x", frame=sys.parent()-2)
    nx <- table(x)
    x <- as.numeric(names(nx)); names(nx) <- NULL
    px <- dDiscreteWeibull(x, shape, scale)
    dpx <- dDiscreteWeibull.deriv(x, shape, scale)
    val _ -c(t(dpx)%*%(nx/px))
    ## cat("shape scale derivative:", shape, scale, ",", val, "\n")
    val
  }
  
  if(length(unique(x))>2) {
    par0 <- unlist(fitDiscreteWeibull.tailprob.ls(x))
    names(par0) <- NULL
    if(max(abs(par0)) < 1e-5) par0 <- c(mean(x),1)
    cat(">>> starting optim()\n")
    cat(">>> initial,   shape scale val:", par0[1], par0[2], obj(par0), "\n")
    ## out <- nlminb(par0, obj, obj.derivative, lower = c(0, 0), upper = c(Inf, Inf))
    out _ optim(par0, obj, method="L-BFGS-B",
                lower=c(1e-10, 1e-10), upper=c(1e10, 1e10))
    cat(">>> optimized, shape scale val:", out$par[1], out$par[2], obj(out$par), "\n")
    list(shape = out$par[1], scale = out$par[2])
  } else list(shape=NA, scale=NA)
}

fitDiscreteWeibull.tailprob.ls <- function(x) {
  ## fit paramters of DiscreteWeibull using tail prob.
  if(F) {
    x <- rDiscreteWeibull(5000, shape=0.5, scale=2)
    out <-  fitDiscreteWeibull.tailprob.ls(x)
  }
  lst <- tailProb(x, discrete=T)
  n <- length(lst$x)
  fit <- lsfit(log(lst$x)[-n], 
		 log(-log(lst$tp))[-n], wt=lst$p[-n])
  te <- fit$coef; names(te) <- NULL
  list(shape=te[2], scale=exp(-te[1]/te[2]))
}

discreteWeibull.qqplot <- function(y,
                                   label=list(), psfile="", append=F, estimate=list(), ...) {
  if(F) {
    y _ rDiscreteWeibull(1000, shape=0.5, scale=1)
    y _ rDiscreteWeibull(1000, shape=1, scale=2)
    para _ fitDiscreteWeibull.MLE(y)
    discreteWeibull.qqplot(y)
    discreteWeibull.qqplot(y, estimate=para)
    discreteWeibull.qqplot(y, label=list(sub=subStr), psfile="demo.ps")
  }

  mypanel <- function(x, y, estimate) { ## y is already log2(y)
    panel.abline(v = log2(-log(c(0.25, 0.75))), col = 1)
    panel.abline(v = log2(-log(c(0.1, 0.9))), col = 1)
    panel.abline(v = log2(-log(c(0.05, 0.95))), col = 1)

    ## least squre fit estimate 
    if(length(estimate)>0){
      panel.abline(log2(estimate$scale), 1/estimate$shape, col=6)
    } else {
      coef <- lsfit(x,y)$coef
      panel.abline(coef, col=6)
      cat("1/slope=",round(1/coef[2],3),"2^intercept=", 
	  round(2^coef[1],6), "\n")
    }
    ## exponential
    mu <- sum(exp(-2^x))+1
    p <- 1-1/mu
    panel.abline(-log2(-log(p)),1, col=8)
    ## estimate
    ## qqplot
    panel.xyplot(x,y,col=1,cex=2,pch=".") 
  }

  if(is.null(label$sub)) {
    if(length(estimate)==0) {
      para _ fitDiscreteWeibull.MLE(y); para$type <- "MLE"
    } else para _ estimate
    label$sub _ paste(para$type, ":shape=", round(para$shape,2),
		      ", scale=", round(para$scale,2), sep="")
  }

  tp <- tailProb(y, discrete=T)
  y <- log2(tp$x); 
  x <- log2(-log(tp$tp)); 
  plt _ xyplot(y~x, panel=mypanel, estimate=estimate)
  plt _ update.plot.labels(plt,
                           xlab = "log2[-log(P(X>x))]",
                           ylab = "log2(x)",
                           main = "DiscreteWeibull Quantile Plot",
                           label = label)
  plt$main$cex _ plt$sub$cex _ 0.8

  if(psfile!="") {
    trellis.device("postscript", color = T, file = psfile, append=append,
                   paper="letter", bg = "white", first=T)
    print(plt)
    dev.off()
  } 
  plt
}
