
##
## Copyright (c) 1998-2002, Statistics Research, Bell Labs, Lucent Technologies.
##   All rights reserved.
## 
## This program is a part of the S-Net Project: a distributed data
## analysis computing environment for Internet traffic data.
##
## http://cm.bell-labs.com/stat/InternetTraffic
##


##
## $Id: flowplot.R,v 1.2 2002/01/16 22:40:59 dxsun Exp $
## $Source: /packet/CVSRoot/SNet_R/SNet/R/flowplot.R,v $
##

grepMatch _
## returns a logical vector with the same length as text
function(pattern, text) {
  if(F) {
    grep("A", c("ABC", "AAA", "CCC"))
    grepMatch("A", c("ABC", "AAA", "CCC"))
  }
  
  ind _ rep(F, length(text))
  sel _ grep(pattern, as.character(text))
  if(length(sel)>0) ind[sel] _ T
  ind
}

##
## 0 box
## 1 circle
## 2 triangle
## 3 +
## 4 x
## 5 diamond
## 6 up-side-down triangle
## 7 x in box
## 8 x and +
## 9 + in diamond
## 10 + in circle
## 15 solid box

## 2 cyan
## 3 magenta
## 4 green
## 5 orange
## 6 blue
## 7 yellow
## 8 red

## flowplot.COLOR _ c(6,8)
flowplot.COLOR _ c("blue","red")

flowplot _ function(flow, flowSummary=NULL, useKB=F, minGapFrac=0,
                    plotWin=F, relativeSeq=T, ipMask=0, xlim=NULL) {
  
  ## - flow object is a data.frame with each row being a packet
  ##   in the flow. It must have these fields:
  ##   "dur" or "time"
  ##   "dir" "seq" "ack" "len" "flag" "win"
  ##   (optional) "mss"
  ##   (optional) "ip0" "port0" "ip1" "port1"
  
  ## - flowSummary object is an optional argument to supply
  ##   additional information about the flow for labels.
  ##   It is a list with these components:
  ##   "ip0", "port0" "ip1" "port1",
  ##   "mss0", "mss1"

  ## Note: if ("ip0", "port0" "ip1" "port1") is avaiable from flowSummary,
  ##       use flowSummary, otherwise, use that from flow object,
  ##       if it's not on both, skip it.
  ##       if ("mss0", "mss1") is avaiable from flowSummary, use it.
  ##       otherwise, if flow has mss field, compute mss0, mss1 from there.
  ##       Only one mss is plotted if mss0 == mss1
  ##       otherwise, both mss0, mss1 are on it.
  ## Options:
  ## useKB=T:         use KBytes for size
  ## minGapFrac=0.01: usr 1/100th of duration range as minimum gap
  ## plotWin=T:      plot advertized window size
  ## relativeSeq=F:  if T, no need to compute relative sequence numbers
  ## ipMask=1:       mask the last byte in ip

  ## Description:
  ## - The red color indicates client->server packets and blue color is for
  ##   server->client.
  ## - x-axis is the time axis
  ## - Each tiny vertical segment represents the payload size of that
  ##   packet, the tiny horizontal bar at the end of the vertical segment is the
  ##   starting relative sequence number of that packet (in KB)
  ## - The circles without the vertical bar represents control packets
  ##   (ACK packet without payload).
  ## - When a packet has both payload and ack number, both vertical bar and
  ##   circle will appear (they are not necessarily connected as the ack
  ##   number corresponds to sequence number from the other direction)
  ## - The S/R/F/P flags of a packet is printed on the bottom on the plot
  ## - The last byte of the IP is masked to avoid identification of
  ##   individual users.
  
  if(F) {
    ## Example:
    flowplot(tcpflow.example1.object,
             relativeSeq=F, minGapFrac=0.01, ipMask=0, useKB=T)

    dir _ "/n/data/sdg1/flowComp.00Mar01-00Apr06/Tue_Apr_4_06_00_00_2000"
    read.tcppacS.flowid("2000.04.04.h07.tcppacS.gz", dir)
    read.flow("2000.04.04.h07.flow.gz", dir, type="flow")
    pac _ tcppacS.2000.04.04.h07
    flow _ flow.2000.04.04.h07
    flowids _ seq(nrow(flow))[flow$port0==80 &
                              flow$fsize0<=10000 &
                              flow$fsize0>=2000
                              ][1:10]
    trellis.device("postscript", file="demo.flowplot.ps", color=T, horizontal=T)
    for(i in flowids) {
      if(flow$port0[i]==80) {
        data _ pac[pac$flowid == i,]
        print(flowplot(data, minGapFrac=0.01))
      }
    }
    dev.off()
  }
  
  cat("$Id: flowplot.R,v 1.2 2002/01/16 22:40:59 dxsun Exp $\n");
  cat(date(), "\n")

  obj _ flow
 
  ## compute flowSummary if it does not exist
  if(is.null(flowSummary)) {
    flowSummary _ list()
    if(!is.null(obj$ip0) && !is.null(obj$port0) &&
       !is.null(obj$ip1) && !is.null(obj$port1)) {
      flowSummary$ip0 _ obj[1,"ip0"]
      flowSummary$port0 _ obj[1,"port0"]
      flowSummary$ip1 _ obj[1,"ip1"]
      flowSummary$port1 _ obj[1,"port1"]
    }
  }
  if(is.null(flowSummary$mss0) && is.null(flowSummary$mss1)) {
    if(!is.null(obj$mss)) {
      ind _ grepMatch("S", obj$flag) & obj$dir == 0; sum(ind)
      if(any(ind)) flowSummary$mss0 _ obj$mss[ind][1]
      ind _ grepMatch("S", obj$flag) & obj$dir == 1; sum(ind)
      if(any(ind)) flowSummary$mss1 _ obj$mss[ind][1]
    }
  }

  ## plot advertised window size
  if(is.null(obj$win)) {
    plotWin _ F
    cat(">>> obj$win does not exist, cannot plot advertised window size\n")
  }
  
  ## - use "dur" if it exists, otherwise compute "dur" from "time"
  if(is.null(obj$dur)) {
    if(is.null(obj$time)) stop("must have at least time or dur")
    obj$dur _ obj$time - min(obj$time)
  }
  obj _ obj[order(obj$dur),]

  if(!is.null(xlim)) {
    ind _ ( obj$dur >= min(xlim) & obj$dur <= max(xlim) )
    if(sum(ind)>1) obj _ obj[ind,]
  }

  ## - figure out minimum gap between packets
  if(minGapFrac > 0 && nrow(obj) > 1) {
    ## 0. adjust starting time to maintain minimum gap to be
    ##    at least as big as 1/50 of the range
    dur.range.orig _ range(obj$dur)
    minGap _ diff(range(obj$dur)) * minGapFrac; minGap
    cat(">>> minGapFrac:", minGapFrac, " minGap:", minGap, "\n")
    dur.diff _ diff(obj$dur)
    dur.diff[ dur.diff < minGap ] _ minGap
    obj$dur _ cumsum(c(obj$dur[1], dur.diff))
    ## 0.1 re-adjust the range of duration to match the original range
    obj$dur _ obj$dur/(diff(range(obj$dur))) * diff(dur.range.orig)
  }

  if( ! relativeSeq ) {
    ## 1. compute relative seq number
    obj$seqR _ rep(NA, length(obj$seq))
    ind0 _ obj$dir==0
    ind1 _ obj$dir==1
    
    if(0) {
      obj$seqR <- obj$seq
      obj$seqR[ind0] _ obj$seqR[ind0] - min(obj$seq[ind0][obj$seq[ind0]!=0])
      obj$seqR[ind1] _ obj$seqR[ind1] - min(obj$seq[ind1][obj$seq[ind1]!=0])
    }
    valid.ind _ grepMatch("S", obj$flag) | grepMatch("F", obj$flag) | obj$len > 0
    ind _ valid.ind & ind0
    obj$seqR[ind] _ obj$seq[ind] - min(obj$seq[ind0 & grepMatch("S", obj$flag)])
    ind _ valid.ind & ind1
    obj$seqR[ind] _ obj$seq[ind] - min(obj$seq[ind1 & grepMatch("S", obj$flag)])
    
    ## 1.2 compute relative ack number
    obj$ackR _ rep(NA, length(obj$ack))
    valid.ind _ grepMatch("A", obj$flag)
    ind _ ind0 & valid.ind
    obj$ackR[ind] _ obj$ack[ind] - min(obj$seq[ind1 & grepMatch("S", obj$flag)])
    ind _ ind1 & valid.ind
    obj$ackR[ind] _ obj$ack[ind] - min(obj$seq[ind0 & grepMatch("S", obj$flag)])
  } else {
    obj$seqR _ obj$seq
    obj$ackR _ obj$ack

    ## fix a problem of relative ack number of the ACK for the server SYN/ACK
    ind0 _ obj$dir==0
    ind1 _ obj$dir==1
    seqInit0 _ min(obj$seq[ind0 & grepMatch("S", obj$flag)])
    seqInit1 _ min(obj$seq[ind1 & grepMatch("S", obj$flag)])

    ind _ ind0 & obj$seq > seqInit0
    obj$seqR[ind] _ obj$seq[ind] - seqInit0
    ind _ ind1 & obj$seq > seqInit1
    obj$seqR[ind] _ obj$seq[ind] - seqInit1

    ind _ ind0 & obj$ack > seqInit1
    obj$ackR[ind] _ obj$ack[ind] - seqInit1
    ind _ ind1 & obj$ack > seqInit0
    obj$ackR[ind] _ obj$ack[ind] - seqInit0
    
    obj$seqR[grepMatch(c("S","R"), obj$flag)] _ 0
    obj$ackR[grepMatch(c("S","R"), obj$flag)] _ 0
  }

  ## 
  if(0) {
    ind _ obj$seq == 0; obj$seqR[ind] _ obj$seq[ind]
    ind _ obj$ack == 0; obj$ackR[ind] _ obj$ack[ind]
  }
  obj$y0 _ obj$seqR; obj$y1 _ obj$seqR + obj$len

  ## 1.3 use Bytes or KB ?
  if(missing(useKB)) {
    if(max(obj$seqR + obj$len, na.rm=T) > 999) useKB _ T
    else useKB _ F
  }
  if(useKB) {
    obj$y0 _ obj$y0/1000; obj$y1 _ obj$y1/1000
    obj$ack _ obj$ack/1000; obj$seq _ obj$seq/1000; obj$len _ obj$len/1000;
    obj$ackR _ obj$ackR/1000; obj$seqR _ obj$seqR/1000; 
    if(!is.null(obj$mss)) obj$mss _ obj$mss/1000
  } 

  sub _ ""
  main _ ""
  main _ paste(main, "TCP flow plot ")

  str _ ""
  if(!is.null(flowSummary) &&
     !is.null(flowSummary$ip0) &&
     !is.null(flowSummary$port0) &&
     !is.null(flowSummary$ip1) &&
     !is.null(flowSummary$port1)
     ) {
    str _ paste("port", flowSummary$port0, "-",
                "port", flowSummary$port1)
    if(F) {
      ip0.str _ unix("awk -F'.' -v OFS='.' '{print $1,$2,$3}'",
                     as.character(flowSummary$ip0))
      ip1.str _ unix("awk -F'.' -v OFS='.' '{print $1,$2,$3}'",
                     as.character(flowSummary$ip1))
    }
    ## ip0.str _ splitString(as.character(flowSummary$ip0), sep=".")
    ## ip1.str _ splitString(as.character(flowSummary$ip1), sep=".")
    ip0.str _ unlist(strsplit(as.character(flowSummary$ip0), "\\."))
    ip1.str _ unlist(strsplit(as.character(flowSummary$ip1), "\\."))

    if(ipMask>0) ip0.str[4:(4+1-ipMask)] _ "x"
    ip0.str _ paste(ip0.str, collapse=".")
    if(ipMask>0) ip1.str[4:(4+1-ipMask)] _ "x"
    ip1.str _ paste(ip1.str, collapse=".")
    str _ paste(ip0.str, " (", flowSummary$port0, ") - ",
                ip1.str, " (", flowSummary$port1, ")", sep="")
  }
  sub _ paste(sub, str)

  ## Draw mss in title label
  if(!is.null(flowSummary) &&
     !is.null(flowSummary$mss0) &&
     !is.null(flowSummary$mss1)) {
    str _ ""
    if(!is.null(flowSummary)) {
      if(flowSummary$mss0 == flowSummary$mss1) {
        str _ paste(str, "mss= ", flowSummary$mss0, sep="")
      } else {
        if(!is.null(flowSummary$mss0)) {
          str _ paste(str, "mss0= ", flowSummary$mss0, sep="")
        }
        if(!is.null(flowSummary$mss1)) {
          str _ paste(str, ", mss1= ", flowSummary$mss1, sep="")
        }
      }
    }
    str _ paste("(", str, ")", sep="")
    sub _ paste(sub, str)
  }
  
  if(F) {
    ## if mss is available, scale obj$win
    if(!is.null(flowSummary) &&
       !is.null(flowSummary$mss0) &&
       !is.null(flowSummary$mss1)) {
      if(!is.null(obj$win)) {
        obj$win _ obj$win/min(flowSummary$mss0,flowSummary$mss1)
      }
    }
  }
  
  ## timestamp
  ## str _ paste("[", timestamp.EST(obj$time[1]), "] ", sep="")
  str _ (structure(0, class = c("POSIXt", "POSIXct"))+obj$time[1])
  str _  paste("[", str, "] ", sep="")
  main _ paste(main, str, sep=" ")
  
  cat('>>> sub: "', sub, '"\n')
  cat('>>> main: "', main, '"\n')
  
  ## panel function
  panel.flowplot _ function(x, y, extraData, ...) {
    cat(">>> panel.flowplot: names(list(...))\n")
    print(names(list(...)))
    cat(">>> panel.flowplot: names(extraData$obj)\n")
    print(names(extraData$obj))
    obj _ extraData$obj
    ind0 _ obj$dir==0
    ind1 _ obj$dir==1

    textLabelAdj _ 0.5
    ## plot non-zero packets
    if(F) {
      hBarWidth _ 3*abs(min(diff(obj$dur)))
      hBarWidth _ abs(min(diff(obj$dur)))/5
      hBarWidth _ min(abs(min(diff(obj$dur)[diff(obj$dur)>0]))/2,
                      diff(range(obj$dur))/50)
    }
    hBarWidth _ diff(range(obj$dur))/500
    ## cat(">>> hBarWidth=", hBarWidth, "\n")

    ## par.usr _ par("usr")
    cv _ current.viewport()
    par.usr _ c(cv$xscale, cv$yscale)
    ## 3. Draw flags
    if(!extraData$opt$plotWin) {
      yloc.baseline _ par.usr[3]
    } else {
      yloc.baseline _ par.usr[3] * 0.45
    }
    ## yloc _ sum(c(yloc.baseline,0)*c(2/3,1/3)) ## par()$cxy[2]
    yloc _ sum(c(yloc.baseline,0)*c(1/2,1/2)) ## par()$cxy[2]
    for(syb in c("S", "F", "R")) {
      for(dir in 0:1) { ## dir _ 0
        ind _ grepMatch(syb, obj$flag) & obj$dir == dir
        if(any(ind)) {
          ##text(x=obj$dur[ind], y=rep(yloc, sum(ind)), labels=rep(syb,sum(ind)),
          ##     col=flowplot.COLOR[obj$dir[ind]+1], adj=textLabelAdj, cex=.5)
          grid.text(rep(syb,sum(ind)), x=obj$dur[ind], y=rep(yloc, sum(ind)),,
                    default.units="native",
                    gp=gpar(col=flowplot.COLOR[dir+1]))
          
          if(F) { ## draw dotted vertical lines
            segments(obj$dur[ind], rep(0, sum(ind)),
                     obj$dur[ind], rep(max(obj$y1,na.rm=T), sum(ind)), 
                     col=flowplot.COLOR[obj$dir[ind]+1], lty=2)
          }
        }
      }
    }
    
    ## 4. Draw mss if avaialble
    if(F) {
      ## do text label instead
      if(!is.null(obj$mss)) {
        ind _ grepMatch("S", obj$flag); sum(ind)
        if(any(ind)) {
          segments(obj$dur[ind], rep(0, sum(ind)),
                   obj$dur[ind], obj$mss[ind], 
                   col=flowplot.COLOR[obj$dir[ind]+1])
        }
      }
    }

    ## 5. Draw plot PSH flag
    if(F) {
      yloc _ sum(c(yloc.baseline,0)*c(1/3,2/3))## par()$cxy[2]
      for(syb in c("P", "U")) {
        ind _ grepMatch(syb, obj$flag)
        if(any(ind)) {
          text(obj$dur[ind], rep(yloc, sum(ind)), syb,
               flowplot.COLOR[obj$dir[ind]+1], adj=textLabelAdj, cex=.5)
        }
      }
    }
    
    ## 6. draw two horizontal dotted lines
    panel.abline(h=0, lty=2, col="black")
    if(F) panel.abline(h=max(obj$y1, na.rm=T), lty=2, col=4)

    ## 2. Draw acks for two directions
    yMax _ par.usr[4]
    ## ind _ ind0 & !grepMatch("S", obj$flag)
    for(dir in 0:1) {
      ## ind _ ind0
      ind _ obj$dir==dir
      if(F) {
        segments(obj$dur[ind], obj$ackR[ind], obj$dur[ind], rep(yMax, sum(ind)),
                 col=flowplot.COLOR[1], lty=2)
        segments(obj$dur[ind]-hBarWidth, obj$ackR[ind],
                 obj$dur[ind]+hBarWidth, obj$ackR[ind],
                 col=flowplot.COLOR[1])
      }
      ## if(any(ind)) points(obj$dur[ind], obj$ackR[ind], col=flowplot.COLOR[1], pch=1,cex=0.3)
      if(any(ind)) grid.points(obj$dur[ind], obj$ackR[ind], pch="o",
                               default.units = "native",
                               gp=gpar(col=flowplot.COLOR[dir+1], fontsize=8))
      ## ind _ ind1 & !grepMatch("S", obj$flag)

      ## 1. Data packets, vertical segments
      ## ind _ ind0 & !grepMatch("S", obj$flag) & obj$len > 0
      ## ind _ ind0 & !grepMatch("S", obj$flag) & !is.na(obj$y0) & !is.na(obj$y1) 
      ind _ obj$dir==dir & !grepMatch("S", obj$flag) & !is.na(obj$y0) & !is.na(obj$y1) & obj$len > 0
      if(any(ind)) {
        ##segments(obj$dur[ind], obj$y0[ind], obj$dur[ind], obj$y1[ind], col=flowplot.COLOR[1],
        ##         lwd=2)
        ##segments(obj$dur[ind]-hBarWidth, obj$y0[ind],
        ##         obj$dur[ind]+hBarWidth, obj$y0[ind], col=flowplot.COLOR[1])
        grid.segments(obj$dur[ind], obj$y0[ind], obj$dur[ind], obj$y1[ind],
                      default.units="native", gp=gpar(col=flowplot.COLOR[dir+1], lwd=2))
        grid.segments(obj$dur[ind]-hBarWidth, obj$y0[ind],
                      obj$dur[ind]+hBarWidth, obj$y0[ind],
                      default.units="native", gp=gpar(col=flowplot.COLOR[dir+1]))
      }
    }
 
    ## draw advertised window size
    if(extraData$opt$plotWin) {
      yloc0 _ par.usr[3]
      yloc1 _ yloc.baseline
      ## win.y _ obj$win
      ## win.y _ win.y/max(win.y,na.rm=T)
      ## win.y _ (win.y-min(win.y,na.rm=T))/diff(range(win.y,na.rm=T))
      ## win.y

      win.size.scale _ function(win.y) {
        ## a rather strange scaling
        if( length(unique(win.y)) == 1 ) {
          ## if all win size are the same, just need to show that
          ## it is constant across time
          win.y _ win.y/win.y[1]*.5
        } else {
          ## if win sizes are all very large, it is hard to see
          ## the differences, unless we remove a big constant from it
          big.const _ max(0, min(win.y,na.rm=T)-diff(range(win.y,na.rm=T))/4)
          win.y _ (win.y-big.const)
          win.y _ win.y/max(win.y,na.rm=T)
        }
        win.y
      }
      
      if(sum(ind0)>0) {
        win.y _ win.size.scale(obj$win[ind0])
        ## print(win.y)
        segments(as.numeric(x[ind0]), rep(yloc0, length(x))[ind0],
                 as.numeric(x[ind0]), (yloc0+win.y*(yloc1-yloc0)), col=flowplot.COLOR[1])
      }
      if(sum(ind1)>0) {
        win.y _ win.size.scale(obj$win[ind1])
        ## print(win.y)
        segments(as.numeric(x[ind1]), rep(yloc0, length(x))[ind1],
                 as.numeric(x[ind1]), (yloc0+win.y*(yloc1-yloc0)), col=flowplot.COLOR[2])
      }
      ##text(par.usr[2]+par("1em")[1]/4,
      ##     (yloc0+yloc1)/2, "adv win size", adj=0)
      ## panel.abline(h=yloc.baseline, col=1)
      box()
    }
  }

  ylab _ "Size (byte)"
  if(useKB) ylab _ "Size (KB)"


  ylim _ c(min(obj$y1,na.rm=T)-diff(range(obj$y1,na.rm=T))/50,
           max(obj$y1,na.rm=T))
  if(F) {
    ## don't plot mss any more.
    if(!is.null(obj$mss)) {
      ylim _ range(ylim, obj$mss[grepMatch("S", obj$flag)])
    }
  }
  if(plotWin) {
    ylim _ c(ylim[1]-diff(ylim)/20, ylim[2])
  }
  cat(">>> ylim:", ylim, "\n")

  
  plt _ xyplot(y1 ~ dur, data=obj, type="n",
               xlab="Time (sec)", ylab=ylab,
               ylim=ylim,
               ## main="TCP flow plot", sub=sub,
               main=main, sub=sub,
               extraData=list(obj=obj,opt=list(plotWin=plotWin)),
               panel=panel.flowplot)
  if(F) {
    plt$key _ list(text=c("client -> server", "server -> client"),
                   lines=list(lty=c(1,1), col=flowplot.COLOR[c(2,1)]),
                   columns=2)
  }
  plt
}

## source("/home/dxsun/Project/Packet/Src/Slib/flowplot.S")
