
##
## Copyright (c) 1998-2002, Statistics Research, Bell Labs, Lucent Technologies.
##   All rights reserved.
## 
## This program is a part of the S-Net Project: a distributed data
## analysis computing environment for Internet traffic data.
##
## http://cm.bell-labs.com/stat/InternetTraffic
##

run.intvl.pac.ia.marg _ function(obj, tag) {
  cat(">>> run.intvl.pac.ia.marg()\n")
  tag _ update.tag.runName(tag, "pac.ia.marg")
  if(nrow(obj)>SNet.settings$min.nrow){
    time _ obj$time - obj$time[1]
    ia _ InterArrival(time)
    tag$weibull.estimate _ weibull.estimate(ia, type="MLE")
    ## tag$weibull.estimate _ weibull.estimate(ia, type="ME")
    ## 1.
    log2.time.plot(time, ia,
                   label=list(main=tag$main,
                     ylab="Log2(Packet Inter Arrival Time)"),
                   psfile=paste(tag$psPrefix, "time.plot", tag$psSuffix, sep="."))
    ## 2.
    weibull.qqplot(ia, 
		   label=list(main=tag$main, 
		       ylab="Log2 Packet Inter Arrival Time"),
		   psfile=paste(tag$psPrefix, "weibull.qqplot", tag$psSuffix, sep="."),
		   estimate=tag$weibull.estimate)
    unif.qqplot(log2(ia), 
		label=list(main=tag$main, 
		    ylab="Log2 Packet Inter Arrival Time"),
		psfile=paste(tag$psPrefix, "unif.qqplot", tag$psSuffix, sep="."))

  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.ia.spec _ function(obj, tag) {
  pac.ia.transform _ tag$pac.ia.transform
  cat(">>> run.intvl.pac.ia.spec():", pac.ia.transform, "\n")
  tag _ update.tag.runName(tag, paste("pac.ia.spec.", pac.ia.transform, sep=""))
  if(nrow(obj)>SNet.settings$min.nrow) {
    time _ obj$time - obj$time[1]
    ia _ InterArrival(time)
    ia[ia<=0] <- rep(1e-6,sum(ia<=0)) ## zero inter-arrival time --> 1usec
    ## y.tr _ gaussianize(ia)                
    if(pac.ia.transform == "gaus") {
      y.tr _ gaussianize(ia)   
    } else if(pac.ia.transform == "std.log") {
      y.tr _ standardize(log(ia))
    } else if(pac.ia.transform == "std.pow") {
      y.tr _ standardize((ia)^{1/6})
    } else {
      stop(paste("unknown pac.ia.transform <", pac.ia.transform, ">\n", sep=""))
    }
    cat("using", pac.ia.transform, "\n")
    print(c(mean(y.tr), var(y.tr)))
    tag _ template.spec.fit(y.tr, tag)
  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.size.marg _ function(obj, tag) {
  cat(">>> run.intvl.pac.size.marg()")
  tag _ update.tag.runName(tag, "pac.size.marg");
  if(nrow(obj)>SNet.settings$min.nrow){
    time _ obj$time -obj$time[1]
    ## pacSize _ obj$len ## wire size
    pacSize _ obj$iplen ## packet size upto ip header
    ## 1.
    time.plot(time, pacSize, label=list(main=tag$main, ylab="pacSize/iplen"),
	      psfile=paste(tag$psPrefix, "time.plot", tag$psSuffix, sep="."))
    ## 2.
    unif.qqplot(pacSize, label=list(main=tag$main, ylab="pacSize/iplen"),
		psfile=paste(tag$psPrefix, "unif.qqplot", tag$psSuffix, sep="."))
    ## 5. compute the minsize, intermediate-size, and max size proportions
    ## minPacSize _ getMinPacSize(pacSize.type)
    minPacSize _ getMinPacSize(tag$data.info.aux$pacSize.type[1])
    maxPacSize _ 1500
    prop <- list(minSize.prob=mean(pacSize<=minPacSize), 
                 intmedSize.prob=mean(pacSize<maxPacSize & pacSize>minPacSize), 
                 maxSize.prob=mean(pacSize>=maxPacSize),
                 meanSize=mean(pacSize))
    tag$sizestat <- prop
    tag$samp <- template.sample(pacSize)
  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.size.spec _ function(obj, tag) {
  cat(">>> run.intvl.pac.size.spec()")
  tag _ update.tag.runName(tag, "pac.size.spec")
  
  if(nrow(obj)>SNet.settings$min.nrow) {
    pacSize _ obj$len ## wire size
    if(diff(range(pacSize))>0) y.tr _ standardize(pacSize) ## was pacSize.tr
    else y.tr _ rep(0,length(pacSize))
    tag _ template.spec.fit(y.tr, tag)
  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.bandwidth _ function(obj, tag) {
  cat(">>> run.intvl.pac.bandwidth()\n")
  tag _ update.tag.runName(tag, "pac.bandwidth")
  
  TIMESTAMP.OPTION _ tag$data.info.aux$TIMESTAMP.OPTION
  LINKSPEED _ tag$data.info.aux$LINKSPEED
  pacSize.type _ tag$data.info.aux$pacSize.type
  minPacSize _ getMinPacSize(pacSize.type)
  STREAMVAR = tag$STREAMVAR

  ## back to back fractions
  if(nrow(obj)>SNet.settings$min.nrow) {
    tag$frac.back2back <- 
      list(minpac.span1=frac.back2back(obj, linkspeed=LINKSPEED, 
	       method="minpac", minPacSize=minPacSize, streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION),
	   allpac.span1=frac.back2back(obj, linkspeed=LINKSPEED, 
	       method="allpac", streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION),
	   minpac.span3=frac.back2back(obj, SPAN=3, 
	       linkspeed=LINKSPEED, method="minpac", minPacSize=minPacSize, 
	       streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION),
	   allpac.span3=frac.back2back(obj, SPAN=3,
	       linkspeed=LINKSPEED, method="allpac",
	       streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION),
	   minpac.span5=frac.back2back(obj, SPAN=5,
	       linkspeed=LINKSPEED, method="minpac", minPacSize=minPacSize, 
	       streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION),
	   allpac.span5=frac.back2back(obj, SPAN=5,
	       linkspeed=LINKSPEED, method="allpac",
	       streamvar=STREAMVAR,
	       TIMESTAMP.OPTION=TIMESTAMP.OPTION)
	   )

    ## bandwidth plots
    include.bandwidth.plot(obj, tag, SPAN=1, PACSIZE=0)
    include.bandwidth.plot(obj, tag, SPAN=3, PACSIZE=0)
    include.bandwidth.plot(obj, tag, SPAN=5, PACSIZE=0)
    include.bandwidth.plot(obj, tag, SPAN=1, PACSIZE=200)
  
    ## ia.vs.size plots
    include.ia.vs.size.plot(obj, tag, SPAN=1)
    include.ia.vs.size.plot(obj, tag, SPAN=3)
    include.ia.vs.size.plot(obj, tag, SPAN=5)
  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.cnt.pkt.marg _ function(obj, tag) {
  cat(">>> run.intvl.pac.cnt.pkt.marg():\n");
  tag _ update.tag.runName(tag, paste("pac.cnt.pkt.marg.", tag$timeunit*1000, "ms", sep=""));
  run.intvl.pac.cnt.marg(obj, tag, cnt.type="pkt")
}
run.intvl.pac.cnt.byt.marg _ function(obj, tag) {
  cat(">>> run.intvl.pac.cnt.byt.marg():\n");
  tag _ update.tag.runName(tag, paste("pac.cnt.byt.marg.", tag$timeunit*1000, "ms", sep=""));
  run.intvl.pac.cnt.marg(obj, tag, cnt.type="byt")
}
run.intvl.pac.cnt.marg _ function(obj, tag, cnt.type) {
  cat(">>> run.intvl.pac.cnt.marg():", cnt.type, "\n");
  ## need two variables: obj, tag
  if(cnt.type == "pkt") cnt.name _ "Packet Count"
  else if(cnt.type == "byt") cnt.name _ "Byte Count"
    
  if(nrow(obj)>SNet.settings$min.nrow) {
    time _ obj$time - obj$time[1]
    cnt _ obj[[cnt.type]]
    ## time plot
    log2.time.plot(time, cnt+1, 
		   label=list(main=tag$main,
		       xlab="Flow Arrival Time",
		       ylab=paste("Log Base 2 ", cnt.name, sep="")),
		   psfile=paste(tag$psPrefix, "log2.time.plot", tag$psSuffix, sep="."))
    time.plot(time, cnt, label=list(main=tag$main,
				xlab="Flow Arrival Time",
				ylab=cnt.name),
	      psfile=paste(tag$psPrefix, "time.plot", tag$psSuffix, sep="."))
  
    ## marginal plot
    log2normal.qqplot(cnt+1, 
		      label=list(main=tag$main,
			  xlab="Normal Quantile",
			  ylab=paste("Log Base 2 ", cnt.name, sep="")),
		      psfile=paste(tag$psPrefix, "log2normal.qqplot", tag$psSuffix, sep="."))
    tag$log2normal.estimate _ list(mean=mean(log2(cnt+1)), var=var(log2(cnt+1)))
    normal.qqplot(cnt, label=list(main=tag$main,
			      xlab="Normal Quantile",
			      ylab=cnt.name),
		  psfile=paste(tag$psPrefix, "normal.qqplot", tag$psSuffix, sep="."))
    tag$normal.estimate _ list(mean=mean((cnt)), var=var((cnt)))
    tag$sdmean.ratio _ sqrt(var(cnt))/mean(cnt)

    ## unif
    unif.qqplot(log2(cnt+1), 
		label=list(main=tag$main, 
		    xlab="Uniform Quantile",
		    ylab=paste("Log Base 2 ", cnt.name, sep="")),
		psfile=paste(tag$psPrefix, "unif.qqplot", tag$psSuffix, sep="."))
  }
  include.assign.result(tag)
  tag$runName
}

run.intvl.pac.cnt.pkt.spec _ function(obj, tag) {
  cat(">>> run.intvl.pac.cnt.pkt.spec():\n")
  tag _ update.tag.runName(tag, paste("pac.cnt.pkt.spec.", tag$timeunit*1000, "ms",
                                      ".", tag$pac.cnt.transform, sep=""))
  run.intvl.pac.cnt.spec(obj, tag, cnt.type="pkt")
}
run.intvl.pac.cnt.byt.spec _ function(obj, tag) {
  cat(">>> run.intvl.pac.cnt.byt.spec():\n")
  tag _ update.tag.runName(tag, paste("pac.cnt.byt.spec.", tag$timeunit*1000, "ms",
                                      ".", tag$pac.cnt.transform, sep=""))
  run.intvl.pac.cnt.spec(obj, tag, cnt.type="byt")
}
run.intvl.pac.cnt.spec _ function(obj, tag, cnt.type) {
  pac.cnt.transform _ tag$pac.cnt.transform
  cat(">>> run.intvl.pac.cnt.spec():", cnt.type, pac.cnt.transform, "\n");
    
  if(nrow(obj)>SNet.settings$min.nrow) {
    time _ obj$time - obj$time[1]
    cnt _ obj[[cnt.type]]

    if(pac.cnt.transform == "gaus") {
      y.tr _ gaussianize(cnt)   
    } else if(pac.cnt.transform == "std.log") {
      y.tr _ standardize(log(cnt))
    } else if(pac.cnt.transform == "std") {
      y.tr _ standardize(cnt)
    } else {
      stop(paste("unknown pac.cnt.transform <", pac.cnt.transform, ">\n", sep=""))
    }
    tag _ template.spec.fit(y.tr, tag)
  }
  include.assign.result(tag)
  tag$runName
}

## summary functions
summary.intvl.pac.ia.marg _ function(settings) {
  cat(">>> summary.intvl.pac.ia.marg()\n")
  ## taskid _ settings$taskid
  ## runName _ paste(taskid, "pac.ia.marg", sep=".")
  vars <- c("shape","scale")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.ia.spec _ function(settings) {
  cat(">>> summary.intvl.pac.ia.spec()\n")
  ## taskid _ settings$taskid
  ## runName _ paste(taskid, "pac.ia.spec", sep=".")
  vars _ c("d","varRatio", 
	   "etp", "etp.fit","geweke.smooth.etp.fit", 
	   "twostep.pred.err", "twostep.pred.err.fit",
	   "geweke.smooth.twostep.pred.err.fit")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.size.marg _ function(settings) {
  cat(">>> summary.intvl.pac.size.marg()\n")
  ##  taskid _ settings$taskid
  ## runName <- paste(taskid, "pac.size.marg", sep=".")
  vars _ c("minSize.prob","intmedSize.prob", "maxSize.prob", "meanSize")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.size.spec _ function(settings) {
  cat(">>> summary.intvl.pac.size.spec()")
  ## taskid _ settings$taskid
  ## runName <- paste(taskid, "pac.size.spec", sep=".")
  vars _ c("d","varRatio", "etp", "etp.fit","geweke.smooth.etp.fit")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.cnt.pkt.marg _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.pkt.marg()\n")
  summary.intvl.pac.cnt.marg(settings)
}
summary.intvl.pac.cnt.byt.marg _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.byt.marg()\n")
  summary.intvl.pac.cnt.marg(settings)
}
summary.intvl.pac.cnt.marg _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.marg()\n")
  ## cnt.type _ settings$cnt.type
  ## taskid _ settings$taskid
  ## taskid <- paste(taskid, ".", settings$timeunit*1000, "ms", sep="")
  ## runName _ paste(taskid, ".", cnt.type, ".marg", sep="")
  vars <- c("log2normal.estimate.mean", "log2normal.estimate.var",
            "normal.estimate.mean","normal.estimate.var", "sdmean.ratio")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.cnt.pkt.spec _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.spec()\n")
  summary.intvl.pac.cnt.spec(settings)
}
summary.intvl.pac.cnt.byt.spec _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.spec()\n")
  summary.intvl.pac.cnt.spec(settings)
}
summary.intvl.pac.cnt.spec _ function(settings) {
  cat(">>> running summary.intvl.pac.cnt.spec()\n")
  ## cnt.type _ settings$cnt.type
  ## taskid _ settings$taskid
  ## taskid <- paste(taskid, ".", settings$timeunit*1000, "ms", sep="")
  ## runName _ paste(taskid, ".", cnt.type, ".spec", sep="")
  vars _ c("d","varRatio", "etp",
	   "etp.fit","geweke.smooth.etp.fit", 
	   "twostep.pred.err", "twostep.pred.err.fit",
	   "geweke.smooth.twostep.pred.err.fit")
  template.summary.intvl(settings$runName, vars, settings)
}
summary.intvl.pac.bandwidth _ function(settings) {
  cat(">>> running summary.intvl.pac.bandwidth()\n")
  ## taskid _ settings$taskid
  ## runName _ paste(taskid, "pac.bandwidth", sep=".")
  vars <- c(t(outer(c("frac.back2back.allpac","frac.back2back.minpac"),
  		     c("span1", "span3","span5"), paste, sep=".")))
  vars <- c(t(outer(vars, c("total", "samestream","diffstream"), paste, sep=".")))
  ##vars <- c(t(outer(c("frac.back2back.allpac","frac.back2back.minpac"),
  ##		     c("total", "samestream","diffstream"), paste, sep=".")))
  template.summary.intvl(settings$runName, vars, settings)
}

update.tag.settings _ function(tag, settings, iter) {
  tag$objnme _ settings$data.info[iter, "objnme"]
  if(settings$subtask.type=="pac") {
    tag$data.info.aux _ settings$data.info[iter,settings$auxVar,drop=T]
  }
  tag$main.part _ tag$intvlLab
  if(!is.null(settings$data.info$flowrate))
    tag$main.part _ paste(tag$main.part, "flowrate=",
                          round(settings$data.info$flowrate[iter],2))
  if(!is.null(settings$data.info$pacrate))
    tag$main.part _ paste(tag$main.part, "pacrate=",
                          round(settings$data.info$pacrate[iter],2))
  if(!is.null(settings$data.info$streamrate))
    tag$main.part _ paste(tag$main.part, "streamrate=",
                          round(settings$data.info$streamrate[iter],2))
  
  ## extra info,
  ## do not put data.info (which is a matrix of all intervals) on tag.
  ## tag is for this interval only
  settings$data.info _ NULL 
  tag _ updateList(tag, settings)
  tag
}

update.tag.runName _ function(tag, outName) {
  ## outName: e.g., "pac.ia.marg"
  runName _ paste(tag$taskid, outName, sep=".")
  tag$runName _ runName 
  tag$main _ paste(tag$main.part, tag$runName)
  tag$main _ paste(tag$main, "\n", tag$objnme, sep="")
  tag$psSuffix _ paste("ps", tag$intvlStr, sep=".")
  tag$psPrefix _ tag$runName
  tag
}
include.assign.result _ function(tag) {
  ## assign results
  addtolist(paste(tag$runName, "compLst", sep="."), tag, tag$intvlStr, sepObj=T)
  NULL
}

template.sample _ function(y, seed=0){
  cat(">>> running template.sample()\n")
  ns <- min(length(y), 200)
  set.seed(seed)
  y[sample(1:length(y), size=ns)]  
  ##this should be the same as sample(y,ns), but sometimes its not
  ##for example, sample(0.21,1) will give an error.
}

template.spec.fit _ function(y.tr, tag) {
  cat(">>> running template.spec.fit()\n")
  fitting.type _ tag$fitting.type
  ## template.spec.fit
  ran <- diff(range(y.tr)); ## ran
  ## spec
  noresult <- T
  if(ran > 0 && length(y.tr) > 30) {
    init _ list(varRatio = .5, d = .45, sigma.lSq = 1)
    ## fit _ fit.log10spec(y.tr, init=init, type = fitting.type)
    fit _ fit.fsd.log10spec.unbiased(y.tr, init=init, type = fitting.type)
    cat(">>> template.spec.fit: after fit.fsd.log10spec.unbiased()\n");
    if(!is.null(fit)) {
      noresult <- F
      cat(">>> template.spec.fit: before geweke.smooth.log10spec()\n");
      fit2 _ geweke.smooth.log10spec(log10spec.obs.lst=fit)
      cat(">>> template.spec.fit: after geweke.smooth.log10spec()\n");
      fit2 _ fit2[c("log10spec.fitted","freqseq.pred",
                    "log10spec.pred","etp.fit", "twostep.pred.err.fit")]
      names(fit2) _ paste("geweke.smooth", names(fit2), sep=".")
      fit _ updateList(fit, fit2)
      for(type in c("fit", "geweke.residual", "fit.residual")) {
        cat(">>> template.spec.fit: before spectrum.plot()\n");
        spectrum.plot(fit, label=list(main=tag$main),
                      psfile=paste(tag$psPrefix,
                        type, "spec.plot", fitting.type, tag$psSuffix, sep="."),
                      type=type)
        cat(">>> template.spec.fit: after spectrum.plot()\n");
      }
      ## drop fields that depend on the number of observations
      ## these fields are only needed for spectrum.plot
      fit _ dropList(fit, c("freqseq", "log10spec.obs.unbiased",
                            "log10spec.fitted", "geweke.smooth.log10spec.fitted"))
      tag$fit _ fit
    }
  }
  if(noresult){
    for(type in c("fit", "geweke.residual", "fit.residual")) {
      empty.plot(label=list(main=list(tag$main, cex=0.8), xlab="Frequency"),
                 psfile=paste(ag$psPrefix,
                   type, "spec.plot", fitting.type, 
                   tag$psSuffix, sep="."))
      tag <- c(tag, list(fit=NULL))
    }
  }
  cat(">>> end of template.spec.fit()\n");
  tag
}

frac.back2back <- function(obj, SPAN.NPAC=1, linkspeed, 
			   method="minpac", minPacSize,
			   streamvar=c("ip0","port0","ip1","port1"),
			   TIMESTAMP.OPTION="pac.end")
{
  if(F){
    index <- 200
    obj <- get(BL.alltcp.samp.df$bigobjnme[index])
    streamvar <- c("ip0","port0","ip1","port1", "dir")
    obj$len <- adjustPacSize(obj$len, "ETH.PAYLOAD")
    frac.back2back(obj=obj, linkspeed=10^8, method="minpac", minPacSize=84,
		   streamvar=streamvar, TIMESTAMP.OPTION="pac.end")*100
    frac.back2back(obj=obj, SPAN=3, linkspeed=10^8, method="allpac",
		   streamvar=streamvar, 
		   TIMESTAMP.OPTION="pac.end")*100
    frac.back2back(obj=obj, linkspeed=10^8, method="allpac",
		   streamvar=streamvar,
		   TIMESTAMP.OPTION="pac.end")*100
  }

  
  if(is.character(obj)) {obj <- get(obj); NULL}
  obj <- obj[order(obj$time),]
  time _ obj$time - obj$time[1]
  ia _ InterArrival(time, TIMESTAMP.OPTION)
  pacSize _ obj$len ## wire size
  if(SPAN.NPAC > 1){
    ia _ movingMean(ia, s=SPAN.NPAC, wrap=T)
    pacSize _ movingMean(pacSize, s=SPAN.NPAC, wrap=T)
    if(method=="minpac") minPacSize _ minPacSize*SPAN.NPAC
  }

  ## transmission time on a 100M link
  ## I am assuming 1 microsec accuracy
  if(method!="minpac" && method!="allpac") stop("wrong in argument method!\n")
  if(method=="minpac") tmthre <- (ceiling((minPacSize*8/linkspeed)*1e6)+1)/1e6
  if(method=="allpac") tmthre <- (ceiling((pacSize*8/linkspeed)*1e6)+1)/1e6
  n <- 0
  id <- (1:nrow(obj))[ia <= tmthre]  
  if(length(id)>0) {
    id <- id[-c(1,length(id))]  #get rid of the first and last one just in case 
    if(TIMESTAMP.OPTION=="pac.start") {id <- id+1; NULL}
    if(!(length(streamvar)==1 && is.na(streamvar[1]))){
      if(!any(is.na(match(streamvar, names(obj))))){
        pconn.1 <- do.call("paste",args = c(obj[id-1, streamvar], list(sep = ":")))
        pconn.2 <- do.call("paste",args = c(obj[id, streamvar], list(sep = ":")))
        n <- sum(pconn.1==pconn.2)
      }
    }
  }
  nr <- nrow(obj)

  list(total=length(id)/nr, samestream=n/nr, diffstream=(length(id)-n)/nr)
}

include.bandwidth.plot _ function(obj, tag=NULL, 
				  SPAN.NPAC=1, PACSIZE.THRE=0) {
  ## need two variables: obj, tag
  ## output in 
  TIMESTAMP.OPTION _ tag$data.info.aux$TIMESTAMP.OPTION
  LINKSPEED _ tag$data.info.aux$LINKSPEED

  time _ obj$time - obj$time[1]
  ia _ InterArrival(time, TIMESTAMP.OPTION)
  pacSize _ obj$len ## wire size
  n <- nrow(obj)
  time _ time[-c(1,n)]; ia _ ia[-c(1,n)]; pacSize _ pacSize[-c(1,n)]

  if(SPAN.NPAC > 1){
    ia _ movingMean(ia, s=SPAN.NPAC, wrap=T)
    pacSize _ movingMean(pacSize, s=SPAN.NPAC, wrap=T)
  }
  ## time plot
  log10.bw _ log10(pacSize*8/ia)
  time.plot(time, log10.bw, 
	    label=list(main=tag$main,
		ylab="log10(bw)"),
	    psfile=paste(tag$psPrefix, ".bw", PACSIZE.THRE,
		".span",SPAN.NPAC,
		".time.plot.", tag$psSuffix, sep=""))
  
  ## bandwidth qunif plot
  if(sum(pacSize>=PACSIZE.THRE) > 0){
    log10.bw _ sort(log10.bw[pacSize>=PACSIZE.THRE])
    unif.qqplot(log10.bw, label=list(main=tag$main, ylab="log10(bw)"),
		psfile=paste(tag$psPrefix, ".bw",PACSIZE.THRE,".span",SPAN.NPAC,
		    ".unif.qqplot.", tag$psSuffix, sep=""))
  }
  NULL
}

include.ia.vs.size.plot <- function(obj, tag, SPAN.NPAC){
  TIMESTAMP.OPTION _ tag$data.info.aux$TIMESTAMP.OPTION
  LINKSPEED _ tag$data.info.aux$LINKSPEED
  
  time _ obj$time - obj$time[1]
  ia _ InterArrival(time, TIMESTAMP.OPTION)
  pacSize _ obj$len ## wire size
  n <- nrow(obj); ia _ ia[-c(1,n)]; pacSize _ pacSize[-c(1,n)]
  if(SPAN.NPAC > 1){
    ia _ movingMean(ia, s=SPAN.NPAC, wrap=T)
    pacSize _ movingMean(pacSize, s=SPAN.NPAC, wrap=T)
  }
  ord <- order(pacSize)
  pacSize <- pacSize[ord]
  ia <- ia[ord]

  plt <- xyplot(log2(ia)~log2(pacSize), 
		linkspeed=LINKSPEED, 
		panel=function(x,y,linkspeed,...){
		  y.limit <- 2^x*8/linkspeed
		  llines(x, log2((ceiling(y.limit*1e6)+1)/1e6), col="cyan", lwd=2)
		  llines(x, log2(y.limit), col="magenta", lwd=2)
		  llines(x, log2((floor(y.limit*1e6)-1)/1e6), col="cyan", lwd=2)
		  ## abline(a=log2(8/linkspeed), b=1, col="cyan", lwd=2)
		  panel.myxyplot(x,y,...)
		})
  plt _ update.plot.labels(plt, main=tag$main)

  psfile <- paste(tag$psPrefix, ".ia_vs_size.span", SPAN.NPAC, ".plot.", tag$psSuffix, sep="")
  trellis.device("postscript", color = T, file = psfile, horizontal=F,
                 paper="letter", bg = "white", first=T)
  print(plt)
  dev.off()
  plt
}

## summary functions
template.summary.intvl.assembly _
  function(runName=paste(taskid, "pac.ia.marg", sep="."), 
           vars=c("shape", "scale"), settings) {
  cat(">>> running template.summary.intvl.assembly(",runName,",",vars,")\n", sep=" ")
  ## task _ "http.pac.ia"
  flst _ drop.names(unlist(ols(paste.ns("*",runName,"*compLst")))); flst
  if(length(flst)>0) {
    df _ NULL
    for(f in flst) { ## f _ flst[1]
      cat(">>> f=", f, "\n")
      cat(">>> starting compLst2df()\n")
      ## df0 _ compLst2df(f.drop, c("flowrate", "pacrate", vars))
      ## df0 _ compLst2df(f, c("flowrate", "pacrate", "streamrate", vars))
      ## names(flattenList(oget(f)[[1]]))
      df0 _ compLst2df(f, vars); ## sapply(df0, data.class)
      cat(">>> ending compLst2df()\n")
      print(dim(df0))
      df _ rbind(df, df0)
    }
    df _ reclass.data.frame(df) ## sapply(df, data.class)
    osave(paste(runName, "df", sep="."), df)
  }
}

template.summary.intvl.plot _
  function(runName=paste(taskid, "pac.ia.marg", sep="."), 
	   vars=c("shape", "scale"), settings) {
  cat(">>> running template.summary.intvl.plot()\n")
  df _ oget(paste(runName, "df", sep="."))

  ##if(exists("taskid") && exists("subtask.type")) {
  if (!is.null(settings$data.info)) {
    addVar _ c("pacrate", "flowrate", "streamrate")
    ## if(exists("rateVar")) addVar <- c(rateVar, addVar)
    ## if(exists("condVar")) addVar <- c(condVar, addVar)
    ## data.info _ get(paste(taskid,subtask.type,"subset.samp.df",sep="."))
    addVar _ addVar[!is.na(match(addVar, names(settings$data.info)))]
    if(length(addVar>0)) addVar _ addVar[is.na(match(addVar, names(df)))]
    if(sum(df$objnme!=settings$data.info$objnme)>0) {
      cat(">>> warning: ", paste(runName, "df", sep="."),
          "$objnme differs from settings$data.info$objnme\n", sep="")
    }
    ## assert(all.equal(drop.names(df$objnme), drop.names(data.info$objnme)))
    if(length(addVar)>0) {
      tmp _ cbind(df, settings$data.info[,addVar,drop=F])
      names(tmp) <- c(names(df), addVar); df <- tmp
    }
    ## I have to do this since direct cbind could change variable names
  }
  rmvars <- rep(F,length(vars))  ##remove variable with all NA values
  for(i in 1:length(vars)){
    if(all(is.na(df[,vars[i]]))) rmvars[i] <- T
  }
  df2 _ unfold.data.frame(df, vars[!rmvars], name="y")
  df2 _ reclass.data.frame(df2)
  ## sapply(df2, data.class)
  include.plot.par.vs.rate(df2, runName)
}

include.plot.par.vs.rate _ function(df2, runName, color=T, horizontal= F) {
  plotfile <- paste(runName,"par_vs_rate.ps", sep=".")
  ## trellis.device(postscript, file=plotfile)
  trellis.device("postscript", file = plotfile,
                 color = color, horizontal=horizontal,
                 paper="letter", bg = "white", first=T)
  colName <- unique(df2$rawColName)
  rateNames <- c("flowrate", "pacrate", "streamrate")
  ## if(exists("rateVar")) rateNames <- c(rateNames,rateVar)
  for(rateName in rateNames) { ## rateName _ rateNames[1]
    if( (!is.null(df2[[rateName]])) && (!all(is.na(df2[[rateName]]))) ) {
      formu <- paste("y ~ log2(", rateName, ")") ## y ~ log2(flowrate)
      if(exists("condVar")) {
        formu <- paste("y ~ log2(", rateName, ") | ", 
		       paste(condVar, collapse="*"))
      }
      formu <- formula(formu)
      for(cname in colName){ ## cname _ colName[1]
	plt _ xyplot(formu, pch="o", data=df2[df2$rawColName==cname,],
		     panel=function(x,y,...) {
		       panel.xyplot(x,y, ...)
		       sel _ !is.na(x) & (!is.na(y))
		       ##panel.loess(x[sel],y[sel], span=10000)
		       fit <- fitted.values(loess(y[sel]~x[sel],span=0.75,degree=1))
		       ord _ order(x[sel])
		       if(!is.null(fit)) llines(x[sel][ord], fit[ord], lty=2, lwd=2, col="black")
		     },
		     ## panel.myxyplot.loess,
		     ##scale=list(y=list(relation="free")),
		     ylab=cname, main=runName, col="black"
		     ##layout=c(2,3)
		     )
	print(plt)
      }
    }
  }
  dev.off()
  system(paste("gzip -f", plotfile))
}

template.summary.intvl _  
  function(runName=paste(taskid, "pac.ia.marg", sep="."), 
	   vars=c("shape", "scale"), settings)
{
  cat(">>> running template.summary.intvl()\n")
  template.summary.intvl.assembly(runName, vars, settings)
  template.summary.intvl.plot(runName, vars, settings)
}

compLst2df <- function(lstnme,
                       vars=c("flowrate","pacrate","streamrate","d","varRatio",
			   "shape","scale", "etp","etp.fit",
			   "geweke.smooth.etp.fit")) {
  ## this is to convert the computed lst objects to standardized data frames,
  ## depending on the subtask (for use at the end of the computation)
  if(F){
    attach("/n/taihu/sdd1/cao_scratch/packet/dataObj/other-traces/SobjStable")
    df <- compLst2df(lstnme="nlanr.0901.1105.pac.ia.compLst",
                     subtask="pac.ia")
  }
  
  df _ sapply.extractList(oget(lstnme), vars)
  ## df$objnme _ unix("awk '{print $NF}'", 
  ##		      unlist(sapply.extractList(get(lstnme),"main"), 
  ##			     use.names=F))
  df$objnme _ unlist(sapply.extractList(oget(lstnme), "objnme"))
  print(dim(df))
  df$compLst <- rep(lstnme, nrow(df))
  ## df$variable <- rep(subtask, nrow(df))
  df$component <- dimnames(df)[[1]]
  df <- df[,c("compLst", "component", "objnme", vars)]
  dimnames(df) <- list(1:nrow(df), dimnames(df)[[2]])
  df
}

merge.intvl.ps _ function(runName) {
  flst _ unique(mysystem(paste.ns("ls *", runName, "*.ps.[0-9]* | sed 's/ps.[0-9]*/ps/'"))); flst
  for(f in flst) {
    cmd _ paste("ps_merge_Splus.pl ", f, ".[0-9]* | gzip > ", f, ".gz", sep="")
    print(cmd)
    system(cmd)
  }
}

merge.intvl.compLst _ function(runName) {
  flst _ unique(mysystem("sed 's/compLst.[0-9]*/compLst/'",
                         unlist(ols(paste.ns("*", runName, "*compLst.[0-9]*"))))); flst
  for(f in flst) { ## f _ flst[1]; f
    nms _ drop.names(unlist(ols(paste.ns(f, ".[0-9]*"))))
    newlst _ list()
    for(nm in nms) {
      cat(">>>", nm, "\n")
      newlst[[nm]] _ oget(nm)
    }
    osave(f, newlst)
  }
}
