### $Id: gls.q,v 1.1 2001/09/24 21:35:31 pinheiro Exp $
###
###  Fit a linear model with correlated errors and/or heteroscedasticity
###
### Copyright 1997, 1999 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA

gls <- 
  ## fits linear model with serial correlation and variance functions,
  ## by maximum likelihood using a Newton-Raphson algorithm.
  function(model,
	   data = sys.parent(),
	   correlation = NULL,
	   weights = NULL,
	   subset,
	   method = c("REML", "ML"), 
	   na.action = na.fail, 
	   control = list(),
	   verbose = F)
{
  Call <- match.call()

  ## control parameters
  controlvals <- glsControl()
  controlvals[names(control)] <- control

  ##
  ## checking arguments
  ##
  if (!inherits(model, "formula") || length(model) != 3) {
    stop("\nModel must be a formula of the form \"resp ~ pred\"")
  }
  method <- match.arg(method)
  REML <- method == "REML"
  ## check if correlation is present and has groups
  if (!is.null(correlation)) {
    groups <- getGroupsFormula(correlation)
    corStr <- getStrataFormula(correlation)
  } else {
    groups <- NULL
    corStr <- NULL
  }
  ## create a gls structure containing the plug-ins
  glsSt <- 
    glsStruct(corStruct = correlation, varStruct = varFunc(weights))

  ## extract a data frame with enough information to evaluate
  ## formula, groups, corStruct, and varStruct
  mfArgs <- list(formula = asOneFormula(formula(glsSt), model, groups,
                   corStr), data = data, na.action = na.action)
  if (!missing(subset)) {
    mfArgs[["subset"]] <- asOneSidedFormula(Call[["subset"]])[[2]]
  }
  dataMod <- do.call("model.frame", mfArgs)
  origOrder <- row.names(dataMod)	# preserve the original order
  if (!is.null(groups)) {
    ## sort the model.frame by groups and get the matrices and parameters
    ## used in the estimation procedures
    ## always use innermost level of grouping
    groups <- eval(parse(text = paste("~1", deparse(groups[[2]]), sep = "|")))
    grps <- getGroups(dataMod, groups,
                      level = length(getGroupsFormula(groups, asList = T)))
    ## ordering data by groups
    ord <- order(grps)	
    grps <- grps[ord]
    dataMod <- dataMod[ord, ,drop = F]
    revOrder <- match(origOrder, row.names(dataMod)) # putting in orig. order
  } else grps <- NULL
  
  ## obtaing basic model matrices
  X <- model.frame(model, dataMod)
  ## keeping the contrasts for later use in predict
  contr <- lapply(X, function(el) 
		  if (inherits(el, "factor")) contrasts(el))
  contr <- contr[!unlist(lapply(contr, is.null))]
  X <- model.matrix(model, X)
  y <- eval(model[[2]], dataMod)
  N <- nrow(X)
  p <- ncol(X)				# number of coefficients
  parAssign <- attr(X, "assign")
    
  ## creating the condensed linear model
  attr(glsSt, "conLin") <-
    list(Xy = array(c(X, y), c(N, ncol(X) + 1), list(row.names(dataMod), 
	     c(dimnames(X)[[2]], deparse(model[[2]])))), 
	 dims = list(N = N, p = p, REML = as.integer(REML)), logLik = 0,
         sigma = controlvals$sigma)

  ## initialization
  glsEstControl <- controlvals[c("singular.ok","qrTol")]
  glsSt <- initialize(glsSt, dataMod, glsEstControl)
  parMap <- attr(glsSt, "pmap")

  ##
  ## getting the fitted object, possibly iterating for variance functions
  ##
  numIter <- numIter0 <- 0
  attach(controlvals)
  repeat {
    oldPars <- c(attr(glsSt, "glsFit")[["beta"]], coef(glsSt))
    if (length(coef(glsSt))) {		# needs ms()
      aMs <- ms(~-logLik(glsSt, glsPars),
                start = list(glsPars = c(coef(glsSt))),
                control = list(rel.tolerance = msTol, maxiter = msMaxIter,
                  scale = msScale), trace = msVerbose)
      coef(glsSt) <- aMs$parameters
      numIter0 <- aMs$numIter <- aMs$flags[31]
    }
    attr(glsSt, "glsFit") <- glsEstimate(glsSt, control = glsEstControl)
    ## checking if any updating is needed
    if (!needUpdate(glsSt)) break
    ## updating the fit information
    numIter <- numIter + 1
    glsSt <- update(glsSt, dataMod)
    ## calculating the convergence criterion
    aConv <- c(attr(glsSt, "glsFit")[["beta"]], coef(glsSt))
    conv <- abs((oldPars - aConv)/ifelse(aConv == 0, 1, aConv))
    aConv <- c("beta" = max(conv[1:p]))
    conv <- conv[-(1:p)]
    for(i in names(glsSt)) {
      if (any(parMap[,i])) {
	aConv <- c(aConv, max(conv[parMap[,i]]))
	names(aConv)[length(aConv)] <- i
      }
    }
    if (verbose) {
      cat("\nIteration:",numIter)
      if (length(coef(glsSt)) > 0) {
        cat("\nObjective:",format(aMs$value),", ms iterations:",
            aMs$numIter, "\n")
        print(glsSt)
      }
      cat("\nConvergence:\n")
      print(aConv)
    }
    if (max(aConv) <= tolerance) {
      break
    }
    if (numIter > maxIter) {
      stop("Maximum number of iterations reached without convergence.")
    }
  }
  detach()
  ## wrapping up
  glsFit <- attr(glsSt, "glsFit")
  namBeta <- names(glsFit$beta)
  attr(parAssign, "varBetaFact") <- varBeta <-
    glsFit$sigma * glsFit$varBeta * sqrt((N - REML * p)/(N - p))
  varBeta <- crossprod(varBeta)
  dimnames(varBeta) <- list(namBeta, namBeta)
  ##
  ## fitted.values and residuals (in original order)
  ##
  Fitted <- fitted(glsSt)
  ## putting groups back in original order, if present
  if (!is.null(grps)) {
    grps <- grps[revOrder]
    Fitted <- Fitted[revOrder]
    Resid <- y[revOrder] - Fitted
    attr(Resid, "std") <- glsFit$sigma/(varWeights(glsSt)[revOrder])
  } else {
    Resid <- y - Fitted
    attr(Resid, "std") <- glsFit$sigma/(varWeights(glsSt))
  }
    
  ## getting the approximate var-cov of the parameters 
  if (controlvals$apVar) {
    apVar <- glsApVar(glsSt, glsFit$sigma, 
		      .relStep = controlvals[[".relStep"]],
                      minAbsPar = controlvals[["minAbsParApVar"]],
                      natural = controlvals[["natural"]],
                      natUncons = controlvals[["natUnconstrained"]])
  } else {
    apVar <- "Approximate variance-covariance matrix not available"
  }
  ## getting rid of condensed linear model and fit
  dims <- attr(glsSt, "conLin")[["dims"]]
  dims[["p"]] <- p
  attr(glsSt, "conLin") <- NULL
  attr(glsSt, "glsFit") <- NULL
  attr(glsSt, "fixedSigma") <- (controlvals$sigma > 0)
  ##
  ## creating the  gls object
  ##
  estOut <- list(modelStruct = glsSt,
		 dims = dims,
		 contrasts = contr,
		 coefficients = glsFit[["beta"]],
		 varBeta = varBeta,
		 sigma = glsFit$sigma,
		 apVar = apVar,
		 logLik = glsFit$logLik,
		 numIter = if (needUpdate(glsSt)) numIter
		   else numIter0, 
		 groups = grps,
		 call = Call,
		 method = method,
		 fitted = Fitted,
		 residuals = Resid,
                 parAssign = parAssign)
  if (inherits(data, "groupedData")) {
    ## saving labels and units for plots
    attr(estOut, "units") <- attr(data, "units")
    attr(estOut, "labels") <- attr(data, "labels")
  }
  if (!is.null(grps) && any(diff(ord) != 1)) {
    attr(estOut, "order") <- list(order = ord, revOrder = revOrder)
  }
  attr(estOut, "namBetaFull") <- dimnames(X)[[2]]
  class(estOut) <- "gls"
  estOut
}

### Auxiliary functions used internally in gls and its methods

glsApVar <-
  function(glsSt, sigma, conLin = attr(glsSt, "conLin"),
           .relStep = (.Machine$double.eps)^(1/3), minAbsPar = 0,
           natural = TRUE, natUncons = TRUE)
{
  ## calculate approximate variance-covariance matrix of all parameters
  ## except the coefficients
  fullGlsLogLik <-
    function(Pars, object, conLin, dims, N, sigUnc)
    {
      ## logLik as a function of sigma and coef(glsSt)
      if (conLin$sigma == 0) {
        npar <- length(Pars)
        if (sigUnc) {                     # within-group std. dev.
          lsigma <- Pars[npar]
        } else {
          lsigma <- log(Pars[npar])/2
        }
        sigma <- 0
        Pars <- Pars[-npar]
      } else {
        sigma <- conLin$sigma
      }
      coef(object) <- Pars
      conLin <- recalc(object, conLin)
      val <- .C("gls_loglik",
                as.double(conLin$Xy),
                as.integer(unlist(dims)),
                logLik = as.double(conLin[["logLik"]]),
                lRSS = double(1),
                as.double(sigma))[c("lRSS", "logLik")]
      if (sigma == 0) {
        aux <- 2 * (val[["lRSS"]] - lsigma)
        return(val[["logLik"]] + (N * aux - exp(aux))/2)
      } else {
        return(val[["logLik"]])
      }
    }
  if (is.logical(natural)) {
    ## repeat for all structures
    natural <- list(reStruct = natural, corStruct = natural,
      varStruct = natural)
  }
  if (is.logical(natUncons)) {
    ## repeat for all structures
    natUncons <- list(corStruct = natUncons, varStruct = natUncons,
                      sig = natUncons)
  }
  if (length(glsCoef <- coef(glsSt)) > 0) {
    for(i in names(glsSt)) {
      if (natural[[i]]) {
        glsSt[[i]] <- asNatural(glsSt[[i]], natUncons[[i]])
      }
    }
    dims <- conLin$dims
    N <- dims$N - dims$REML * dims$p
    conLin[["logLik"]] <- 0               # making sure
    if (conLin$sigma == 0) {
      if (natUncons[["sig"]]) {
        Pars <- c(coef(glsSt), lSigma = log(sigma))
      } else {
        Pars <- c(coef(glsSt), sig2 = sigma * sigma)
      }
    } else {
      Pars <- coef(glsSt)
    }
    val <- fdHess(Pars, fullGlsLogLik, glsSt, conLin, dims, N,
		  .relStep = .relStep, minAbsPar = minAbsPar,
                  sigUnc = natUncons[["sig"]])[["Hessian"]]
    if (all(eigen(val)$values < 0)) {
      ## negative definite - OK
      val <- solve(-val)
      nP <- names(Pars)
      dimnames(val) <- list(nP, nP)
      attr(val, "Pars") <- Pars
      attr(val, "natural") <- natural
      attr(val, "natUncons") <- natUncons
      val
    } else {
      ## problem - solution is not a maximum
      "Non-positive definite approximate variance-covariance"
    }
  } else {
    NULL
  }
}

glsEstimate <-
  function(object, conLin = attr(object, "conLin"), 
	   control = list(singular.ok = F, qrTol = .Machine$single.eps))
{
  dd <- conLin$dims
  p <- dd$p
  oXy <- conLin$Xy
  sigma <- conLin$sigma
  conLin <- recalc(object, conLin)	# updating for corStruct and varFunc
  val <- .C("gls_estimate",
	    as.double(conLin$Xy),
	    as.integer(unlist(dd)),
	    beta = double(p),
	    sigma = as.double(sigma),
	    logLik = double(1),
	    varBeta = double(p * p),
	    rank = integer(1),
	    pivot = as.integer(1:(p + 1)))[c("beta","sigma","logLik","varBeta",
		"rank", "pivot")]
  rnk <- val[["rank"]]
  rnkm1 <- rnk - 1
  if (!(control$singular.ok) && (rnkm1 < p )) {
    stop(paste("computed gls fit is singular, rank", rnk))
  }
  N <- dd$N - dd$REML * p
  namCoef <- dimnames(oXy)[[2]][val[["pivot"]][1:rnkm1] + 1]	# coef names
  ll <- conLin$logLik + val[["logLik"]]
  varBeta <- t(array(val[["varBeta"]], c(rnkm1, rnkm1), 
		     list(namCoef, namCoef)))
  beta <- val[["beta"]][1:rnkm1]
  names(beta) <- namCoef
  fitVal <- oXy[, namCoef, drop = F] %*% beta
  if (sigma > 0) {
    logLik <- -N * log(2 * pi)/2 + ll
  } else {
    logLik <- N * (log(N) - (1 + log(2 * pi)))/2 + ll
  }
  list(logLik = logLik, beta = beta,
       sigma = val[["sigma"]], varBeta = varBeta, 
       fitted = c(fitVal), resid = c(oXy[, p + 1] - fitVal))
}

### Methods for standard generics

ACF.gls <-
  function(object, maxLag, resType = c("pearson", "response", "normalized"),
           form = ~1, na.action = na.fail, ...)
{
  resType <- match.arg(resType)
  res <- resid(object, type = resType)
  wchRows <- NULL
  if (is.null(grps <- getGroups(object))) {
    ## check if formula defines groups
    if (!is.null(grpsF <- getGroupsFormula(form))) {
      if (is.null(data <- getData(object))) {
        ## will try to construct
        allV <- all.vars(grpsF)
        if (length(allV) > 0) {
          alist <- lapply(as.list(allV), as.name)
          names(alist) <- allV
          alist <- c(as.list(as.name("data.frame")), alist)
          mode(alist) <- "call"
          data <- eval(alist, sys.parent(1))
        } 
      }
      grps <- model.frame(grpsF, data, na.action = na.action)
      wchRows <- !is.na(match(row.names(data), row.names(grps)))
      grps <- getGroups(grps, grpsF)
    }
  }
  if (!is.null(grps)) {
    if (!is.null(wchRows)) {
      res <- res[wchRows]
    }
    res <- split(res, grps)
  } else {
    res <- list(res)
  }
  if(missing(maxLag)) {
    maxLag <- min(c(maxL <- max(sapply(res, length)) - 1, 
                    as.integer(10 * log10(maxL + 1))))
  }
  val <- lapply(res,
                function(el, maxLag) {
                  N <- maxLag + 1
                  tt <- double(N)
                  nn <- integer(N)
                  N <- min(c(N, n <- length(el)))
                  nn[1:N] <- n + 1 - 1:N
                  ## el <- el - mean(el) 
                  for(i in 1:N) {
                    el1 <- el[1:(n-i+1)]
                    el2 <- el[i:n]
                    tt[i] <- sum(el1 * el2)
                  }
                  array(c(tt,nn), c(length(tt), 2))
                }, maxLag = maxLag)
  val0 <- apply(sapply(val, function(x) x[,2]), 1, sum)
  val1 <- apply(sapply(val, function(x) x[,1]), 1, sum)/val0
  val2 <- val1/val1[1]
  z <- data.frame(lag = 0:maxLag, ACF = val2)
  attr(z, "n.used") <- val0
  class(z) <- c("ACF", "data.frame")
  z
}

anova.gls <- 
  function(object, ..., test = TRUE, type = c("sequential", "marginal"),
           adjustSigma = TRUE, Terms, L, verbose = F)
{
  ## returns the likelihood ratio statistics, the AIC, and the BIC
  dots <- list(...)
  if ((rt <- length(dots) + 1) == 1) {
    if (!inherits(object,"gls")) {
      stop("Object must inherit from class \"gls\" ")
    }
    if (inherits(object, "gnls") && missing(adjustSigma)) {
      ## REML correction already applied to gnls objects
      adjustSigma <- FALSE
    }
    dims <- object$dims
    N <- dims$N
    p <- dims$p
    REML <- dims$REML
    assign <- object$parAssign
    vBeta <- attr(assign, "varBetaFact")
    if ((!REML) && !adjustSigma) {
      ## reverting ML-like estimate of sigma under ML
      vBeta <- sqrt((N - p)/N) * vBeta
    }
    c0 <- solve(t(vBeta), coef(object))
    nTerms <- length(assign)
    dDF <- N - p
    lab <- paste("Denom. DF:", dDF,"\n")
    if (missing(Terms) && missing(L)) {
      ## returns the F.table (Wald) for the fixed effects
      type <- match.arg(type)
      Fval <- Pval <- double(nTerms)
      nDF <- integer(nTerms)
      for(i in 1:nTerms) {
        nDF[i] <- length(assign[[i]])
        if (type == "sequential") {       # type I SS
          c0i <- c0[assign[[i]]]
        } else {
          c0i <- c(qr.qty(qr(vBeta[, assign[[i]], drop = F]), c0))[1:nDF[i]]
        }
        Fval[i] <- sum(c0i^2)/nDF[i]
        Pval[i] <- 1 - pf(Fval[i], nDF[i], dDF)
      }
      ##
      ## fixed effects F-values, df, and p-values
      ##
      aod <- data.frame(nDF, Fval, Pval)
      dimnames(aod) <- 
        list(names(assign),c("numDF", "F-value", "p-value"))
    } else {
      if (missing(L)) {                 # terms is given
        if (is.numeric(Terms) && all(Terms == as.integer(Terms))) {
          if (min(Terms) < 1 || max(Terms) > nTerms) {
            stop(paste("Terms must be between 1 and", nTerms))
          }
        } else {
          if (is.character(Terms)) {
            if (any(noMatch <- is.na(match(Terms, names(assign))))) {
              stop(paste("Term(s)", paste(Terms[noMatch], collapse = ", "),
                         "not matched"))
            }
          } else {
            stop("Terms can only be integers or characters")
          }
        }
        lab <-
          paste(lab, "F-test for:",
                paste(names(assign[Terms]),collapse=", "),"\n")
        L <- diag(p)[unlist(assign[Terms]),,drop=F]
      } else {
        L <- as.matrix(L)
        if (ncol(L) == 1) L <- t(L)     # single linear combination
        nrowL <- nrow(L)
        ncolL <- ncol(L)
        if (ncol(L) > p) {
          stop(paste("L must have at most", p,"columns"))
        }
        dmsL1 <- dimnames(L)[[1]]
        L0 <- array(0, c(nrowL, p), list(NULL, names(coef(object))))
        if (is.null(dmsL2 <- dimnames(L)[[2]])) {
          ## assume same order as effects
          L0[, 1:ncolL] <- L
        } else {
          if (any(noMatch <- is.na(match(dmsL2, dimnames(L0)[[2]])))) {
            stop(paste("Effects",paste(dmsL2[noMatch],collapse=", "),
                       "not matched"))
          }
          L0[, dmsL2] <- L
        }
        L <- L0[noZeroRowL <- as.logical((L0 != 0) %*% rep(1, p)), , drop = F]
        nrowL <- nrow(L)
        noZeroColL <- as.logical(c(rep(1,nrowL) %*% (L != 0)))
        if (is.null(dmsL1)) {
          dmsL1 <- 1:nrowL
        } else {
          dmsL1 <- dmsL1[noZeroRowL]
        }
        dimnames(L)[[1]] <- dmsL1
        lab <- paste(lab, "F-test for linear combination(s)\n")
      }
      nDF <- sum(svd(L)$d > 0)
      c0 <- c(qr.qty(qr(vBeta %*% t(L)), c0))[1:nDF]
      Fval <- sum(c0^2)/nDF
      Pval <- 1 - pf(Fval, nDF, dDF)
      aod <- data.frame(nDF, Fval, Pval)
      names(aod) <- c("numDF", "F-value", "p-value")
      if (!missing(L)) {
        if (nrow(L) > 1) attr(aod, "L") <- L[, noZeroColL, drop = F]
        else attr(aod, "L") <- L[, noZeroColL]
      }
    }
    attr(aod, "label") <- lab
    attr(aod,"rt") <- rt
    class(aod) <- c("anova.lme", "data.frame")
    aod
  }
  ##
  ## Otherwise construct the likelihood ratio and information table
  ## objects in ... may inherit from gls, lm, lmList, and lme (for now)
  ##
  else do.call("anova.lme", as.list(match.call()[-1]))
}

augPred.gls <- 
  function(object, primary = NULL, minimum = min(primary), 
	   maximum = max(primary), length.out = 51, ...)
{
#  data <- eval(object$call$data)
  data <- getData(object)
  if (!inherits(data, "data.frame")) {
    stop(paste("Data in", substitute(object),
               "call must evaluate to a data frame"))
  }
  if(is.null(primary)) {
    if (!inherits(data, "groupedData")) {
      stop(paste(sys.call()[[1]],
      "without \"primary\" can only be used with fits of groupedData objects"))
    }
    primary <- getCovariate(data)
    prName <- deparse(getCovariateFormula(data)[[2]])
  } else{
    primary <- asOneSidedFormula(primary)[[2]]
    prName <- deparse(primary)
    primary <- eval(primary, data)
  }
  newprimary <- seq(from = minimum, to = maximum, length.out = length.out)
  groups <- getGroups(object)
  grName <- ".groups"
  if (is.null(groups)) {		# no groups used
    noGrp <- T
    groups <- rep("1", length(primary))
    value <- data.frame(newprimary, rep("1", length(newprimary)))
  } else {
    noGrp <- F
    ugroups <- unique(groups)
    value <- data.frame(rep(newprimary, length(ugroups)),
			rep(ugroups, rep(length(newprimary), length(ugroups))))
  }
  names(value) <- c(prName, grName)
  ## recovering other variables in data that may be needed for predictions
  ## varying variables will be replaced by their means
  summData <- gsummary(data, groups = groups)
  if (any(toAdd <- is.na(match(names(summData), names(value))))) {
    summData <- summData[, toAdd, drop = F]
  }
  value[, names(summData)] <- summData[value[, 2], ]
  pred <- predict(object, value)
  newvals <- cbind(value[, 1:2], pred)
  names(newvals)[3] <- respName <-
    deparse(getResponseFormula(object)[[2]])
  orig <- data.frame(primary, groups, getResponse(object))
  names(orig) <- names(newvals)
  value <- rbind(orig, newvals)
  attributes(value[, 2]) <- attributes(groups)
  value[, ".type"] <- ordered(c(rep("original", nrow(data)),
				rep("predicted", nrow(newvals))),
			      levels = c("predicted", "original"))
  labs <- list(x = prName, y = respName)
  unts <- list(x = "", y = "")
  if(inherits(data, "groupedData")) {
    labs[names(attr(data, "labels"))] <- attr(data, "labels")
    unts[names(attr(data, "units"))] <- attr(data, "units")
    attr(value, "units") <- attr(data, "units")
  } 
  attr(value, "labels") <- labs
  attr(value, "units") <- unts
  if (noGrp) {
    attr(value, "formula") <- 
      eval(parse(text = paste(respName, prName, sep = "~")))
  } else {
    attr(value, "formula") <- 
      eval(parse(text = paste(respName, "~", prName, "|", grName)))
  }
  class(value) <- c("augPred", class(value))
  value
}

coef.gls <-
  function(object, allCoef = F)
{
  val <- object$coefficients
  if (allCoef) {
    namFull <- attr(object, "namBetaFull")
    if (length(val) != (lF <- length(namFull))) {
      aux <- rep(NA, lF)
      names(aux) <- namFull
      aux[names(val)] <- val
      val <- aux
    }
  }
  val
}

comparePred.gls <-
  function(object1, object2, primary = NULL, 
	   minimum = min(primary), maximum = max(primary),
	   length.out = 51, level = NULL, ...) 
{
  if (length(level) > 1) {
    stop("Only one level allowed for predictions")
  }
  args <- list(object = object1, 
	       primary = primary,
	       level = level,
	       length.out = length.out)
  if (!is.null(primary)) {
    args[["minimum"]] <- minimum
    args[["maximum"]] <- maximum
  }
  val1 <- do.call("augPred", args)
  dm1 <- dim(val1)
  c1 <- deparse(substitute(object1))
  levels(val1[,4])[1] <- c1
  args[["object"]] <- object2
  val2 <- do.call("augPred", args)
  dm2 <- dim(val2)
  c2 <- deparse(substitute(object2))
  levels(val2[, 4])[1] <- c2
  val2 <- val2[val2[, 4] != "original", , drop = F]
  names(val2) <- names(val1)

  if (dm1[1] == dm2[1]) {
    lv1 <- sort(levels(val1[, 2]))
    lv2 <- sort(levels(val2[, 2]))
    if ((length(lv1) != length(lv2)) || any(lv1 != lv2)) {
      stop(paste(c1, "and", c2, "must have the same group levels"))
    }
    val <- rbind(val1[, -4], val2[, -4])
    val[, ".type"] <- 
      ordered(c(as.character(val1[,4]), as.character(val2[,4])),
		levels = c(c1, c2, "original"))
    attr(val, "formula") <- attr(val1, "formula")
  } else {				# one may have just "fixed"
    if (dm1[1] > dm2[1]) {
      mult <- dm1[1] %/% dm2[1]
      if ((length(levels(val2[, 2])) != 1) ||
	  (length(levels(val1[, 2])) != mult))
	{
	  stop("Wrong group levels")
	}
      val <- 
	data.frame(c(val1[,1], rep(val2[,1], mult)), rep(val1[,1], 2),
	   c(val1[,3], rep(val2[,3], mult)),
	   ordered(c(as.character(val1[,4]), 
		     rep(as.character(val2[,4]), mult)), 
		   levels = c(c1, c2, "original")))
      attr(val, "formula") <- attr(val1, "formula")
    } else {
      mult <- dm2[1] %/% dm1[1]
      if ((length(levels(val1[, 2])) != 1) ||
	  (length(levels(val2[, 2])) != mult))
	{
	  stop("Wrong group levels")
	}
      val <- 
	data.frame(c(rep(val1[,1], mult), val2[,1]), rep(val2[,1], 2),
	   c(rep(val1[,3], mult), val2[,3]),
	   ordered(c(rep(as.character(val1[,4]), mult), 
		     as.character(val1[,4])), levels = c(c1, c2, "original")))
      attr(val, "formula") <- attr(val2, "formula")
    }
  }
  class(val) <- c("comparePred", "augPred", class(val))
  attr(val, "labels") <- attr(val1, "labels")
  attr(val, "units") <- attr(val1, "units")
  val
}

fitted.gls <-
  function(object)
{
  val <- object$fitted
  lab <- "Fitted values"
  if (!is.null(aux <- attr(object, "units")$y)) {
    lab <- paste(lab, aux)
  }
  attr(val, "label") <- lab
  val
}


formula.gls <- function(object) eval(object$call$model)

getData.gls <-
  function(object)
{
  mCall <- object$call
  data <- eval(mCall$data)
  if (is.null(data)) return(data)
  naAct <- eval(mCall$na.action)
  subset <- mCall$subset
  if (!is.null(naAct)) {
    modSt <- object$modelStruct
    mfArgs <- list(formula = asOneFormula(formula(object),
                     formula(modSt),
                     if (!is.null(modSt$corStruct)) {
                       getStrataFormula(modSt$corStruct)
                     } else { NULL }),
                   data = data, na.action = naAct)
    if (!is.null(subset)) {
      mfArgs[["subset"]] <- asOneSidedFormula(subset)[[2]]
    }
    data <- data[as.character(row.names(do.call("model.frame", mfArgs))),]
  } else {
    if (!is.null(subset)) {
      subset <- eval(asOneSidedFormula(subset)[[2]], data)
      data <- data[subset, ]
    }
  }
  data
}

getGroups.gls <- function(object, form, level, data, sep) object$groups

getGroupsFormula.gls <-
  function(object, asList = FALSE, sep)
{
  if (!is.null(cSt <- object$modelStruct$corStruct)) {
    getGroupsFormula(cSt, asList)
  } else {
    NULL
  }
}

getResponse.gls <-
  function(object, form)
{
  val <- resid(object) + fitted(object)
  if (is.null(lab <- attr(object, "labels")$y)) {
    lab <- deparse(getResponseFormula(object)[[2]])
  }
  if (!is.null(aux <- attr(object, "units")$y)) {
    lab <- paste(lab, aux)
  }
  attr(val, "label") <- lab
  val
}

intervals.gls <-
  function(object, level = 0.95, which = c("all", "var-cov", "coef"))
{
  which <- match.arg(which)
  val <- list()
  dims <- object$dims
  if (which != "var-cov") {		# coefficients included
    len <- -qt((1-level)/2, dims$N - dims$p) * sqrt(diag(object$varBeta))
    est <- coef(object)
    val[["coef"]] <- 
      array(c(est - len, est, est + len), 
	    c(length(est), 3), list(names(est), c("lower", "est.", "upper")))
    attr(val[["coef"]], "label") <- "Coefficients:"
  }

  ## back compatibility with 3.1 and lower
  fixSig <- attr(object[["modelStruct"]], "fixedSigma")
  fixSig <- !is.null(fixSig) && fixSig
  if (which != "coef") {		# variance-covariance included
    if (is.null(aV <- object$apVar)) {
      if (!fixSig) {                    # only sigma
        if (inherits(object, "gnls")) {   #always REML-like sigma
          Nr <- dims$N - dims$p
        } else {
          Nr <- dims$N - dims$REML * dims$p
        }
        est <- object$sigma * sqrt(Nr)
        val[["sigma"]] <- c(est/sqrt(qchisq((1+level)/2, Nr)), object$sigma, 
                            est/sqrt(qchisq((1-level)/2, Nr)))
        names(val[["sigma"]]) <- c("lower", "est.", "upper")
        attr(val[["sigma"]], "label") <- "Residual standard error:"
      }
    } else {
      if (is.character(aV)) {
	stop(paste("Cannot get confidence intervals on var-cov components:",
		   aV))
      }
      len <- -qnorm((1-level)/2) * sqrt(diag(aV))
      est <- attr(aV, "Pars")
      nP <- length(est)
      glsSt <- object[["modelStruct"]]
      if (!all(whichKeep <- apply(attr(glsSt, "pmap"), 2, any))) {
        ## need to deleted components with fixed coefficients
        aux <- glsSt[whichKeep]
        class(aux) <- class(glsSt)
        attr(aux, "settings") <- attr(glsSt, "settings")
        attr(aux, "pmap") <- attr(glsSt, "pmap")[, whichKeep, drop = F]
        glsSt <- aux
      }
      natural <- attr(aV, "natural")
      natUncons <- attr(aV, "natUncons")
      for (i in names(glsSt)) {
        if (natural[[i]]) {
          glsSt[[i]] <- asNatural(glsSt[[i]], natUncons[[i]])
        }
      }
      pmap <- attr(glsSt, "pmap")
      namG <- names(glsSt)
      if (!fixSig) {
        auxVal <- vector("list", length(namG) + 1)
        names(auxVal) <- c(namG, "sigma")
      } else {
        auxVal <- vector("list", length(namG))
        names(auxVal) <- namG
      }        
      aux <-
	array(c(est - len, est, est + len),
	      c(nP, 3), list(NULL, c("lower", "est.", "upper")))
      if (!fixSig) {
        if (natUncons[["sig"]]) {
          auxVal[["sigma"]] <- exp(aux[nP, ])
          attr(auxVal[["sigma"]], "label") <- "Residual standard error:"
        } else {
          auxVal[["sigma"]] <- aux[nP,]
          attr(auxVal[["sigma"]], "label") <- "Residual variance:"
        }
        aux <- aux[-nP,, drop = F]
      }
      dimnames(aux)[[1]] <- namP <- names(coef(glsSt, F))
      ## avoiding unnecessary updates
      for (i in names(glsSt)) {
        attr(glsSt[[i]], "noUpdate") <- TRUE
      }
      for(i in 1:3) {
	coef(glsSt) <- aux[,i]
	aux[,i] <- coef(glsSt, unconstrained = F)
      }
      for(i in namG) {
	auxVal[[i]] <- aux[pmap[,i], , drop = F]
	dimnames(auxVal[[i]])[[1]] <- 
	  substring(dimnames(auxVal[[i]])[[1]], nchar(i) + 2)
	attr(auxVal[[i]], "label") <-
	  switch(i,
		 corStruct = "Correlation structure:",
		 varStruct = "Variance function:",
		 paste(i,":",sep=""))
      }
      val <- c(val, auxVal)
    }
  }
  attr(val, "level") <- level
  class(val) <- "intervals.gls"
  val
}

logLik.gls <-
  function(object, REML)
{
  p <- object$dims$p
  N <- object$dims$N
  Np <- N - p
  estM <- object$method
  if (missing(REML)) REML <- estM == "REML"
  val <- object[["logLik"]]
  if (REML && (estM == "ML")) {			# have to correct logLik
    val <- val + (p * (log(2 * pi) + 1) + Np * log(1 - p/N) +
		  sum(log(abs(svd(object$varBeta)$d)))) / 2
  }
  if (!REML && (estM == "REML")) {	# have to correct logLik
    val <- val - (p * (log(2*pi) + 1) + N * log(1 - p/N) +
		  sum(log(abs(svd(object$varBeta)$d)))) / 2
  }
  attr(val, "nall") <- N
  attr(val, "nobs") <- N - REML * p
  ## back compatibility with 3.1 and lower
  fixSig <- attr(object[["modelStruct"]], "fixedSigma")
  fixSig <- !is.null(fixSig) && fixSig
  attr(val, "df") <- p + length(coef(object[["modelStruct"]])) +
    as.integer(!fixSig)
  class(val) <- "logLik"
  val
}

plot.gls <- 
  function(object, form = resid(., type = "pearson") ~ fitted(.), abline, 
	   id = NULL, idLabels = NULL, idResType = c("pearson", "normalized"),
           grid, subset = T, ...)  
  ## Diagnostic plots based on residuals and/or fitted values
{
  do.call("plot.lme", as.list(match.call()[-1]))
}

predict.gls <- 
  function(object, newdata, na.action = na.fail)  
{
  ##
  ## method for predict() designed for objects inheriting from class gls
  ##
  if (missing(newdata)) {		# will return fitted values
    return(fitted(object))
  }
  form <- getCovariateFormula(object)
  mfArgs <- list(formula = form, data = newdata, na.action = na.action)
  dataMod <- do.call("model.frame", mfArgs)
  ## making sure factor levels are the same as in contrasts
  contr <- object$contrasts
  for(i in names(dataMod)) {
    if (inherits(dataMod[,i], "factor") && !is.null(contr[[i]])) {
      levs <- levels(dataMod[,i])
      levsC <- dimnames(contr[[i]])[[1]]
      if (any(wch <- is.na(match(levs, levsC)))) {
        stop(paste("Levels", paste(levs[wch], collapse = ","),
                   "not allowed for", i))
      }
#      attr(dataMod[,i], "contrasts") <- contr[[i]][levs, , drop = FALSE]
      contr[[i]] <- contr[[i]][levs, , drop = FALSE]
#      if (length(levs) < length(levsC)) {
#        if (inherits(dataMod[,i], "ordered")) {
#          dataMod[,i] <- ordered(as.character(dataMod[,i]), levels = levsC)
#        } else {
#          dataMod[,i] <- factor(as.character(dataMod[,i]), levels = levsC)
#        }
#      }
    }
  }
  N <- nrow(dataMod)
  if (length(all.vars(form)) > 0) {
    X <- model.matrix(form, dataMod, contr)
#    X <- model.matrix(form, dataMod)
  } else {
    X <- array(1, c(N, 1), list(row.names(dataMod), "(Intercept)"))
  }
  cf <- coef(object)
  val <- c(X[, names(cf), drop = F] %*% cf)
  attr(val, "label") <- "Predicted values"
  if (!is.null(aux <- attr(object, "units")$y)) {
    attr(val, "label") <- paste(attr(val, "label"), aux)
  }
  val
}

print.intervals.gls <-
  function(x, ...)
{
  if (length(x) == 0) {
    stop("No confidence intervals for specified parameters")
  }
  cat(paste("Approximate ", attr(x, "level") * 100,
	    "% confidence intervals\n", sep = ""))
  for(i in names(x)) {
    aux <- x[[i]]
    cat("\n ",attr(aux, "label"), "\n", sep = "")
    if (i == "sigma") print(c(aux), ...)
    else print.matrix(aux, ...)
  }
}

print.gls <- 
  ## method for print() used for gls objects
  function(x, ...)
{
  dd <- x$dims
  mCall <- x$call
  if (inherits(x, "gnls")) {
    cat("Generalized nonlinear least squares fit\n")
  } else {
    cat("Generalized least squares fit by ")
    cat(ifelse(x$method == "REML", "REML\n", "maximum likelihood\n"))
  }
  cat("  Model:", deparse(as.vector(mCall$model)), "\n")
  cat("  Data:", deparse( mCall$data ), "\n")
  if (!is.null(mCall$subset)) {
    cat("  Subset:", deparse(asOneSidedFormula(mCall$subset)[[2]]),"\n")
  }
  if (inherits(x, "gnls")) {
    cat("  Log-likelihood: ", format(x$logLik), "\n", sep = "")
  } else {
    cat("  Log-", ifelse(x$method == "REML", "restricted-", ""),
        "likelihood: ", format(x$logLik), "\n", sep = "")
  }
  cat("\nCoefficients:\n")
  print(coef(x))
  cat("\n")
  if (length(x$modelStruct) > 0) {
    print(summary(x$modelStruct))
  }
  cat("Degrees of freedom:", dd[["N"]],"total;",dd[["N"]] - dd[["p"]],
      "residual\n")
  cat("Residual standard error:", format(x$sigma),"\n")
}

print.summary.gls <-
  function(x, verbose = FALSE, digits = .Options$digits, ...)
{
  dd <- x$dims
  verbose <- verbose || attr(x, "verbose")
  mCall <- x$call
  if (inherits(x, "gnls")) {
    cat("Generalized nonlinear least squares fit\n")
  } else {
    cat("Generalized least squares fit by ")
    cat(ifelse(x$method == "REML", "REML\n", "maximum likelihood\n"))
  }
  cat("  Model:", deparse(as.vector(mCall$model)), "\n")
  cat("  Data:", deparse( mCall$data ), "\n")
  if (!is.null(mCall$subset)) {
    cat("  Subset:", deparse(asOneSidedFormula(mCall$subset)[[2]]),"\n")
  }
  print( data.frame(AIC=x$AIC,BIC=x$BIC,logLik=x$logLik,row.names = " "))
  if (verbose) { cat("Convergence at iteration:",x$numIter,"\n") }
  if (length(x$modelStruct)) {
    cat("\n")
    print(summary(x$modelStruct))
  }
  cat("\nCoefficients:\n")
  xtTab <- as.data.frame(x$tTable)
  wchPval <- match("p-value", names(xtTab))
  for(i in names(xtTab)[-wchPval]) {
    xtTab[, i] <- format(zapsmall(xtTab[, i]))
  }
  xtTab[,wchPval] <- format(round(xtTab[,wchPval], 4))
  if (any(wchLv <- (as.double(levels(xtTab[, wchPval])) == 0))) {
    levels(xtTab[, wchPval])[wchLv] <- "<.0001"
  }
  row.names(xtTab) <- dimnames(x$tTable)[[1]]
  print(xtTab)
  if (nrow(x$tTable) > 1) {
    corr <- x$corBeta
    class(corr) <- "correlation"
    print(corr,
	  title = "\n Correlation:",
	  ...)
  }
  cat("\nStandardized residuals:\n")
  print(x$residuals)
  cat("\n")
  cat("Residual standard error:", format(x$sigma),"\n")
  cat("Degrees of freedom:", dd[["N"]],"total;",dd[["N"]] - dd[["p"]],
      "residual\n")
}

qqnorm.gls <-
  function(object, form = ~ resid(., type = "p"), abline = NULL,
           id = NULL, idLabels = NULL, grid = FALSE, ...)
  ## normal probability plots for residuals
{
  if (!inherits(form, "formula")) {
    stop("\"Form\" must be a formula")
  }
  ## constructing data 
  allV <- all.vars(asOneFormula(form, id, idLabels))
  allV <- allV[is.na(match(allV,c("T","F","TRUE","FALSE")))]
  if (length(allV) > 0) {
    data <- getData(object)
    if (is.null(data)) {		# try to construct data
      alist <- lapply(as.list(allV), as.name)
      names(alist) <- allV
      alist <- c(as.list(as.name("data.frame")), alist)
      mode(alist) <- "call"
      data <- eval(alist, sys.parent(1))
    } else {
      if (any(naV <- is.na(match(allV, names(data))))) {
	stop(paste(allV[naV], "not found in data"))
      }
    }
  } else data <- NULL
  ## argument list
  dots <- list(...)
  if (length(dots) > 0) args <- dots
  else args <- list()
  ## appending object to data
  data <- as.list(c(as.list(data), . = list(object)))

  ## covariate - must always be present
  covF <- getCovariateFormula(form)
  .x <- eval(covF[[2]], data)
  labs <- attr(.x, "label")
  if (is.null(labs) || ((labs != "Standardized residuals") &&
                        (labs != "Normalized residuals") &&
                        (substring(labs, 1, 9) != "Residuals"))) {
    stop("Only residuals allowed")
  }
  if (is.null(args$xlab)) args$xlab <- labs
  if (is.null(args$ylab)) args$ylab <- "Quantiles of standard normal"
  fData <- qqnorm(.x, plot.it = F)
  data[[".y"]] <- fData$x
  data[[".x"]] <- fData$y
  dform <- ".y ~ .x"
  if (!is.null(grp <- getGroupsFormula(form))) {
    dform <- paste(dform, deparse(grp[[2]]), sep = "|")
  }
  if (!is.null(id)) {			# identify points in plot
    id <- 
      switch(mode(id),
             numeric = {
               if ((id <= 0) || (id >= 1)) {
                 stop("Id must be between 0 and 1")
               }
               if (labs == "Normalized residuals") {
                 as.logical(abs(resid(object, type="normalized"))
                            > -qnorm(id / 2))
               } else {
                 as.logical(abs(resid(object, type="pearson"))
                            > -qnorm(id / 2))
               }
             },
             call = eval(asOneSidedFormula(id)[[2]], data),
             stop("\"Id\" can only be a formula or numeric.")
             )
    if (is.null(idLabels)) {
      idLabels <- getGroups(object)
      if (length(idLabels) == 0) idLabels <- 1:object$dims$N
      idLabels <- as.character(idLabels)
    } else {
      if (mode(idLabels) == "call") {
        idLabels <-
          as.character(eval(asOneSidedFormula(idLabels)[[2]], data))
      } else if (is.vector(idLabels)) {
        if (length(idLabels <- unlist(idLabels)) != length(id)) {
          stop("\"IdLabels\" of incorrect length")
        } 
        idLabels <- as.character(idLabels)
      } else {
        stop("\"IdLabels\" can only be a formula or a vector")
      }
    }
  }
  assign("id", if (is.null(id)) NULL else as.logical(as.character(id)),
         frame = 1)
  assign("idLabels", as.character(idLabels), frame = 1)
  assign("grid", grid, frame = 1)
  assign("abl", abline, frame = 1)
  if (is.null(args$strip)) {
    args$strip <- function(...) strip.default(..., style = 1)
  }
  if (is.null(args$cex)) args$cex <- par("cex")
  if (is.null(args$adj)) args$adj <- par("adj")

  args <- c(list(formula = eval(parse(text = dform)),
                 data = substitute(data)),
                 args)
  if (is.null(args$panel)) {
    args <- c(list(panel = function(x, y, subscripts, ...){
      dots <- list(...)
      if (grid) panel.grid()
      panel.xyplot(x, y, ...)
      if (!all(is.na(aux <- id[subscripts])) &&
          !is.null(aux) && any(aux)) {
        text(x[aux], y[aux], idLabels[subscripts][aux],
             cex = dots$cex, adj = dots$adj)
      }
      if (!is.null(abl)) panel.abline(abl, ...)
    }), args)
  }
  do.call("xyplot", args)
}

residuals.gls <- 
  function(object, type = c("response", "pearson", "normalized"))
{
  type <- match.arg(type)
  val <- object$residuals
  if (type != "response") {
    ord <- attr(object, "order")
    val <- val/attr(val, "std")
    attr(val, "label") <- "Standardized residuals"
    if (type == "normalized") {
      if (!is.null(cSt <- object$modelStruct$corStruct)) {
        ## normalize according to inv-trans factor
        Xy <- as.matrix(val)
        if (!is.null(ord)) {
          ## need to be careful with ordering
          Xy <- Xy[ord$order, , drop = F]
        }
        val <- recalc(cSt, list(Xy = Xy))$Xy[, 1]
        if (!is.null(ord)) {
          val <- val[ord$revOrder]
        }
        val <- as.vector(val)
        attr(val, "label") <- "Normalized residuals"
      }
    } 
  } else {
    lab <- "Residuals"
    if (!is.null(aux <- attr(object, "units")$y)) {
      lab <- paste(lab, aux)
    }
    attr(val, "label") <- lab
  }
  val
}

summary.gls <- function(object, verbose = F) {
  ##
  ## generates an object used in the print.summary method for lme
  ##
  ##  variance-covariance estimates for coefficients
  ##
  stdBeta <- sqrt(diag(as.matrix(object$varBeta)))
  corBeta <- t(object$varBeta/stdBeta)/stdBeta
  ##
  ## coefficients, std. deviations and z-ratios
  ##
  beta <- coef(object)
  dims <- object$dims
  dimnames(corBeta) <- list(names(beta),names(beta))
  object$corBeta <- corBeta
  tTable <- data.frame(beta, stdBeta, beta/stdBeta, beta)
  dimnames(tTable)<-
    list(names(beta),c("Value","Std.Error","t-value","p-value"))
  tTable[, "p-value"] <- 2 * pt(-abs(tTable[,"t-value"]), dims$N - dims$p)
  object$tTable <- as.matrix(tTable)
  ##
  ## residuals
  ##
  resd <- resid(object, type = "pearson")
  if (length(resd) > 5) {
    resd <- quantile(resd)
    names(resd) <- c("Min","Q1","Med","Q3","Max")
  }
  object$residuals <- resd
  ##
  ## generating the final object
  ##
  aux <- logLik(object)
  object$BIC <- BIC(aux)
  object$AIC <- AIC(aux)
  attr(object, "verbose") <- verbose
  class(object) <- c("summary.gls", class(object))
  object
}

update.gls <-
  function(object, model, data, correlation, weights, subset, method,
	   na.action, control, verbose)
{
  thisCall <- as.list(match.call())[-(1:2)]
  nextCall <- as.list(object$call)[-1]
  if (is.na(match("correlation", names(thisCall))) &&
      !is.null(thCor <- object$modelStruct$corStruct)) {
    if (!is.null(thisCall$data) || !is.null(thisCall$subset) ||
        !is.null(thisCall$na.action)) {
      ## forcing initialization
      isInitialized(thCor) <- FALSE
    }
    thisCall$correlation <- thCor
  }
  if (is.na(match("weights", names(thisCall))) &&
      !is.null(thWgt <- object$modelStruct$varStruct)) {
    if (!is.null(thisCall$data) || !is.null(thisCall$subset) ||
        !is.null(thisCall$na.action)) {
      ## forcing initialization
      isInitialized(thWgt) <- FALSE
    }
    thisCall$weights <- thWgt
  }
  if (!is.null(thisCall$model)) {
    thisCall$model <- update(as.formula(nextCall$model), thisCall$model)
  }
  nextCall[names(thisCall)] <- thisCall
  do.call("gls", nextCall)
}

Variogram.gls <-
  function(object, distance, form = ~1,
           resType = c("pearson", "response", "normalized"),
           data, na.action = na.fail, maxDist, length.out = 50,
           collapse = c("quantiles", "fixed", "none"), nint = 20, breaks,
           robust = FALSE, metric = c("euclidean", "maximum", "manhattan"))
{
  resType <- match.arg(resType)
  ## checking if object has a corSpatial element
  csT <- object$modelStruct$corStruct
  wchRows <- NULL
  if (missing(distance)) {
    if (missing(form) && inherits(csT, "corSpatial")) {
      distance <- getCovariate(csT)
      grps <- getGroups(object)
    } else {
      metric <- match.arg(metric)
      if (missing(data)) {
        data <- getData(object)
      }
      if (is.null(data)) {			# will try to construct
        allV <- all.vars(form)
        if (length(allV) > 0) {
          alist <- lapply(as.list(allV), as.name)
          names(alist) <- allV
          alist <- c(as.list(as.name("data.frame")), alist)
          mode(alist) <- "call"
          data <- eval(alist, sys.parent(1))
        } 
      }
      grpsF <- getGroupsFormula(form)
      grps <- NULL
      if (is.null(grpsF) || is.null(grps <- getGroups(data, grpsF))) {
        ## try to get from object
        grps <- getGroups(object)
      }
      covForm <- getCovariateFormula(form)
      if (length(all.vars(covForm)) > 0) {
        if (attr(terms(covForm), "intercept") == 1) {
          covForm <-
            eval(parse(text = paste("~", deparse(covForm[[2]]),"-1",sep="")))
        }
        covar <- model.frame(covForm, data, na.action = na.action)
        ## making sure grps is consistent
        wchRows <- !is.na(match(row.names(data), row.names(covar)))
        if (!is.null(grps)) {
          grps <- pruneLevels(grps[wchRows])
        }
        covar <- as.data.frame(unclass(model.matrix(covForm, covar)))
      } else {
        if (is.null(grps)) {
          covar <- 1:nrow(data)
        } else {
          covar <-
            data.frame(dist = unlist(tapply(rep(1, nrow(data)), grps, cumsum)))
        }
      }
      if (is.null(grps)) {
        distance <- dist(as.matrix(covar), metric = metric)
      } else {
        covar <- split(covar, grps)
        ## getting rid of 1-observation groups
        covar <- covar[sapply(covar, function(el) nrow(as.matrix(el))) > 1]
        distance <- lapply(covar,
                           function(el, metric) dist(as.matrix(el), metric),
                           metric = metric)
      }
    }
  }
  res <- resid(object, type = resType)
  if (!is.null(wchRows)) {
    res <- res[wchRows]
  }
  if (is.null(grps)) {
    val <- Variogram(res, distance)
  } else {
    res <- split(res, grps)
    res <- res[sapply(res, length) > 1] # no 1-observation groups
    levGrps <- levels(grps)
    val <- structure(vector("list", length(levGrps)), names = levGrps)
    for(i in levGrps) {
      val[[i]] <- Variogram(res[[i]], distance[[i]])
    }
    val <- do.call("rbind", val)
  }
  if (!missing(maxDist)) {
    val <- val[val$dist <= maxDist, ]
  }
  collapse <- match.arg(collapse)
  if (collapse != "none") {             # will collapse values
    dst <- val$dist
    udist <- sort(unique(dst))
    ludist <- length(udist)
    if (!missing(breaks)) {
      if (min(breaks) > udist[1]) {
        breaks <- c(udist[1], breaks)
      }
      if (max(breaks) < udist[2]) {
        breaks <- c(breaks, udist[2])
      }
      if (!missing(nint) && nint != (length(breaks) - 1)) {
        stop("Nint is not consistent with breaks.")
      }
      nint <- length(breaks) - 1
    }
    if (nint < ludist) {
      if (missing(breaks)) {
        if (collapse == "quantiles") {    # break into equal groups
          breaks <- unique(quantile(dst, seq(0, 1, 1/nint)))
        } else {                          # fixed length intervals
          breaks <- seq(udist[1], udist[length(udist)], length = nint + 1)
        }
      }
      cutDist <- cut(dst, breaks)
    } else {
      cutDist <- dst
    }
    val <- lapply(split(val, cutDist),
                  function(el, robust) {
                    nh <- nrow(el)
                    vrg <- el$variog
                    if (robust) {
                      vrg <- ((mean(vrg^0.25))^4)/(0.457+0.494/nh)
                    } else {
                      vrg <- mean(vrg)
                    }
                    dst <- median(el$dist)
                    data.frame(variog = vrg, dist = dst)
                  }, robust = robust)
    val <- do.call("rbind", as.list(val))
    val$n.pairs <- table(na.omit(cutDist))
  }
  row.names(val) <- 1:nrow(val)
  if (inherits(csT, "corSpatial") && resType != "normalized") {
    ## will keep model variogram
    if (resType == "pearson") {
      sig2 <- 1
    } else {
      sig2 <- object$sigma^2
    }
    attr(val, "modelVariog") <-
      Variogram(csT, sig2 = sig2, length.out = length.out)
  }
  attr(val, "collapse") <- collapse != "none"
  class(val) <- c("Variogram", "data.frame")
  val
}

###*### glsStruct - a model structure for gls fits

glsStruct <-
  ## constructor for glsStruct objects
  function(corStruct = NULL, varStruct = NULL)
{
  val <- list(corStruct = corStruct, varStruct = varStruct)
  val <- val[!sapply(val, is.null)]	# removing NULL components
  class(val) <- c("glsStruct", "modelStruct")
  val
}

##*## glsStruct methods for standard generics

fitted.glsStruct <-
  function(object, glsFit = attr(object, "glsFit"))
{
  glsFit[["fitted"]]
}

initialize.glsStruct <-
  function(object, data, control = list(singular.ok = F,
                           qrTol = .Machine$single.eps))
{
  if (length(object)) {
    object[] <- lapply(object, initialize, data)
    theta <- lapply(object, coef)
    len <- unlist(lapply(theta, length))
    num <- seq(along = len)
    if (sum(len) > 0) {
      pmap <- outer(rep(num, len), num, "==")
    } else {
      pmap <- array(F, c(1, length(len)))
    }
    dimnames(pmap) <- list(NULL, names(object))
    attr(object, "pmap") <- pmap
    attr(object, "glsFit") <- 
      glsEstimate(object, control = control)
    if (needUpdate(object)) {
      object <- update(object, data)
    } 
  }
  object
}

logLik.glsStruct <-
  function(object, Pars, conLin = attr(object, "conLin"))
{
  coef(object) <- Pars			# updating parameter values
  conLin <- recalc(object, conLin)	# updating conLin
  .C("gls_loglik",
     as.double(conLin[["Xy"]]),
     as.integer(unlist(conLin[["dims"]])),
     logLik = as.double(conLin[["logLik"]]),
     double(1),
     as.double(conLin$sigma))[["logLik"]]
}

residuals.glsStruct <-
  function(object, glsFit = attr(object, "glsFit"))
{
  glsFit[["resid"]]
}

varWeights.glsStruct <-
  function(object)
{
  if (is.null(object$varStruct)) rep(1, attr(object, "conLin")$dims$N)
  else varWeights(object$varStruct)
}

## Auxiliary control functions

glsControl <-
  ## Control parameters for gls
  function(maxIter = 50, msMaxIter = 50, tolerance = 1e-6, msTol = 1e-7, 
	   msScale = lmeScale, msVerbose = F, singular.ok = F, 
	   qrTol = .Machine$single.eps, returnObject = F,
	   apVar = TRUE, .relStep = (.Machine$double.eps)^(1/3),
           minAbsParApVar = 0.05, natural = TRUE, natUnconstrained = TRUE,
           sigma = NULL)
{
  if (is.logical(natural)) {
    ## repeat for all structures
    natural <- list(corStruct = natural, varStruct = natural)
  }
  if (is.logical(natUnconstrained)) {
    ## repeat for all structures
    natUnconstrained <- list(corStruct = natUnconstrained,
            varStruct = natUnconstrained, sig = natUnconstrained)
  }
  if (!is.null(sigma)) {
    if (!is.numeric(sigma) || (length(sigma) != 1) || (sigma <= 0)) {
      stop("Within-group std. dev. must be a positive numeric value")
    }
  } else {
    sigma <- 0
  }
  
  list(maxIter = maxIter, msMaxIter = msMaxIter, tolerance = tolerance,
       msTol = msTol, msScale = msScale, msVerbose = msVerbose, 
       singular.ok = singular.ok, qrTol = qrTol, 
       returnObject = returnObject, apVar = apVar,
       minAbsParApVar = minAbsParApVar, .relStep = .relStep,
       natural = natural, natUnconstrained = natUnconstrained, sigma = sigma)
}

### local generics for objects inheriting from class lme




## Local Variables:
## mode:S
## End:



