\name{S4R}
\alias{ErrorClass}
\alias{usingR}

\title{R compatibility with S version 4/S-Plus 5+ support functions}
\description{
  These objects ease the task of porting functions into R
  from S Version 4 and S-Plus 5.0 and later.  See the documentation
  of the lower-case functions there. May be obsolete in the future.
}
\usage{
usingR(major, minor)
}
\keyword{internal}
% vim:syntax=tex
\eof
\name{SQLite}
\alias{SQLite}
\alias{SQLiteDriver}
\title{
  Instantiate the SQLite engine from the current R/S-Plus session.
}
\description{
This function creates and initializes the SQLite engine.
It returns an object that allows you to connect to the SQLite 
embedded engine.
}
\usage{
SQLite(max.con = 16, fetch.default.rec = 500, force.reload = F)
}
\arguments{
\item{max.con }{
  maximum number of connections that may be open at one time.
  This can be up to 100, a limit defined at compilation time.
  Note that since the SQLite engine is embedded (i.e., a set of C 
  functions within the R/S-Plus process) connections consume very
  little resources.
  }
\item{fetch.default.rec}{
  number of records to fetch at one time from the database.
  (The \code{fetch} method uses this number as a default.)
  }
\item{force.reload}{
  should the package code be reloaded (reinitialized)?
  Setting this to \code{TRUE} allows you to change
  default settings.  Notice that all connections should be closed
  before re-loading.
  }
}
\value{
An object of class \code{SQLiteDriver} which extends 
\code{dbDriver} and \code{dbObjectId}.
This object is required to create connections
to the embedded SQLite database.
There can be many SQLite database instances running
simultaneously.
}
\section{Side Effects}{
The R/S-Plus client part of the database communication is initialized,
but note that connecting to database instances needs to be done through
calls to \code{dbConnect}.
}
\details{
This object is a singleton, that is, on subsequent invocations
it returns the same initialized object. 

This implementation allows the R/S-Plus embedded SQLite engine
to work with multiple database instances through multiple connections 
simultaneously.

SQLite keeps each database instance in one single file. The name
of the database \emph{is} the file name, thus database names
should be legal file names in the running platform.
}
\section{User authentication}{
SQLite is a single-user database engine, so no authentication
is required.
}
\author{David A. James}
\section{References}{
See the Omega Project for Statistical Computing
at \url{http://www.omegahat.org} and \url{http://stat.bell-labs.com/RS-DBI}
for more details on the R/S-Plus database interface.

See the Adobe PDF file \code{DBI.pdf} under the \code{doc}
subdirectory of the DBI package, i.e., 
\code{system.file("doc", "DBI.pdf", package = "DBI")}

See the documentation at the SQLite Web site
\url{http://www.hwaci/com/sw/sqlite/index.html} for details.
}
\seealso{
On database drivers:

\code{\link[DBI]{dbDriver}},
\code{\link[DBI]{dbUnloadDriver}},
\code{\link[DBI]{dbListConnections}}.

On connections, SQL statements and resultSets:

\code{\link[DBI]{dbConnect}},
\code{\link[DBI]{dbDisconnect}},
\code{\link[DBI]{dbSendQuery}},
\code{\link[DBI]{dbGetQuery}},
\code{\link[DBI]{fetch}},
\code{\link[DBI]{dbListResults}}.

On transaction management:

\code{\link[DBI]{dbCommit}}, 
\code{\link[DBI]{dbRollback}}.

On meta-data:

\code{\link{summary}},
\code{\link[DBI]{dbGetInfo}},
\code{\link[DBI]{dbListTables}},
\code{\link[DBI]{dbListFields}},
\code{\link[DBI]{dbColumnsInfo}},
\code{\link[DBI]{dbGetException}},
\code{\link[DBI]{dbGetStatement}},
\code{\link[DBI]{dbHasCompleted}},
\code{\link[DBI]{dbGetRowCount}},
\code{\link[DBI]{dbGetAffectedRows}}.
}
\examples{\dontrun{
   # create a SQLite instance and create one connection.
   m <- dbDriver("SQLite")
   
   # initialize a new database "base.dbms" in the current directory
   # and copy some data.frame from the base package into it
   
   con <- dbConnect(m, dbname = "base.dbms")
   data(USArrests)
   dbWriteTable(con, "USArrests", USArrests, overwrite = T)
   
   # query
   rs <- dbSendQuery(con, "select * from USArrests")
   d1 <- fetch(rs, n = 10)      # extract data in chunks of 10 rows
   dbHasCompleted(rs)
   d2 <- fetch(rs, n = -1)      # extract all remaining data
   dbHasCompleted(rs)
   dbClearResult(rs)
   dbListTables(con)    
   }
}
\keyword{interface}
\keyword{database}
% docclass is function
%  vim: syntax=tex
\eof
\name{SQLiteConnection-class}
\docType{class}
\alias{SQLiteConnection-class}
\title{Class SQLiteConnection}
\description{SQLite connection class.}
\section{Generators}
    The method \code{\link[DBI]{dbConnect}} is the main generator.
}
\section{Extends}{
Class \code{"DBIConnection"}, directly.
Class \code{"SQLiteObject"}, directly.
Class \code{"DBIObject"}, by class "DBIConnection".
Class \code{"dbObjectId"}, by class "SQLiteObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "SQLiteConnection", to = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbCallProc}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbCommit}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbConnect}}{\code{signature(drv = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbDisconnect}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbExistsTable}}{\code{signature(conn = "SQLiteConnection", name = "character")}: ... }
    \item{\link[DBI]{dbGetException}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbGetQuery}}{\code{signature(conn = "SQLiteConnection", statement = "character")}: ... }
    \item{\link[DBI]{dbListFields}}{\code{signature(conn = "SQLiteConnection", name = "character")}: ... }
    \item{\link[DBI]{dbListResults}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbListTables}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbReadTable}}{\code{signature(conn = "SQLiteConnection", name = "character")}: ... }
    \item{\link[DBI]{dbRemoveTable}}{\code{signature(conn = "SQLiteConnection", name = "character")}: ... }
    \item{\link[DBI]{dbRollback}}{\code{signature(conn = "SQLiteConnection")}: ... }
    \item{\link[DBI]{dbSendQuery}}{\code{signature(conn = "SQLiteConnection", statement = "character")}: ... }
    \item{\link[DBI]{dbWriteTable}}{\code{signature(conn = "SQLiteConnection", name = "character", value = "data.frame")}: ... }
    \item{summary}{\code{signature(object = "SQLiteConnection")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\author{R-SIG-DB}

\seealso{
 DBI classes:
 \code{\link{SQLiteObject-class}}
 \code{\link{SQLiteDriver-class}}
 \code{\link{SQLiteConnection-class}}
 \code{\link{SQLiteResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\eof
\name{SQLiteDriver-class}
\docType{class}
\alias{SQLiteDriver-class}
\title{Class SQLiteDriver}
\description{
   An SQLite driver implementing the R/S-Plus database (DBI) API.
}
\section{Generators}{
   The main generators are \code{\link[DBI]{dbDriver}} and 
   \code{\link{SQLite}}.
}
\section{Extends}{
Class \code{"DBIDriver"}, directly.
Class \code{"SQLiteObject"}, directly.
Class \code{"DBIObject"}, by class "DBIDriver".
Class \code{"dbObjectId"}, by class "SQLiteObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "SQLiteObject", to = "SQLiteDriver")}: ... }
    \item{\link[DBI]{dbConnect}}{\code{signature(drv = "SQLiteDriver")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "SQLiteDriver")}: ... }
    \item{\link[DBI]{dbListConnections}}{\code{signature(drv = "SQLiteDriver")}: ... }
    \item{\link[DBI]{dbUnloadDriver}}{\code{signature(drv = "SQLiteDriver")}: ... }
    \item{\link{summary}}{\code{signature(object = "SQLiteDriver")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\author{R-SIG-DB}

\seealso{
 DBI classes:
 \code{\link{SQLiteObject-class}}
 \code{\link{SQLiteDriver-class}}
 \code{\link{SQLiteConnection-class}}
 \code{\link{SQLiteResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, "user/password@dbname")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\eof
\name{SQLiteObject-class}
\docType{class}
\alias{SQLiteObject-class}
\title{Class SQLiteObject}
\description{
   Base class for all SQLite-specific DBI classes
}
\section{Objects from the Class}{
   A virtual Class: No objects may be created from it.
}
\section{Extends}{
Class \code{"DBIObject"}, directly.
Class \code{"dbObjectId"}, directly.
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "SQLiteObject", to = "SQLiteDriver")}: ... }
    \item{\link[DBI]{dbDataType}}{\code{signature(dbObj = "SQLiteObject")}: ... }
    \item{\link[DBI]{isSQLKeyword}}{\code{signature(dbObj = "SQLiteObject", name = "character")}: ... }
    \item{\link[DBI]{make.db.names}}{\code{signature(dbObj = "SQLiteObject", snames = "character")}: ... }
    \item{\link[DBI]{SQLKeywords}}{\code{signature(dbObj = "SQLiteObject")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\author{R-SIG-DB}

\seealso{
 DBI classes:
 \code{\link{SQLiteObject-class}}
 \code{\link{SQLiteDriver-class}}
 \code{\link{SQLiteConnection-class}}
 \code{\link{SQLiteResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\eof
\name{SQLiteResult-class}
\docType{class}
\alias{SQLiteResult-class}
\title{Class SQLiteResult}
\description{
   SQLite's query results class.  This classes encapsulates the
   result of an SQL statement (either \code{select} or not).
}
\section{Generators}{
   The main generator is \code{\link[DBI]{dbSendQuery}}.
}
\section{Extends}{
Class \code{"DBIResult"}, directly.
Class \code{"SQLiteObject"}, directly.
Class \code{"DBIObject"}, by class "DBIResult".
Class \code{"dbObjectId"}, by class "SQLiteObject".
}
\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "SQLiteConnection", to = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbClearResult}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbColumnInfo}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbGetException}}{\code{signature(conn = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbGetInfo}}{\code{signature(dbObj = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbGetRowCount}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbGetRowsAffected}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbGetStatement}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbHasCompleted}}{\code{signature(res = "SQLiteResult")}: ... }
    \item{\link[DBI]{dbListFields}}{\code{signature(conn = "SQLiteResult", name = "missing")}: ... }
    \item{\link[DBI]{fetch}}{\code{signature(res = "SQLiteResult", n = "numeric")}: ... }
    \item{\link[DBI]{fetch}}{\code{signature(res = "SQLiteResult", n = "missing")}: ... }
    \item{\link{summary}}{\code{signature(object = "SQLiteResult")}: ... }
  }
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\author{R-SIG-DB}

\seealso{
 DBI classes:
 \code{\link{SQLiteObject-class}}
 \code{\link{SQLiteDriver-class}}
 \code{\link{SQLiteConnection-class}}
 \code{\link{SQLiteResult-class}}
}

\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, dbname = "rsdbi.db")
}
}
\keyword{database}
\keyword{interface}
\keyword{classes}
% vim: syntax=tex
\eof
% $Id: dbCallProc-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbCallProc-methods}
\docType{methods}
\alias{dbCallProc-methods}
\alias{dbCallProc,SQLiteConnection-method}
\title{
  Call an SQL stored procedure
}
\description{
  Not yet implemented.
}
\section{Methods}{\describe{
\item{conn}{
  a \code{SQLiteConnection} object.
  }
\item{\dots }{
  additional arguments are passed to the implementing method.
  }
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
   \code{\link{SQLite}},
   \code{\link[DBI]{dbConnect}},
   \code{\link[DBI]{dbSendQuery}},
   \code{\link[DBI]{dbGetQuery}},
   \code{\link[DBI]{fetch}},
   \code{\link[DBI]{dbCommit}},
   \code{\link[DBI]{dbGetInfo}},
   \code{\link[DBI]{dbReadTable}}.
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbCommit-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbCommit-methods}
\docType{methods}
\alias{dbCommit-methods}
\alias{dbRollback-methods}
\alias{dbCommit,SQLiteConnection-method}
\alias{dbRollback,SQLiteConnection-method}
\title{
  DBMS Transaction Management
}
\description{
   Not yet implemented.
}
\section{Methods}{\describe{
\item{conn}{
  a \code{SQLiteConnection} object, as produced by the function 
  \code{dbConnect}.
  }
\item{\dots }{
  any database-specific arguments.
  }
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or
  \url{http://developer.r-project.org/db}.
}
\seealso{
   \code{\link{SQLite}},
   \code{\link[DBI]{dbConnect}},
   \code{\link[DBI]{dbSendQuery}},
   \code{\link[DBI]{dbGetQuery}},
   \code{\link[DBI]{fetch}},
   \code{\link[DBI]{dbCommit}},
   \code{\link[DBI]{dbGetInfo}},
   \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, dbname = "mySQLite.db")
rs <- dbSendQuery(con, 
      "delete * from PURGE as p where p.wavelength<0.03")
if(dbGetInfo(rs, what = "rowsAffected") > 250){
  warning("dubious deletion -- rolling back transaction")
  dbRollback(con)
}
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbConnect-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbConnect-methods}
\docType{methods}
\alias{dbDisconnect-methods}
\alias{dbConnect-methods}
\alias{dbDisconnect,SQLiteConnection-method}
\alias{dbConnect,SQLiteDriver-method}
\alias{dbConnect,SQLiteConnection-method}
\alias{dbConnect,character-method}
\title{
  Create a connection object to an SQLite DBMS
}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{
\item{drv}{
  an object of class \code{SQLiteDriver}, or
  the character string "SQLite" or an \code{SQLiteConnection}.
  }
\item{conn}{
  an \code{SQLiteConnection} object as produced by \code{dbConnect}.
  }
\item{\dots }{
  In the case of SQLite all you need is \code{dbname}, which
  should refer to a file name.
  }
}
}
\section{Side Effects}{
  A connection between R/S-Plus and the embeddable SQLite server is 
  established.  Note that since the SQLite is embedded in R/S-Plus,
  connections are not too resource hungry.

  SQLite connections only require the file name where the SQLite
  database reside.  For details see \code{\link{SQLite}}.
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{dbGetQuery}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
# create an SQLite instance and create one connection.
drv <- dbDriver("SQLite")

# open the connection using user, passsword, etc., as
con <- dbConnect(drv, dbname = "sqlite.db")

# Run an SQL statement by creating first a resultSet object
rs <- dbSendQuery(con, statement = paste(
                      "SELECT w.laser_id, w.wavelength, p.cut_off",
                      "FROM WL w, PURGE P", 
                      "WHERE w.laser_id = p.laser_id", 
                      "SORT BY w.laser_id")
# we now fetch records from the resultSet into a data.frame
data <- fetch(rs, n = -1)   # extract all rows
dim(data)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbDataType-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbDataType-methods}
\docType{methods}
\alias{dbDataType-methods}
\alias{dbDataType,SQLiteObject-method}
\title{
  Determine the SQL Data Type of an S object
}
\description{
   This method is a straight-forward implementation of the corresponding
   generic function.
}
\section{Methods}{\describe{
\item{dbObj}{
  a \code{SQLiteDriver} object, e.g., \code{ODBCDriver},
  \code{OracleDriver}.
  }
\item{obj}{
  R/S-Plus object whose SQL type we want to determine.
  }
\item{\dots }{
  any other parameters that individual methods may need.
  }
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link[DBI]{isSQLKeyword}}
  \code{\link[DBI]{make.db.names}}
}
\examples{\dontrun{
data(quakes)
drv <- dbDriver("SQLite")
sql.type <- dbDataType(drv, quakes)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% docclass is function
% Converted by Sd2Rd version 1.15.2.1.
% vim: syntax=tex
\eof
% $Id: dbDriver-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbDriver-methods}
\docType{methods}
\alias{dbDriver-methods}
\alias{dbUnloadDriver-methods}
\alias{dbDriver,character-method}
\alias{dbUnloadDriver,SQLiteDriver-method}
\title{
  SQLite implementation of the Database Interface (DBI) classes 
  and drivers 
}
\description{
  SQLite driver initialization and closing
}
\section{Methods}{\describe{
\item{drvName}{
  character name of the driver to instantiate.
  }
\item{drv}{
  an object that inherits from \code{SQLiteDriver} as created by
  \code{dbDriver}.
}
\item{...}{
  any other arguments are passed to the driver \code{drvName}.
  }
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or
  \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{dbGetQuery}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbListTables}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
# create an SQLite instance for capacity of up to 25 simultaneous
# connections.
m <- dbDriver("SQLite", max.con = 25)

con <- dbConnect(m, dbname="sqlite.db")
rs <- dbSubmitQuery(con, 
         "select * from HTTP_ACCESS where IP_ADDRESS = '127.0.0.1'")
df <- fetch(rs, n = 50)
df2 <- fetch(rs, n = -1)
dbClearResult(rs)

pcon <- dbConnect(p, "user", "password", "dbname")
dbListTables(pcon)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbGetInfo-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbGetInfo-methods}
\docType{methods}
\alias{dbGetInfo}
\alias{dbGetDBIVersion-methods}
\alias{dbGetStatement-methods}
\alias{dbGetRowCount-methods}
\alias{dbGetRowsAffected-methods}
\alias{dbColumnInfo-methods}
\alias{dbHasCompleted-methods}
\alias{dbGetInfo,SQLiteObject-method}
\alias{dbGetInfo,SQLiteDriver-method}        % BUG: this is not needed
\alias{dbGetInfo,SQLiteConnection-method}        % BUG: this is not needed
\alias{dbGetInfo,SQLiteResult-method}        % BUG: this is not needed
\alias{dbGetStatement,SQLiteResult-method}
\alias{dbGetRowCount,SQLiteResult-method}
\alias{dbGetRowsAffected,SQLiteResult-method}
\alias{dbColumnInfo,SQLiteResult-method}
\alias{dbHasCompleted,SQLiteResult-method}
\title{
  Database interface meta-data
}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{
\item{dbObj}{
  any object that implements some functionality in the R/S-Plus
  interface to databases (a driver, a connection or a result set).
  }
%\item{drv}{an \code{SQLiteDriver}.}
%\item{conn}{an \code{SQLiteConnection}.}
\item{res}{ an \code{SQLiteResult}.}
\item{\dots}{currently not being used.}
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbDriver}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{dbGetQuery}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbListTables}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv)

dbListTables(con)

rs <- dbSendQuery(con, query.sql)
dbGetStatement(rs)
dbHasCompleted(rs)

info <- dbGetInfo(rs)
names(dbGetInfo(drv))  

# DBIConnection info
names(dbGetInfo(con))

# DBIResult info
names(dbGetInfo(rs)) 
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbListTables-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbListTables-methods}
\docType{methods}
\alias{dbListTables-methods}
\alias{dbListFields-methods}
\alias{dbListConnections-methods}
\alias{dbListResults-methods}
\alias{dbListTables,SQLiteConnection-method}
\alias{dbListFields,SQLiteConnection,character-method}
\alias{dbListConnections,SQLiteDriver-method}
\alias{dbListResults,SQLiteConnection-method}
\title{
  List items from an SQLite DBMS and from objects 
}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{
  \item{drv}{an \code{SQLiteDriver}.}
  \item{conn}{an \code{SQLiteConnection}.}
  \item{name}{a character string with the table name.}
  \item{\dots}{currently not used.}
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or
  \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbColumnInfo}},
  \code{\link[DBI]{dbDriver}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}}
}

\examples{\dontrun{
drv <- dbDriver("SQLite")
# after working awhile...
for(con in dbListConnections(odbc)){
   dbGetStatement(dbListResults(con))
}
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
\name{dbObjectId-class}
\docType{class}
\alias{dbObjectId-class}
\title{Class dbObjectId}
\description{  
   A helper (mixin) class to provide external references in
   an R/S-Plus portable way.
}
\section{Objects from the Class}{A virtual Class: No objects may be created from it.}
\section{Slots}{
  \describe{
    \item{\code{Id}:}{Object of class \code{"integer"} 
    this is an integer vector holding an opaque reference into a C struct
    (may or may not be a C pointer, may or may not have length one).
    }
  }
}

\section{Methods}{
  \describe{
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "integer")}: ... }
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "numeric")}: ... }
    \item{\link{coerce}}{\code{signature(from = "dbObjectId", to = "character")}: ... }
    \item{\link{format}}{\code{signature(x = "dbObjectId")}: ... }
    \item{\link{print}}{\code{signature(x = "dbObjectId")}: ... }
    \item{\link{show}}{\code{signature(object = "dbObjectId")}: ... }
  }
}

\note{A cleaner mechanism would use external references, but
historically this class has existed mainly for R/S-Plus portability.}

\examples{\dontrun{
  pg <- dbDriver("PostgreSQL")
  con <- dbConnect(pg, "user", "password")
  is(pg, "dbObjectId")   ## True
  is(con, "dbObjectId")  ## True
  isIdCurrent(con)       ## True
  q("yes")
  \$ R 
  isIdCurrent(con)       ## False
}
}
\keyword{classes}
% vim: syntax=tex
\eof
% $Id: dbReadTable-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbReadTable-methods}
\docType{methods}
\alias{dbReadTable-methods}
\alias{dbWriteTable-methods}
\alias{dbExistsTable-methods}
\alias{dbRemoveTable-methods}
\alias{dbReadTable,SQLiteConnection,character-method}
\alias{dbWriteTable,SQLiteConnection,character,data.frame-method}
\alias{dbExistsTable,SQLiteConnection,character-method}
\alias{dbRemoveTable,SQLiteConnection,character-method}
\title{
  Convenience functions for Importing/Exporting DBMS tables
}
\description{
  These functions mimic their R/S-Plus counterpart 
  \code{get},
  \code{assign},
  \code{exists}, 
  \code{remove}, and
  \code{objects},
  except that they generate code that gets remotely executed
  in a database engine.
}
\section{Methods}{\describe{
\item{conn}{
  an \code{SQLiteConnection} database connection object.
  }
\item{name}{
  a character string specifying a table name.
  }
\item{value}{
  a data.frame (or coercible to data.frame).
  }
\item{row.names}{
  in the case of \code{dbReadTable}, this argument can be a string or
  an index specifying the column in the DBMS table to be used as 
  \code{row.names} in the output data.frame (a \code{NULL}, \code{""}, or 0 
  specifies that no column should be used as \code{row.names} in the output).

  In the case of \code{dbWriteTable}, this argument should be a logical
  specifying whether the \code{row.names} should be output to the
  output DBMS table; if \code{TRUE}, an extra field whose name will be 
  whatever the R/S-Plus identifier \code{"row.names"} maps to the DBMS
  (see \code{\link[DBI]{make.db.names}}).
  }
\item{overwrite}{
  a logical specifying whether to overwrite an existing table
  or not.  Its default is \code{FALSE}.
  }
\item{append}{
  a logical specifying whether to append to an existing table
  in the DBMS.  Its default is \code{FALSE}.
  }
\item{\dots }{any optional arguments.}
}
}

\value{
   A data.frame in the case of \code{dbReadTable}; otherwise a logical 
   indicating whether the operation was successful.  
}
\note{
   Note that the data.frame returned by \code{dbReadTable} only has
   primitive data, e.g., it does not coerce character data to factors.

   SQLite table names are \emph{not} case sensitive, e.g., table
   names \code{ABC} and \code{abc} are considered equal.
}
\references{
   See the Database Interface definition document
   \code{DBI.pdf} in the base directory of this package
   or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbDriver}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{dbGetQuery}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbListTables}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
conn <- dbConnect("SQLite", dbname = "sqlite.db")
if(dbExistsTable(con, "fuel_frame")){
   dbRemoveTable(conn, "fuel_frame")
   dbWriteTable(conn, "fuel_frame", fuel.frame)
}
if(dbExistsTable(conn, "RESULTS")){
   dbWriteTable(conn, "RESULTS", results2000, append = T)
else
   dbWriteTable(conn, "RESULTS", results2000)
}
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbSendQuery-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbSendQuery-methods}
\docType{methods}
\alias{dbSendQuery-methods}
\alias{dbGetQuery-methods}
\alias{dbClearResult-methods}
\alias{dbGetException-methods}
\alias{dbSendQuery,SQLiteConnection,character-method}
\alias{dbGetQuery,SQLiteConnection,character-method}
\alias{dbClearResult,SQLiteResult-method}
\alias{dbGetException,SQLiteConnection-method}
\title{
  Execute a statement on a given database connection
}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{
\item{conn}{
  an \code{SQLiteConnection} object.
  }
\item{statement}{a character vector of length 1 with the SQL statement.}
\item{res}{an \code{SQLiteResult} object.}
\item{\dots }{additional parameters.}
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbDriver}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
drv <- dbDriver("MySQL")
con <- dbConnect(drv)
res <- dbSendQuery(con, "SELECT * from liv25")
data <- fetch(res, n = -1)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: dbSetDataMappings-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{dbSetDataMappings-methods}
\docType{methods}
\alias{dbSetDataMappings-methods}
\alias{dbSetDataMappings,SQLiteResult,data.frame-method}
\title{
  Set data mappings between SQLite and R/S-Plus
}
\description{
  Not yet implemented
}
\section{Methods}{\describe{
\item{res}{
  a \code{SQLiteResult} object as returned by \code{dbSendQuery}.
  }
\item{flds}{
  a data.frame with field descriptions as returned by 
  \code{\link[DBI]{dbColumnInfo}}.
  }
\item{\dots }{
  any additional arguments are passed to the implementing method.
  }
}
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{fetch}},
  \code{\link[DBI]{dbColumnInfo}}.
}
\examples{\dontrun{
makeImage <- function(x) {
  .C("make_Image", as.integer(x), length(x))
}

res <- dbSendQuery(con, statement)
flds <- dbColumnInfo(res)
flds[3, "Sclass"] <- makeImage

dbSetDataMappings(rs, flds)

im <- fetch(rs, n = -1)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
% $Id: fetch-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{fetch-methods}
\docType{methods}
\alias{fetch-methods}
\alias{fetch,SQLiteResult,numeric-method}
\alias{fetch,SQLiteResult-method}
\title{
  Fetch records from a previously executed query
}
\description{
   This method is a straight-forward implementation of the corresponding
   generic function.
}
\section{Methods}{\describe{

\item{res}{
  an \code{SQLiteResult} object.
  }
\item{n}{
  maximum number of records to retrieve per fetch.
  Use \code{n = -1} to retrieve all pending records;
  use a value of \code{n = 0} for fetching the default number 
  of rows \code{fetch.default.rec} defined in the 
  \code{\link{SQLite}} initialization invocation.
  }
\item{\dots }{currently not used.}
}
}
\details{
  The \code{RSQLite} implementations retrieves all records into a 
  buffer internally managed by the RSQLite driver (thus this memory
  in not managed by R but its part of the R process), and
  \code{fetch} simple returns records from this internal buffer.
}
\references{
  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbConnect}},
  \code{\link[DBI]{dbSendQuery}},
  \code{\link[DBI]{dbGetQuery}},
  \code{\link[DBI]{dbClearResult}},
  \code{\link[DBI]{dbCommit}},
  \code{\link[DBI]{dbGetInfo}},
  \code{\link[DBI]{dbReadTable}}.
}
\examples{\dontrun{
drv <- dbDriver("SQLite")
con <- dbConnect(drv, dbname = "sqlite.db")
res <- dbSendQuery(con, statement = paste(
                      "SELECT w.laser_id, w.wavelength, p.cut_off",
                      "FROM WL w, PURGE P", 
                      "WHERE w.laser_id = p.laser_id",
                      "ORDER BY w.laser_id"))
# we now fetch the first 100 records from the resultSet into a data.frame
data1 <- fetch(res, n = 100)   
dim(data1)

dbHasCompleted(res)

# let's get all remaining records
data2 <- fetch(res, n = -1)
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
\name{isIdCurrent}
\alias{isIdCurrent}
\title{
  Check whether an dbObjectId handle object is valid or not
}
\description{
Support function that verifies that an dbObjectId holding a reference
to a foreign object is still valid for communicating with the RDBMS
}
\usage{
isIdCurrent(obj)
}
\arguments{
\item{obj}{
any \code{dbObjectId} (e.g., \code{dbDriver},
\code{dbConnection}, \code{dbResult}).
}
}
\value{
a logical scalar.
}
\details{
\code{dbObjectId} are R/S-Plus remote references to foreign (C code) objects.
This introduces differences to the object's semantics such as
persistence (e.g., connections may be closed unexpectedly),
thus this function provides a minimal verification to ensure 
that the foreign object being referenced can be contacted.
}
\seealso{
\code{\link[DBI]{dbDriver}}
\code{\link[DBI]{dbConnect}}
\code{\link[DBI]{dbSendQuery}}
\code{\link[DBI]{dbGetQuery}}
\code{\link[DBI]{fetch}}
}
\examples{\dontrun{
cursor <- dbSendQuery(con, sql.statement)
isIdCurrent(cursor)
}
}
\keyword{interface}
\keyword{database}
% vim:syntax=tex
\eof
% $Id: make.db.names-methods.Rd,v 1.1 2003/10/17 17:27:33 dj Exp dj $
\name{make.db.names-methods}
\docType{methods}
\alias{SQLKeywords-methods}
\alias{isSQLKeyword-methods}
\alias{make.db.names,SQLiteObject,character-method}
\alias{SQLKeywords,SQLiteObject-method}
\alias{SQLKeywords,missing-method}
\alias{isSQLKeyword,SQLiteObject,character-method}
\title{
  Make R/S-Plus identifiers into legal SQL identifiers
}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{
\item{dbObj}{
  any SQLite object (e.g., \code{SQLiteDriver}).
  }
\item{snames}{
  a character vector of R/S-Plus identifiers (symbols) from which
  we need to make SQL identifiers.
  }
\item{name}{
  a character vector of SQL identifiers we want to check against
  keywords from the DBMS.
}
\item{unique}{
  logical describing whether the resulting set of SQL names should
  be unique.  Its default is \code{TRUE}.  Following the SQL 92
  standard, uniqueness of SQL identifiers is determined regardless
  of whether letters are upper or lower case.
  }
\item{allow.keywords }{
  logical describing whether SQL keywords should be allowed in the
  resulting set of SQL names.  Its default is \code{TRUE}
  }
\item{keywords}{
  a character vector with SQL keywords, namely 
  \code{.SQL92Keywords} defined in the \code{DBI} package.
  }
\item{case}{
  a character string specifying whether to make the comparison
  as lower case, upper case, or any of the two.
  it defaults to \code{any}.
  }
\item{\dots}{currently not used.}
}
}
\references{
  The set of SQL keywords is stored in the character vector
  \code{.SQL92Keywords} and reflects the SQL ANSI/ISO standard as
  documented in "X/Open SQL and RDA", 1994, ISBN 1-872630-68-8.
  Users can easily override or update this vector.

  SQLite does not add keywords to the SQL 92 standard.

  See the Database Interface definition document
  \code{DBI.pdf} in the base directory of this package
  or \url{http://developer.r-project.org/db}.
}
\seealso{
  \code{\link{SQLite}},
  \code{\link[DBI]{dbReadTable}},
  \code{\link[DBI]{dbWriteTable}},
  \code{\link[DBI]{dbExistsTable}},
  \code{\link[DBI]{dbRemoveTable}},
  \code{\link[DBI]{dbListTables}}.
}
\examples{\dontrun{
# This example shows how we could export a bunch of data.frames
# into tables on a remote database.

con <- dbConnect("SQLite", dbname = "sqlite.db")

export <- c("trantime.email", "trantime.print", "round.trip.time.email")
tabs <- make.db.names(export, unique = T, allow.keywords = T)

for(i in seq(along = export) )
   dbWriteTable(con, name = tabs[i],  get(export[i]))
}
}
\keywords{methods}
\keyword{interface}
\keyword{database}
% vim: syntax=tex
\eof
\name{safe.write}
\alias{safe.write}
\title{Write a data.frame avoiding exceeding memory limits}
\description{
   This function batches calls to \code{write.table} to avoid
   exceeding memory limits for very large data.frames.
}
\usage{
safe.write(value, file, batch, ..., quote.string=FALSE)
}
\arguments{
  \item{value}{a data.frame;}
  \item{file}{a file object (connection, file name, etc).}
  \item{batch}{maximum number of rows to write at a time.}
  \item{\dots}{any other arguments are passed to \code{write.table}.}
  \item{quote.string}{logical value passed to \code{write.table}.}
}
\details{
   The function has a while loop invoking \code{\link[base]{write.table}}
   for subsets of \code{batch} rows of \code{value}.  Since this is
   a helper function for \code{\link[RMySQL]{mysqlWriteTable}} has hardcoded
   other arguments to \code{write.table}.
}
\value{
    \code{NULL}, invisibly.
}
\note{No error checking whatsoever is done.}

\seealso{\code{\link[base]{write.table}}} 

\examples{\dontrun{
   ctr.file <- file("dump.sqloader", "w")
   safe.write(big.data, file = ctr.file, batch = 25000)
}
}
\keyword{internal}
%% vim: syntax=tex
\eof
\name{sqliteSupport}                % adapted from ROracle
\alias{sqliteInitDriver}            % driver-related function
\alias{sqliteDriverInfo}
\alias{sqliteDescribeDriver}
\alias{sqliteCloseDriver}           % connection-related 
\alias{sqliteNewConnection}
\alias{sqliteConnectionInfo}
\alias{sqliteDescribeConnection}
\alias{sqliteCloseConnection}
\alias{sqliteExecStatement}         % result-related
\alias{sqliteFetch}
\alias{sqliteQuickSQL}
\alias{sqliteResultInfo}
\alias{sqliteDescribeResult}
\alias{sqliteCloseResult}
\alias{sqliteReadTable}
\alias{sqliteWriteTable}            % convenience functions
\alias{sqliteTableFields}
\alias{sqliteDataType}
\alias{.SQLitePkgName}              % constants
\alias{.SQLitePkgVersion}
\alias{.SQLitePkgRCS}
\alias{.SQLite.NA.string}
\alias{last.warning}
%\alias{.SQLiteSQLKeywords}
\alias{.conflicts.OK}
%\non_function{}
\title{Support Functions}
\description{
   These functions are the workhorses behind the RSQLite package, but
   users need not invoke these directly.
}
\usage{
   ## SQLiteDriver-related
   sqliteInitDriver(max.con=10, fetch.default.rec = 500, force.reload=FALSE)
   sqliteDriverInfo(obj, what, ...)
   sqliteDescribeDriver(obj, verbose = FALSE, ...)
   sqliteCloseDriver(drv, ...)

   ## SQLiteConnection-related
   sqliteNewConnection(drv, dbname, mode = 0)
   sqliteConnectionInfo(obj, what, ...)
   sqliteDescribeConnection(obj, verbose = FALSE, ...)
   sqliteCloseConnection(con, ...)

   ## SQLiteResult-related
   sqliteExecStatement(con, statement, limit)
   sqliteFetch(res, n=0, ...)
   sqliteQuickSQL(con, statement)
   sqliteResultInfo(obj, what, ...)
   sqliteDescribeResult(obj, verbose = FALSE, ...)
   sqliteCloseResult(res, ...)

   ## data mappings and convenience functions
   sqliteDataType(obj, ...)
   sqliteReadTable(con, name, row.names = "row.names", check.names = TRUE, ...)
   sqliteWriteTable(con, name, value, field.types, row.names = TRUE, 
      overwrite=FALSE, append=FALSE, ...)
   sqliteTableFields(con, name, ...)
}
\arguments{
\item{max.con}{
   positive integer specifying maximum number of open connections.
   The default is 10.  Note that since SQLite is embedded in R/S-Plus
   connections are simple, very efficient direct C calls.
}
\item{fetch.default.rec}{
   default number of rows to fetch (move to R/S-Plus).  This default is
   used in \code{sqliteFetch}. 
   The default is 500.
}
\item{force.reload}{
   logical indicating whether to re-initialize the driver.  This may be
   useful if you want to change the defaults (e.g., \code{fetch.default.rec}).
   Note that the driver is a singleton (subsequent inits just returned the
   previously initialized driver, thus this argument).
}
\item{obj}{
   any of the SQLite DBI objects (e.g., \code{SQLiteConnection},
   \code{SQLiteResult}).
}
\item{what}{
   character vector of metadata to extract, e.g., "version", "statement",
   "isSelect".
}
\item{verbose}{
   logical controlling how much information to display.
   Defaults to \code{FALSE}.
}
\item{drv}{
   an \code{SQLiteDriver} object as produced by \code{sqliteInit}.
}
\item{con}{
   an \code{SQLiteConnection} object as produced by \code{sqliteNewConnection}.
}
\item{res}{
   an \code{SQLiteResult} object as produced by by \code{sqliteExecStatement}.
}
\item{dbname}{
   character string with the SQLite database file name (SQLite, like
   Microsoft's Access, stores an entire database in one file).
}
\item{mode}{
   positive integer describing the mode for opening the database
   (SQLite's underlying API currently does not yet use it, but
   it is reserved for future functionality).  The default is 0.
}
%\item{max.results}{
%   positive integer indicating the maximum number of results that
%   SQLite connections will hold open.  The current default of 1 
%   is hardcoded in the C code.
%}
\item{force}{
   logical indicating whether to close a connection that has open
   result sets.  The default is \code{FALSE}.
}
\item{statement}{
   character string holding SQL statements.
}
\item{limit}{
   integer specifying maximum number or rows to fetch.  The default
   is -1 (fetch all rows).
}
\item{n}{
   number of rows to fetch from the given result set. A value of -1
   indicates to retrieve all the rows.  The default of 0 specifies
   to extract whatever the \code{fetch.default.rec} was specified
   during driver initialization \code{sqliteInit}.
}
\item{name}{
   character vector of names (table names, fields, keywords).
}
\item{value}{
   a data.frame.
}
\item{field.types}{
   a list specifying the mapping from R/S-Plus fields in the data.frame
   \code{value} to SQL data types.  The default is
   \code{sapply(value,SQLDataType)}, see \code{SQLiteSQLType}.
}
\item{row.names}{
   a logical specifying whether to prepend the \code{value} data.frame 
   row names or not.  The default is \code{TRUE}.
}
\item{check.names}{
   a logical specifying whether to convert DBMS field names into
   legal S names. Default is \code{TRUE}.
}
\item{overwrite}{
   logical indicating whether to replace the table \code{name} with
   the contents of the data.frame \code{value}.
   The defauls is \code{FALSE}.
}
\item{append}{
   logical indicating whether to append \code{value} to the existing
   table \code{name}.
}
\item{\dots}{
   placeholder for future use.
}
}
\value{
   \code{sqliteInitDriver} returns an \code{SQLiteDriver} object.

   \code{sqliteDriverInfo} returns a list of name-value metadata pairs.

   \code{sqliteDescribeDriver} returns \code{NULL} (displays the object's 
   metadata).

   \code{sqliteCloseDriver} returns a logical indicating whether the 
   operation succeeded or not.

   \code{sqliteNewConnection} returns an \code{SQLiteConnection} object.

   \code{sqliteConnectionInfo}returns a list of name-value metadata pairs.

   \code{sqliteDescribeConnection} returns \code{NULL} (displays the 
   object's metadata).

   \code{sqliteCloseConnection} returns a logical indicating whether the 
   operation succeeded or not.

   \code{sqliteExecStatement} returns an \code{SQLiteResult} object.

   \code{sqliteFetch} returns a data.frame.

   \code{sqliteQuickSQL} returns either a data.frame if the \code{statement} is
   a \code{select}-like or NULL otherwise.

   \code{sqliteDescribeResult} returns \code{NULL} (displays the object's 
   metadata).
   
   \code{sqliteCloseResult} returns a logical indicating whether the 
   operation succeeded or not.
   
   \code{sqliteReadTable}  returns a data.frame with the contents of the
   DBMS table.
   
   \code{sqliteWriteTable}  returns a logical indicating whether the 
   operation succeeded or not.
   
   \code{sqliteTableFields} returns a character vector with the 
   table \code{name} field names.
   
   \code{sqliteDataType} retuns a character string with the closest
   SQL data type.  Note that SQLite is typeless, so this is mostly for
   creating table that are compatible across RDBMS.
   
   \code{sqliteResultInfo} returns a list of name-value metadata pairs.
}
\section{Constants}{
\code{.SQLitePkgName} (currently \code{"RSQLite"}),
\code{.SQLitePkgVersion} (the R package version),
\code{.SQLitePkgRCS} (the RCS revision),
\code{.SQLitecle.NA.string} (character that SQLite uses to 
denote \code{NULL} on input),
%\code{.SQLiteSQLKeywords}    % TODO: should we add these?
\code{.conflicts.OK}.
%\non_function{}
}
\keyword{datasets}
\keyword{interface}
\keyword{database}
%\keyword{internal}
% vim:syntax=tex
\eof
\name{summary-methods}
\docType{methods}
\alias{coerce-methods}
\alias{summary-methods}
\alias{format-methods}
\alias{show-methods}
\alias{coerce,dbObjectId,integer-method}
\alias{coerce,dbObjectId,numeric-method}
\alias{coerce,dbObjectId,character-method}
\alias{coerce,SQLiteConnection,SQLiteDriver-method}
\alias{coerce,SQLiteResult,SQLiteConnection-method}
\alias{format,dbObjectId-method}
\alias{print,dbObjectId-method}
\alias{show,dbObjectId-method}
\alias{summary,SQLiteObject-method}
\alias{summary,SQLiteDriver-method}
\alias{summary,SQLiteConnection-method}
\alias{summary,SQLiteResult-method}
\title{Summarize an SQLite object}
\description{
   These methods are straight-forward implementations of the corresponding
   generic functions.
}
\section{Methods}{\describe{

\item{object = "DBIObject"}{
   Provides relevant metadata information on \code{object}, 
   for instance, the SQLite server file, the
   SQL statement associated with a result set, etc.  
   }
\item{from}{object to be coerced}
\item{to}{coercion class}
\item{x}{object to \code{format} or \code{print} or \code{show}}
}
}
\keyword{methods}
\keyword{database}
\keyword{interface}
% vim: syntax=tex
\eof
