### $Id: corStruct.q,v 1.1 2001/09/24 21:35:31 pinheiro Exp $
###
###              Classes of correlation structures
###
### Copyright 1997, 1999 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA


##*## Generics that should be implemented for any corStruct class

corFactor <-
  ## extractor for transpose inverse square root factor of corr matrix
  function(object, ...) UseMethod("corFactor")

corMatrix <-
  ## extractor for correlation matrix or the transpose inverse 
  ## square root matrix
  function(object, ...) UseMethod("corMatrix")

###*# Constructor
### There is no constructor function for this class (i.e. no function
### called corStruct) because the class is virtual.

###*# Methods for local generics

corFactor.corStruct <-
  function(object) 
{
  if (!is.null(aux <- attr(object, "factor"))) {
    return(aux)
  }
  corD <- Dim(object)
  val <- .C("corStruct_factList",
	    as.double(unlist(corMatrix(object))),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corStruct <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (corr) {
    ## Do not know how to calculate the correlation matrix
    stop(paste("Don't know how to calculate correlation matrix of",
	       class(object)[1],"object"))
  } else {
    ## transpose inverse square root
    if (data.class(covariate) == "list") {
      if (is.null(names(covariate))) {
	names(covariate) <- 1:length(covariate)
      }
      corD <- Dim(object, rep(names(covariate),
			      unlist(lapply(covariate, length))))
    } else {
      corD <- Dim(object, rep(1, length(covariate)))
    }
    val <- .C("corStruct_factList",
	      as.double(unlist(corMatrix(object, covariate))),
	      as.integer(unlist(corD)),
	      factor = double(corD[["sumLenSq"]]),
	      logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
    if (corD[["M"]] > 1) {
      val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
      val <- lapply(val, function(el) {
        nel <- round(sqrt(length(el)))
        array(el, c(nel, nel))
      })
      names(val) <- names(corD[["len"]])
      val <- as.list(val)
    } else {
      val <- array(val, c(corD[["N"]], corD[["N"]]))
    }
    attr(val, "logDet") <- lD
    val
  }
}

###*# Methods for standard generics

as.matrix.corStruct <-
  function(x) corMatrix(x)

asNatural.corStruct <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    return(object)
  }
  stop(paste("Don't know how to convert corStruct object to natural,",
             "constrained parameterization."))
}

coef.corStruct <-
  ## Accessor for constrained or unconstrained parameters of
  ## corStruct objects 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained) {
    if (is.null(isFix <- attr(object, "fixed"))) {
      stop("corStruct object must have a \"fixed\" attribute.")
    }
    if (isFix) {
      numeric(0)
    } else {
      as.vector(object)
    }
  } else {
    stop(paste("Don't know how to obtain parameters of",
	       class(object)[1], "object"))
  }
}

"coef<-.corStruct" <-
  function(object, value)
{
  ## Assignment of the unconstrained parameter of corStruct objects
  value <- as.numeric(value)
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  ## updating the factor list and logDet, by forcing a recalculation
  attr(object, "factor") <- NULL
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- NULL
  attr(object, "logDet") <- logDet(object)
  object
}

Dim.corStruct <-
  function(object, groups) 
{
  if (missing(groups)) return(attr(object, "Dim"))
  ugrp <- unique(groups)
  groups <- factor(groups, levels = ugrp)
  len <- table(groups)
  list(N = length(groups),
       M = length(len),
       maxLen = max(len),
       sumLenSq = sum(len^2),
       len = len,
       start = match(ugrp, groups) - 1)
}

formula.corStruct <-
  ## Accessor for the covariate formula
  function(object) eval(attr(object, "formula"))

"formula<-.corStruct" <-
  function(object, value)
{
  attr(object, "formula") <- value
  object
}

getCovariate.corStruct <- 
  function(object, form = formula(object), data) 
{
  if (!missing(form)) {
    form <- formula(object)
    warning("Cannot change \"form\".")
  }
  if (is.null(covar <- attr(object, "covariate"))) { # need to calculate it
    if (missing(data)) {
      stop("Need data to calculate covariate of corStruct object")
    }
    covForm <- getCovariateFormula(form)
    if (!is.null(getGroupsFormula(form))) {
      grps <- getGroups(object, data = data)
    } else {
      grps <- NULL
    }
    if (length(all.vars(covForm)) > 0) { # primary covariate present
      if (is.null(grps)) {
        covar <- getCovariate(data, covForm)
      } else {
        if (all(all.vars(covForm) == sapply(splitFormula(covForm, "+"),
                          function(el) deparse(el[[2]])))) {
          covar <- split(getCovariate(data, covForm), grps)
        } else {
          covar <- lapply(split(data, grps), getCovariate, covForm)
        }
      }
    } else {
      if (is.null(grps)) {
        covar <- 1:nrow(data)
      } else {
	covar <- lapply(split(grps, grps), function(x) 1:length(x))
      }
    }
    if (!is.null(grps)) {
      covar <- as.list(covar)
    }
  }
  covar
}

getGroups.corStruct <-
  function(object, form = formula(object), level, data, sep)
{
  if (is.null(val <- attr(object, "groups"))) { # need to calculate
    if (!missing(data)) {
      if ((grpLev <- length(getGroupsFormula(form, asList = TRUE))) > 0) {
        ## use innermost grouping level
        val <- getGroups(data, form, level = grpLev)
        factor(val, levels = unique(as.character(val)))
      } else {
        rep(1, dim(data)[1])
      }
    } else {
      NULL
    }
  } else {
    val
  }
}

getStrata.corStruct <-
  function(object) attr(object, "strataFull")

getStrataFormula.corStruct <-
  function(object) attr(object, "stratForm")

initialize.corStruct <-
  ## Initializes some attributes of corStruct objects
  function(object, data, ...)
{
  form <- formula(object)
  ## obtaining the groups information, if any
  if (!is.null(getGroupsFormula(form))) {
    attr(object, "groups") <- getGroups(object, form, data = data)
    attr(object, "Dim") <- Dim(object, attr(object, "groups"))
  } else {                              # no groups
    attr(object, "Dim") <- Dim(object, as.factor(rep(1, nrow(data))))
  }
  ## obtaining the covariate(s)
  attr(object, "covariate") <- getCovariate(object, data = data)
  object
}

"isInitialized<-.corStruct" <-
  function(object, value)
{
  if (!is.logical(value)) {
    stop("Value must be of mode logical")
  }
  if (value) {
    if (is.null(getCovariate(object))) {
      stop("Don't know how to change initialization status")
    }
  } else {
    attr(object, "covariate") <- attr(object, "groups") <-
      attr(object, "factor") <- NULL
  }
  object
}

logDet.corStruct <- 
  function(object, covariate = getCovariate(object))
{
  if (!is.null(aux <- attr(object, "logDet"))) {
    return(aux)
  }
  if (is.null(aux <- attr(object, "factor"))) {
    ## getting the transpose sqrt factor
    aux <- corMatrix(object, covariate = covariate, corr = FALSE)
  }
  if (is.null(aux1 <- attr(aux, "logDet"))) {
    ## checking for logDet attribute; if not present, get corr matrix
    aux <- corMatrix(object, covariate)
    if (data.class(aux) == "list") {    # by group
      sum(log(abs(unlist(lapply(aux, function(el) svd(el)$d)))))/2
    } else {
      sum(log(abs(svd(aux)$d)))/2
    }
  } else {
    -aux1
  }
}

logLik.corStruct <-
  function(object, data) -logDet(object)

needUpdate.corStruct <-
  function(object) FALSE

print.corStruct <-
  function(x, ...)
{
  if (length(aux <- coef(x, FALSE)) > 0) {
    cat("Correlation structure of class", class(x)[1], "representing\n")
    print(invisible(aux), ...)
  } else {
    cat("Uninitialized correlation structure of class", class(x)[1], "\n")
  }
}

print.summary.corStruct <-
  function(x, ...)
{
  class(x) <- attr(x, "oClass")
  stNam <- attr(x, "structName")
  form <- formula(x)
  if (!is.null(stNam)) {
    cat(paste("Correlation Structure: ", stNam, "\n", sep = ""))
  }
  if (!is.null(form)) {
    cat(paste(" Formula:", deparse(as.vector(form)),"\n"))
  }
  cat(" Parameter estimate(s):\n")
  print(coef(x, FALSE))
}


recalc.corStruct <-
  function(object, conLin)
{
  conLin[["Xy"]][] <-
    .C("corStruct_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(unlist(corFactor(object))))[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + logLik(object)
  conLin
}
	 
summary.corStruct <-
  function(object, structName = class(object)[1])
{
  attr(object, "structName") <- structName
  attr(object, "oClass") <- class(object)
  class(object) <- c("summary.corStruct", class(object))
  object
}

update.corStruct <-
  function(object, data)
{
  object
}

##*## Classes that substitute for (i.e. inherit from) corStruct

###*# corSymm - general, unstructured correlation 

####* Constructor

corSymm <-
  ## Constructor for the corSymm class
  function(value = numeric(0), form = ~ 1, fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "fixed") <- fixed
  class(value) <- c("corSymm", "corStruct")
  value
}

###*# Methods for local generics

corFactor.corSymm <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("symm_factList",
	    as.double(as.vector(object)),
	    as.integer(unlist(attr(object, "covariate"))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corSymm <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  if (corr) {
    val <- .C("symm_matList",
	      as.double(as.vector(object)),
	      as.integer(unlist(covariate)),
	      as.integer(attr(object, "maxCov")),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("symm_factList",
              as.double(as.vector(object)),
              as.integer(unlist(covariate)),
              as.integer(attr(object, "maxCov")),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corSymm <-
  function(object, unconstrained = TRUE)
{
  corSymmNat(object, unconstrained = unconstrained)
}

coef.corSymm <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
  mC <- attr(object, "maxCov")
  .C("symm_fullCorr", as.double(object), 
     as.integer(mC), corr = double(round(mC * (mC - 1) / 2)))[["corr"]]
}

"coef<-.corSymm" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("symm_factList",
	    as.double(as.vector(object)),
	    as.integer(unlist(getCovariate(object))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corSymm <-
  function(object, data, ...)
{
  if (!is.null(attr(object, "maxCov"))) {# initialized - nothing to do
    return(object)
  }
  object <- NextMethod()

  covar <- attr(object, "covariate")
  if (data.class(covar) != "list") {
    covar <- list(covar)
  }
  if (any(unlist(lapply(covar, duplicated)))) {
    stop(paste("Covariate must have unique values",
	       "within groups for corSymm objects"))
  }
  covar <- unlist(covar) - 1
  maxCov <- max(uCov <- unique(covar)) + 1
  if (length(uCov) != maxCov) {
    stop(paste("Unique values of the covariate  for \"corSymm\"",
	       "objects must be a sequence of consecutive integers"))
  }
  if (Dim(object)[["M"]] > 1) {
    attr(object, "covariate") <- split(covar, getGroups(object))
  } else {
    attr(object, "covariate") <- covar
  }
  attr(object, "maxCov") <- maxCov
  natPar <- as.vector(object)
  if (length(natPar) > 0) {
    ## parameters assumed in constrained form
    if (length(natPar) != round(maxCov * (maxCov - 1) / 2)) {
      stop("Initial value for corSymm parameters of wrong dimension")
    }
    if (max(abs(natPar)) >= 1) {
      stop("Initial values for corSymm must be between -1 and 1")
    }
    natMat <- diag(maxCov)/2
    natMat[lower.tri(natMat)] <- natPar
    natMat <- (t(natMat) + natMat)
    ## checking if positive-definite
    if (any(eigen(natMat)$values <= 0)) {
      stop(paste("Initial values for corSymm do not define",
                 "a positive-definite correlation structure"))
    }
    natMat <- chol(natMat)
    uncPar <- numeric(0)
    for(i in 2:maxCov) {
      aux <- acos(natMat[1:(i-1),i]/sqrt(cumsum(natMat[i:1,i]^2)[i:2]))
      uncPar <- c(uncPar, log(aux/(pi - aux)))
    }
    coef(object) <- uncPar
  } else {				# initializing the parameters
    oldAttr <- attributes(object)
    object <- double(round(maxCov * (maxCov - 1) / 2))
    attributes(object) <- oldAttr
    attr(object, "factor") <- corFactor(object)
    attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  }
  object
}

"isInitialized<-.corSymm" <-
  function(object, value)
{
  object <- NextMethod()
  if (!value) {
    attr(object, "maxCov") <- NULL
  }
  object
}

print.corSymm <-
  function(x, ...)
{
  if (length(as.vector(x)) > 0 &&
      !is.null(mC <- attr(x, "maxCov"))) {
    aux <- coef(x, FALSE)
    val <- diag(mC)
    dimnames(val) <- list(1:mC, 1:mC)
    val[lower.tri(val)] <- aux
    class(val) <- "correlation"
    cat("Correlation structure of class corSymm representing\n")
    print(val, ...)
  }
  else cat("Unitialized correlation structure of class corSymm\n")
}

print.summary.corSymm <-
  function(x, ...)
{
  if (length(as.vector(x)) > 0 &&
      !is.null(mC <- attr(x, "maxCov"))) {
    stNam <- attr(x, "structName")
    form <- formula(x)
    if (!is.null(stNam)) {
      cat("Correlation Structure: General\n")
    }
    if (!is.null(form)) {
      cat(paste(" Formula:", deparse(as.vector(form)),"\n"))
    }
    cat(" Parameter estimate(s):\n")
    aux <- coef(x, FALSE)
    val <- diag(mC)
    dimnames(val) <- list(1:mC, 1:mC)
    val[lower.tri(val)] <- aux
    class(val) <- "correlation"
    print(val, ...)
  } else cat("Unitialized correlation structure of class corSymm\n")
}

recalc.corSymm <- 
  function(object, conLin)
{
  val <-
    .C("symm_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(as.vector(object)),
       as.integer(unlist(getCovariate(object))),
       as.integer(attr(object, "maxCov")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corSymm <- 
  function(object, structName = "General correlation")
{
  attr(object, "structName") <- structName
  class(object) <- c("summary.corSymm", class(object))
  object
}

###*# corSymmNat - general correlation in natural parameterization.
###*# This is NOT an unconstrained parameterization

####* Constructor

corSymmNat <-
  ## Constructor for the corSymmNat class
  function(object, unconstrained = TRUE)
{
  if (!inherits(object, "corSymm")) {
    stop("Object must inherit from \"corSymm\"")
  }
  val <- coef(object, unconstrained = FALSE)
  if (unconstrained) {
    val <- log((val + 1)/(1-val))
  }
  attr(object, "uncons") <- unconstrained
  class(object) <- c("corSymmNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for local generics

#corFactor.corSymmNat <-
#  function(object)
#{
#  corD <- Dim(object)
#  val <- .C("corNat_factList",
#	    as.double(as.vector(object)),
#	    as.integer(unlist(attr(object, "covariate"))),
#	    as.integer(attr(object, "maxCov")),
#	    as.integer(unlist(corD)),
#            asinteger(attr(object, "uncons")),
#	    factor = double(corD[["sumLenSq"]]),
#	    logDet = double(1))[c("factor", "logDet")]
#  lD <- val[["logDet"]]
#  val <- val[["factor"]]
#  attr(val, "logDet") <- lD
#  val
#}

#corMatrix.corSymmNat <-
#  function(object, covariate = getCovariate(object), corr = TRUE)
#{
##ifndef SP5    
#  if (data.class(covariate) == "list") {
##else
#  if (inherits(covariate, "list")) {
##endif
#    if (is.null(names(covariate))) {
#      names(covariate) <- 1:length(covariate)
#    }
#    corD <- Dim(object, rep(names(covariate), 
#			    unlist(lapply(covariate, length))))
#  } else {
#    corD <- Dim(object, rep(1, length(covariate)))
#  }
#  if (corr) {
#    val <- .C("corNat_matList",
#	      as.double(as.vector(object)),
#	      as.integer(unlist(covariate)),
#	      as.integer(attr(object, "maxCov")),
#	      as.integer(unlist(corD)),
#              as.integer(attr(object, "uncons")),
#	      mat = double(corD[["sumLenSq"]]))[["mat"]]
#    lD <- NULL
#  } else {
#    val <- .C("corNat_factList",
#              as.double(as.vector(object)),
#              as.integer(unlist(covariate)),
#              as.integer(attr(object, "maxCov")),
#              as.integer(unlist(corD)),
#              as.integer(attr(object, "uncons")),
#              factor = double(corD[["sumLenSq"]]),
#              logDet = double(1))[c("factor", "logDet")]
#    lD <- val[["logDet"]]
#    val <- val[["factor"]]
#  }
#  if (corD[["M"]] > 1) {
#    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
#    val <- lapply(val, function(el) {
#      nel <- round(sqrt(length(el)))
#      array(el, c(nel, nel))
#    })
#    names(val) <- names(corD[["len"]])
#    val <- as.list(val)
#  } else {
#    val <- array(val, c(corD[["N"]], corD[["N"]]))
#  }
#  attr(val, "logDet") <- lD
#  val
#}

###*# Methods for standard generics

coef.corSymmNat <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
  mC <- attr(object, "maxCov") 
  val <- .C("corNat_fullCorr",
            as.double(object), 
            as.integer(mC),
            as.integer(attr(object, "uncons")),
            corr = double(round(mC * (mC - 1) / 2)))[["corr"]]
  names(val) <- outer(1:mC, 1:mC,
                      function(x,y) {
                        paste("cor(",y,",",x,")",sep="")
                      })[lower.tri(diag(mC))]
  val
}

"coef<-.corSymmNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("corNat_factList",
	    as.double(as.vector(object)),
	    as.integer(unlist(getCovariate(object))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
            as.integer(attr(object, "uncons")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

#initialize.corSymmNat <-
#  function(object, data, ...)
#{
#  if (!is.null(attr(object, "maxCov"))) {# initialized - nothing to do
#    return(object)
#  }
#  object <- NextMethod()

#  covar <- attr(object, "covariate")
##ifndef SP5    
#  if (data.class(covar) != "list") {
##else
#  if (!inherits(covar, "list")) {
##endif
#    covar <- list(covar)
#  }
#  if (any(unlist(lapply(covar, duplicated)))) {
#    stop(paste("Covariate must have unique values",
#	       "within groups for corSymmNat objects"))
#  }
#  covar <- unlist(covar) - 1
#  maxCov <- max(uCov <- unique(covar)) + 1
#  if (length(uCov) != maxCov) {
#    stop(paste("Unique values of the covariate  for corSymmNat",
#	       "objects must be a sequence of consecutive integers"))
#  }
#  if (Dim(object)[["M"]] > 1) {
#    attr(object, "covariate") <- split(covar, getGroups(object))
#  } else {
#    attr(object, "covariate") <- covar
#  }
#  attr(object, "maxCov") <- maxCov
#  natPar <- as.vector(object)
#  if (length(natPar) > 0) {
#    ## parameters assumed in constrained form
#    if (length(natPar) != round(maxCov * (maxCov - 1) / 2)) {
#      stop("Initial value for corSymmNat parameters of wrong dimension")
#    }
#    if (max(abs(natPar)) >= 1) {
#      stop("Initial values for corSymmNat must be between -1 and 1")
#    }
#    natMat <- diag(maxCov)/2
#    natMat[lower.tri(natMat)] <- natPar
#    natMat <- (t(natMat) + natMat)
#    ## checking if positive-definite
#    if (any(eigen(natMat)$values <= 0)) {
#      stop(paste("Initial values for corSymmNat do not define",
#                 "a positive-definite correlation structure"))
#    }
#    coef(object) <- log((natPar + 1)/(1 - natPar))
#  } else {				# initializing the parameters
#    oldAttr <- attributes(object)
#    object <- double(round(maxCov * (maxCov - 1) / 2))
#    attributes(object) <- oldAttr
#    attr(object, "factor") <- corFactor(object)
#    attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
#  }
#  object
#}

#print.corSymmNat <-
#  function(x, ...)
#{
#  if (length(as.vector(x)) > 0 &&
#      !is.null(mC <- attr(x, "maxCov"))) {
#    aux <- coef(x, FALSE)
#    val <- diag(mC)
#    dimnames(val) <- list(1:mC, 1:mC)
#    val[lower.tri(val)] <- aux
#    class(val) <- "correlation"
#    cat("Correlation structure of class corSymmNat representing\n")
#    print(val, ...)
#  }
#  else cat("Unitialized correlation structure of class corSymmNat\n")
#}

#print.summary.corSymmNat <-
#  function(x, ...)
#{
#  if (length(as.vector(x)) > 0 &&
#      !is.null(mC <- attr(x, "maxCov"))) {
#    cat("Correlation Structure: General\n")
#    cat(paste(" Formula:", deparse(as.vector(formula(x))),"\n"))
#    cat(" Parameter estimate(s):\n")
#    aux <- coef(x, FALSE)
#    val <- diag(mC)
#    dimnames(val) <- list(1:mC, 1:mC)
#    val[lower.tri(val)] <- aux
#    class(val) <- "correlation"
#    print(val, ...)
#  } else cat("Unitialized correlation structure of class corSymmNat\n")
#}

recalc.corSymmNat <- 
  function(object, conLin)
{
  val <-
    .C("corNat_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(as.vector(object)),
       as.integer(unlist(getCovariate(object))),
       as.integer(attr(object, "maxCov")),
       as.integer(attr(object, "uncons")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

#summary.corSymmNat <- 
#  function(object,
#           structName = "General correlation, with natural parametrization")
#{
#  attr(object, "structName") <- structName
#  class(object) <- "summary.corSymmNat"
#  object
#}

###*# corBand - banded correlation 

####* Constructor

corBand <-
  ## Constructor for the corBand class
  function(value = numeric(0), form = ~ 1, fixed = FALSE, ord = 2)
{
  if (ord <= 0) {
    stop("Order must be a positive integer")
  }
  if (ord == 1) {
    return(corIdent(form))
  }
  attr(value, "formula") <- form
  attr(value, "fixed") <- fixed
  attr(value, "order") <- ord
  class(value) <- c("corBand", "corStruct")
  value
}

###*# Methods for local generics

corFactor.corBand <-
  function(object)
{
  cf <- coef(object, allCoef = TRUE)
  corD <- Dim(object)
  val <- .C("symm_factList",
	    as.double(cf),
	    as.integer(unlist(attr(object, "covariate"))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corBand <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  if (corr) {
    val <- .C("symm_matList",
	      as.double(coef(object, allCoef = TRUE)),
	      as.integer(unlist(covariate)),
	      as.integer(attr(object, "maxCov")),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("symm_factList",
              as.double(coef(object, allCoef = TRUE)),
              as.integer(unlist(covariate)),
              as.integer(attr(object, "maxCov")),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corBand <-
  function(object, unconstrained = TRUE)
{
  corBandNat(object, unconstrained = unconstrained)
}

coef.corBand <-
  function(object, unconstrained = TRUE, allCoef = FALSE)
{
  mC <- attr(object, "maxCov")
  if (!allCoef) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    }
    if (unconstrained) {
      return(as.vector(object))
    }
  }
  aux <- diag(mC)
  cf <- double(mC * (mC - 1)/2)
  cf[c((row(aux) > (col(aux) - attr(object, "order")))[(row(aux) <
                            col(aux))])] <- as.vector(object)
  if (unconstrained) {
    return(cf)
  }
  val <- .C("symm_fullCorr",
            as.double(cf), 
            as.integer(mC),
            corr = double(round(mC * (mC - 1) / 2)))[["corr"]]
  names(val) <- outer(1:mC, 1:mC,
                      function(x,y) {
                        paste("cor(",y,",",x,")",sep="")
                      })[lower.tri(diag(mC))]
  if (!allCoef) {
    aux <- diag(mC)
    val <- val[(row(aux) < (col(aux) +
                            attr(object, "order")))[lower.tri(aux)]] 
  }
  val
}

"coef<-.corBand" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("symm_factList",
	    as.double(coef(object, allCoef = TRUE)),
	    as.integer(unlist(getCovariate(object))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corBand <-
  function(object, data, ...)
{
  if (!is.null(attr(object, "maxCov"))) {# initialized - nothing to do
    return(object)
  }
  object <- NextMethod()

  covar <- attr(object, "covariate")
  if (data.class(covar) != "list") {
    covar <- list(covar)
  }
  if (any(unlist(lapply(covar, duplicated)))) {
    stop(paste("Covariate must have unique values",
	       "within groups for corBand objects"))
  }
  covar <- unlist(covar) - 1
  maxCov <- max(uCov <- unique(covar)) + 1
  if (length(uCov) != maxCov) {
    stop(paste("Unique values of the covariate  for \"corBand\"",
	       "objects must be a sequence of consecutive integers"))
  }
  if (Dim(object)[["M"]] > 1) {
    attr(object, "covariate") <- split(covar, getGroups(object))
  } else {
    attr(object, "covariate") <- covar
  }
  attr(object, "maxCov") <- maxCov
  natPar <- as.vector(object)
  ord <- attr(object, "order")
  if (length(natPar) > 0) {
    ## parameters assumed in constrained form
    if (length(natPar) != round((2*maxCov*(ord - 1) - ord^2 + ord)/2)) {
      stop("Initial value for corBand parameters of wrong dimension")
    }
    if (max(abs(natPar)) >= 1) {
      stop("Initial values for corBand must be between -1 and 1")
    }
    natMat <- diag(maxCov)/2
    natMat[row(natMat) < col(natMat) + ord &
           row(natMat) > col(natMat)] <- natPar
    natMat <- (t(natMat) + natMat)
    ## checking if positive-definite
    if (any(eigen(natMat)$values <= 0)) {
      stop(paste("Initial values for corBand do not define",
                 "a positive-definite correlation structure"))
    }
    natMat <- chol(natMat)
    uncPar <- numeric(0)
    for(i in 2:maxCov) {
      aux <- acos(natMat[1:(i-1),i]/sqrt(cumsum(natMat[i:1,i]^2)[i:2]))
      uncPar <- c(uncPar, log(aux/(pi - aux)))
    }
    coef(object) <- uncPar[c((row(natMat) > col(natMat) - ord)[row(natMat)
                                                   < col(natMat)])]
  } else {				# initializing the parameters
    oldAttr <- attributes(object)
    object <- double(round((2 * maxCov * (ord - 1) - ord^2 + ord)/2))
    attributes(object) <- oldAttr
    attr(object, "factor") <- corFactor(object)
    attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  }
  object
}

"isInitialized<-.corBand" <-
  function(object, value)
{
  object <- NextMethod()
  if (!value) {
    attr(object, "maxCov") <- NULL
  }
  object
}

print.corBand <-
  function(x, ...)
{
  if (length(as.vector(x)) > 0 &&
      !is.null(mC <- attr(x, "maxCov"))) {
    aux <- coef(x, FALSE, allCoef = TRUE)
    val <- diag(mC)
    dimnames(val) <- list(1:mC, 1:mC)
    val[lower.tri(val)] <- aux
    class(val) <- "correlation"
    cat("Correlation structure of class corBand representing\n")
    print(val, ...)
  }
  else cat("Unitialized correlation structure of class corBand\n")
}

print.summary.corBand <-
  function(x, ...)
{
  if (length(as.vector(x)) > 0 &&
      !is.null(mC <- attr(x, "maxCov"))) {
    stNam <- attr(x, "structName")
    form <- formula(x)
    if (!is.null(stNam)) {
      cat(paste("Correlation Structure:", stNam, "\n"))
    }
    if (!is.null(form)) {
      cat(paste(" Formula:", deparse(as.vector(form)),"\n"))
    }
    cat(" Parameter estimate(s):\n")
    aux <- coef(x, FALSE, allCoef = TRUE)
    val <- diag(mC)
    dimnames(val) <- list(1:mC, 1:mC)
    val[lower.tri(val)] <- aux
    class(val) <- "correlation"
    print(val, ...)
  } else cat("Unitialized correlation structure of class corBand\n")
}

recalc.corBand <- 
  function(object, conLin)
{
  val <-
    .C("symm_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(coef(object, allCoef = TRUE)),
       as.integer(unlist(getCovariate(object))),
       as.integer(attr(object, "maxCov")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corBand <- 
  function(object, structName = paste("Banded correlation of order",
                     attr(object, "order")))
{
  attr(object, "structName") <- structName
  class(object) <- c("summary.corBand", class(object))
  object
}

###*# corBandNat - banded correlation in natural parameterization.
###*# This is NOT an unconstrained parameterization

####* Constructor

corBandNat <-
  ## Constructor for the corBandNat class
  function(object, unconstrained = TRUE)
{
  if (!inherits(object, "corBand")) {
    stop("Object must inherit from \"corBand\"")
  }
  val <- coef(object, unconstrained = FALSE)
  if (unconstrained) {
    val <- log((val + 1)/(1-val))
  }
  attr(object, "uncons") <- unconstrained
  class(object) <- c("corBandNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corBandNat <-
  function(object, unconstrained = TRUE, allCoef = FALSE)
{
  mC <- attr(object, "maxCov")
  aux <- diag(mC)
  cf <- double(mC * (mC - 1)/2)
  wchEl <-
    c((row(aux) > (col(aux) - attr(object, "order")))[(row(aux) < col(aux))])
  cf[wchEl] <- as.vector(object)
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      if (allCoef) {
        return(cf)
      } else {
        return(as.vector(object))
      }
    }
  }
  val <- .C("corNat_fullCorr",
            as.double(cf), 
            as.integer(mC),
            as.integer(attr(object, "uncons")),
            corr = double(round(mC * (mC - 1) / 2)))[["corr"]]
  names(val) <- outer(1:mC, 1:mC,
                      function(x,y) {
                        paste("cor(",y,",",x,")",sep="")
                      })[row(aux) < col(aux)]
  if (!(allCoef)) {
    val <- val[wchEl]
  }
  val
}

"coef<-.corBandNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("corNat_factList",
	    as.double(coef(object, allCoef = TRUE)),
	    as.integer(unlist(getCovariate(object))),
	    as.integer(attr(object, "maxCov")),
	    as.integer(unlist(corD)),
            as.integer(attr(object, "uncons")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corBandNat <- 
  function(object, conLin)
{
  val <-
    .C("corNat_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(coef(object, allCoef = TRUE)),
       as.integer(unlist(getCovariate(object))),
       as.integer(attr(object, "maxCov")),
       as.integer(attr(object, "uncons")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

###*# corIdent - independent structure

####* Constructor

corIdent <-
  ## Constructor for the corIdent class
  function(form = NULL)
{
  value <- numeric(0)
  attr(value, "formula") <- form
  attr(value, "fixed") <- TRUE
  class(value) <- c("corIdent", "corStruct")
  value
}

###*# Methods for local generics

corMatrix.corIdent <-
  function(object, covariate = getCovariate(object), corr) 
{
  if (data.class(covariate) == "list") {# by group
    as.list(lapply(covariate, function(el, object) corMatrix(object, el)))
  } else {
    diag(length(covariate))
  }
}
  
###*# Methods for standard generics

asNatural.corIdent <-
  function(object, unconstrained) object
          
coef.corIdent <-
  function(object, unconstrained = TRUE) numeric(0)

"coef<-.corIdent" <- 
  function(object, value) object

initialize.corIdent <- 
  function(object, data, ...)
{
  attr(object, "logDet") <- 0
  object
}

logDet.corIdent <-
  function(object, covariate) 0

recalc.corIdent <- 
  function(object, conLin)
{
  conLin
}

summary.corIdent <-
  function(object, structName = "Independent")
{
  summary.corStruct(object, structName)
}

###*# corAR1 - autoregressive of order one structure

####* Constructor

corAR1 <-
  ## Constructor for the corAR1 class
  function(value = 0, form = ~ 1, fixed = FALSE)
{
  if (abs(value) >= 1) {
    stop("Parameter in AR(1) structure must be between -1 and 1")
  }
  value <- log((1 + value)/( 1 - value))
  attr(value, "formula") <- form
  attr(value, "fixed") <- fixed
  class(value) <- c("corAR1", "corStruct")
  value
}

###*# Methods for local generics

corFactor.corAR1 <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("AR1_factList",
	    as.double(as.vector(object)),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corAR1 <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  if (corr) {
    val <- .C("AR1_matList",
	      as.double(as.vector(object)),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("AR1_factList",
              as.double(as.vector(object)),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corAR1 <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corAR1Nat(object)
  }
}

coef.corAR1 <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
  aux <- exp(as.vector(object))
  aux <- c((aux - 1)/(aux + 1))
  names(aux) <- "Phi"
  aux
}

"coef<-.corAR1" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("AR1_factList",
	    as.double(as.vector(object)),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corAR1 <-
  ## Initializes corAR1 objects
  function(object, data, ...)
{
  object <- NextMethod()
  covar <- attr(object, "covariate")
  if (data.class(covar) != "list") {
    covar <- list(covar)
  }
  if (any(unlist(lapply(covar, duplicated)))) {
    stop(paste("Covariate must have unique values",
	       "within groups for corAR1 objects"))
  }
  if (any(unlist(lapply(covar, diff)) != 1)) {
    ## Cannot use formulas for inverse of square root matrix
    ## will convert to class ARMA(1,0)
    attr(object, "p") <- 1
    attr(object, "q") <- 0
    class(object) <- c("corARMA", "corStruct")
    initialize(object, data)
  } else {
    ## obtaining the factor list and logDet
    attr(object, "factor") <- corFactor(object)
    attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
    object
  }
}

recalc.corAR1 <- 
  function(object, conLin)
{
  val <-
    .C("AR1_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(as.vector(object)),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corAR1 <- 
  function(object, structName = "AR(1)")
{
  summary.corStruct(object, structName)
}

###*# corAR1Nat - autoregressive of order one structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corAR1Nat <-
  ## Constructor for the corAR1Nat class
  function(object)
{
  if (!inherits(object, "corAR1")) {
    stop("Object must inherit from \"corAR1\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corAR1Nat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corAR1Nat <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained && attr(object, "fixed")) {
    return(numeric(0))
  }
  val <- as.vector(object)
  names(val) <- "Phi"
  val
}

"coef<-.corAR1Nat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  val <- log((value+1)/(1-value))
  aux <- .C("AR1_factList",
	    as.double(val),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corAR1Nat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  aux <- log((aux+1)/(1-aux))
  val <-
    .C("AR1_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

####*# corCAR1 - continuous time autoregressive of order one structure

#####* Constructor

corCAR1 <-
  ## Constructor for the corCAR1 class
  function(value = 0.2, form = ~ 1, fixed = FALSE)
{
  if (value <= 0 | value >= 1) {
    stop("Parameter in CAR(1) structure must be between 0 and 1")
  }
  value <- log(value / (1 - value))
  attr(value, "formula") <- form
  attr(value, "fixed") <- fixed
  class(value) <- c("corCAR1", "corStruct")
  value
}


###*# Methods for local generics

corFactor.corCAR1 <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("CAR1_factList",
	    as.double(as.vector(object)),
	    as.double(unlist(attr(object, "covariate"))),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corCAR1 <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  if (corr) {
    val <- .C("CAR1_matList",
	      as.double(as.vector(object)),
	      as.double(unlist(covariate)),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("CAR1_factList",
              as.double(as.vector(object)),
              as.double(unlist(covariate)),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corCAR1 <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corCAR1Nat(object)
  }
}

coef.corCAR1 <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
  aux <- c(exp(as.vector(object)))
  aux <- aux/(1+aux)  
  names(aux) <- "Phi"
  aux
}

"coef<-.corCAR1" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("CAR1_factList",
	    as.double(as.vector(object)),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corCAR1 <-
  ## Initializes corCAR1 objects
  function(object, data, ...)
{
  object <- NextMethod()
  covar <- attr(object, "covariate")
  if (data.class(covar) != "list") {
    covar <- list(covar)
  }

  if (any(unlist(lapply(covar, duplicated)))) {
    stop(paste("Covariate must have unique values",
	       "within groups for corCAR1 objects"))
  }
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  object
}

recalc.corCAR1 <- 
  function(object, conLin)
{
  val <- 
    .C("CAR1_recalc", 
     Xy = as.double(conLin[["Xy"]]),
     as.integer(unlist(Dim(object))),
     as.integer(ncol(conLin[["Xy"]])),
     as.double(as.vector(object)),
     as.double(unlist(getCovariate(object))),
     logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corCAR1 <- 
  function(object, structName = "Continuous AR(1)")
{
  summary.corStruct(object, structName)
}

###*# corCAR1Nat - continuous autoregressive of order one structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corCAR1Nat <-
  ## Constructor for the corCAR1Nat class
  function(object)
{
  if (!inherits(object, "corCAR1")) {
    stop("Object must inherit from \"corCAR1\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corCAR1Nat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corCAR1Nat <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained && attr(object, "fixed")) {
    return(numeric(0))
  }
  val <- as.vector(object)
  names(val) <- "Phi"
  val
}

"coef<-.corCAR1Nat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  val <- log(value/(1-value))
  aux <- .C("CAR1_factList",
	    as.double(val),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corCAR1Nat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  aux <- log(aux/(1-aux))
  val <-
    .C("CAR1_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

###*# corARMA - autoregressive-moving average structures

####* Constructor

corARMA <-
  ## Constructor for the corARMA class
  function(value = double(p + q), form = ~ 1, p = 0, q = 0, fixed = FALSE)
{
  if (!(p >= 0 && (p == round(p)))) {
    stop("Autoregressive order must be a non-negative integer")
  }
  if (!(q >= 0 && (q == round(q)))) {
    stop("Moving average order must be a non-negative integer")
  }
  if (0 == (p + q)) {
    return(corIdent())
  }
  if (length(value) != p + q) {
    stop("Initial value for parameter of wrong length")
  }
  if (max(abs(value)) >= 1) {
    stop("Parameters in ARMA structure must be < 1 in absolute value")
  }
  ## unconstrained parameters
  value <- .C("ARMA_unconstCoef", 
	      as.integer(p), 
	      as.integer(q), 
	      pars = as.double(value))$pars
  attributes(value) <- list(formula = form, p = p, q = q, fixed = fixed)
  class(value) <- c("corARMA", "corStruct")
  value
}


###*# Methods for local generics

corFactor.corARMA <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("ARMA_factList",	
	    as.double(as.vector(object)),
	    as.integer(attr(object, "p")),
	    as.integer(attr(object, "q")),
	    as.integer(unlist(attr(object, "covariate"))),
	    as.integer(attr(object, "maxLag")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

    
corMatrix.corARMA <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  p <- attr(object, "p")
  q <- attr(object, "q")
  maxLag <- attr(object, "maxLag")
  if (corr) {
    val <- .C("ARMA_matList",
	      as.double(as.vector(object)),
	      as.integer(p),
	      as.integer(q),
	      as.integer(unlist(covariate)),
	      as.integer(maxLag),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("ARMA_factList",	
              as.double(as.vector(object)),
              as.integer(attr(object, "p")),
              as.integer(attr(object, "q")),
              as.integer(unlist(covariate)),
              as.integer(attr(object, "maxLag")),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corARMA <-
  function(object, unconstrained = TRUE)
{
  corARMANat(object)
}
  
coef.corARMA <- 
  function(object, unconstrained = TRUE) 
{
  if (attr(object, "fixed") && unconstrained) {
    return(numeric(0))
  } 
  val <-  as.vector(object)
  if (!unconstrained) {
    p <- attr(object, "p")
    q <- attr(object, "q")
    nams <- NULL
    if (p > 0) {
      nams <- paste(rep("Phi", p), 1:p, sep="")
    }
    if (q > 0) {
      nams <- c(nams, paste(rep("Theta", q), 1:q, sep=""))
    }
    val <- c(.C("ARMA_constCoef", as.integer(attr(object,"p")), 
		as.integer(attr(object,"q")),
		pars = as.double(val))$pars)
    names(val) <- nams
  }
  val
}

"coef<-.corARMA" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  p <- attr(object, "p")
  q <- attr(object, "q")
  object[] <- value
  ## updating the factor list and logDet
  corD <- Dim(object)
  aux <- .C("ARMA_factList",
	    as.double(as.vector(object)),
	    as.integer(p),
	    as.integer(q),
	    as.integer(unlist(getCovariate(object))),
	    as.integer(attr(object, "maxLag")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corARMA <-
  function(object, data, ...)
{
  ## Initializes corARMA objects
  object <- NextMethod()
  covar <- attr(object, "covariate")
  if (data.class(covar) != "list") {
    covar <- list(covar)
  }
  if (any(unlist(lapply(covar, duplicated)))) {
    stop(paste("Covariate must have unique values",
	       "within groups for corARMA objects"))
  }
  if ((attr(object, "p") == 1) && (attr(object, "q") == 0) &&
     all(unlist(lapply(covar, diff)) == 1)) {
    ## Use AR1 methods instead
    class(object) <- c("corAR1", "corStruct")
    initialize(object, data)
  } else {
    attr(object, "maxLag") <- 
      max(unlist(lapply(covar, function(el) max(abs(outer(el,el,"-"))))))
    attr(object, "factor") <- corFactor(object)
    attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
    object
  }
}

recalc.corARMA <- 
  function(object, conLin)
{
  val <- 
    .C("ARMA_recalc", 
     Xy = as.double(conLin[["Xy"]]),
     as.integer(unlist(Dim(object))),
     as.integer(ncol(conLin[["Xy"]])),
     as.double(as.vector(object)),
     as.integer(attr(object, "p")),
     as.integer(attr(object, "q")),
     as.integer(unlist(getCovariate(object))),
     as.integer(attr(object, "maxLag")),
     logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corARMA <- 
  function(object, structName = paste("ARMA(",attr(object,"p"),",",
		     attr(object,"q"), ")", sep = ""))
{
  summary.corStruct(object, structName)
}

###*# corARMANat - ARMA structure in natural parameterization.
###*# This is NOT an unconstrained parameterization

####* Constructor

corARMANat <-
  ## Constructor for the corCAR1Nat class
  function(object)
{
  if (!inherits(object, "corARMA")) {
    stop("Object must inherit from \"corARMA\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corARMANat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corARMANat <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
             
  p <- attr(object, "p")
  q <- attr(object, "q")
  val <- as.vector(object)
  nams <- NULL
  if (p > 0) {
    nams <- paste(rep("Phi", p), 1:p, sep="")
  }
  if (q > 0) {
    nams <- c(nams, paste(rep("Theta", q), 1:q, sep=""))
  }
  names(val) <- nams
  val
}

"coef<-.corARMANat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  noUp <- attr(object, "noUpdate")
  if (!is.null(noUp) && noUp) return(object)
  p <- attr(object, "p")
  q <- attr(object, "q")
  corD <- attr(object, "Dim")
  ## in unconstrained form, original parameterization
  val <- .C("ARMA_unconstCoef", 
            as.integer(p), 
            as.integer(q), 
            pars = as.double(value))$pars
  ## updating the factor list and logDet
  aux <- .C("ARMA_factList",
	    as.double(val),
            as.integer(p),
            as.integer(q),
            as.integer(unlist(getCovariate(object))),
            as.integer(attr(object, "maxLag")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corARMANat <- 
  function(object, conLin)
{
  val <- as.vector(object)
  p <- attr(object, "p")
  q <- attr(object, "q")
  val <- .C("ARMA_unconstCoef", 
            as.integer(p), 
            as.integer(q), 
            pars = as.double(val))$pars
  val <-
    .C("ARMA_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(val),
       as.integer(p),
       as.integer(q),
       as.integer(unlist(getCovariate(object))),
       as.integer(attr(object, "maxLag")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

###*# corCompSymm - Compound symmetry structure structure

####* Constructor

corCompSymm <-
  ## Constructor for the corCompSymm class
  function(value = 0, form = ~ 1, fixed = FALSE)
{
  if (abs(value) >= 1) {
    stop(paste("Parameter in \"corCompSymm\" structure",
	       "must be < 1 in absolute value"))
  }
  attr(value, "formula") <- form
  attr(value, "fixed") <- fixed
  class(value) <- c("corCompSymm", "corStruct")
  value
}

###*# Methods for local generics

corFactor.compSymm <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("compSymm_factList",
	    as.double(as.vector(object)),
	    as.double(attr(object, "inf")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corCompSymm <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, length))))
  } else {
    corD <- Dim(object, rep(1, length(covariate)))
  }
  if (corr) {
    val <- .C("compSymm_matList",
	      as.double(as.vector(object)),
	      as.double(attr(object, "inf")),
	      as.integer(unlist(corD)),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("compSymm_factList",
              as.double(as.vector(object)),
              as.double(attr(object, "inf")),
              as.integer(unlist(corD)),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for local generics

asNatural.corCompSymm <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corCompSymmNat(object)
  }
}

coef.corCompSymm <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained) {
    if (attr(object, "fixed")) {
      return(numeric(0))
    } else {
      return(as.vector(object))
    }
  }
  val <- exp(as.vector(object))
  val <- c((val + attr(object, "inf"))/(val + 1))
  names(val) <- "Rho"
  val
}

"coef<-.corCompSymm" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("compSymm_factList",
	    as.double(as.vector(object)),
	    as.double(attr(object, "inf")),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

initialize.corCompSymm <-
  ## Initializes corCompSymm objects
  function(object, data, ...)
{
  if (!is.null(attr(object, "inf"))) {   # initialized - nothing to do
    return(object)
  }
  object <- NextMethod()
  natPar <- as.vector(object)
  corD <- Dim(object)
  if (natPar <= (attr(object, "inf") <- -1/(corD[["maxLen"]] - 1))) {
    stop(paste("Initial value in corCompSymm must be > than",
               attr(object, "inf")))
  }
  object[] <- log((natPar - attr(object, "inf"))/(1 - natPar))	
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  object
}

"isInitialized<-.corCompSymm" <-
  function(object, value)
{
  object <- NextMethod()
  if (!value) {
    attr(object, "inf") <- NULL
  }
  object
}

recalc.corCompSymm <- 
  function(object, conLin)
{
  val <- 
    .C("compSymm_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(as.vector(object)),
       as.double(attr(object, "inf")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

summary.corCompSymm <- 
  function(object, structName = "Compound symmetry")
{
  summary.corStruct(object, structName)
}

###*# corCompSymmNat - compound symmetry structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corCompSymmNat <-
  ## Constructor for the corCompSymmNat class
  function(object)
{
  if (!inherits(object, "corCompSymm")) {
    stop("Object must inherit from \"corCompSymm\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corCompSymmNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corCompSymmNat <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained && attr(object, "fixed")) {
    return(numeric(0))
  }
  val <- as.vector(object)
  names(val) <- "Rho"
  val
}

"coef<-.corCompSymmNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  inf <- attr(object, "inf")
  val <- log((value - inf)/(1 - value))
  ## updating the factor list and logDet
  aux <- .C("compSymm_factList",
	    as.double(val),
            as.double(inf),
	    as.integer(unlist(corD)),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corCompSymmNat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  inf <- attr(object, "inf")
  aux <- log((aux - inf)/(1 - aux))
  val <-
    .C("compSymm_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       as.double(inf),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

####*# corHF - Huyn-Feldt structure

#corHF <-
#  ## Constructor for the corHuynFeldt class
#  function(value = numeric(0), form = ~ 1)
#{
#  attr(value, "formula") <- form
#  class(value) <- c("corHF", "corStruct")
#  value
#}

####*# Methods for local generics

#corFactor.corHF <-
#  function(object)
#{
#  corD <- Dim(object)
#  val <- .C("HF_factList",
#	    as.double(as.vector(object)),
#	    as.integer(attr(object, "maxCov")),
#	    as.integer(unlist(getCovariate(object))),
#	    as.integer(unlist(corD)),
#	    factor = double(corD[["sumLenSq"]]),
#	    logDet = double(1))[c("factor", "logDet")]
#  lD <- val[["logDet"]]
#  val <- val[["factor"]]
#  attr(val, "logDet") <- lD
#  val
#}

#corMatrix.corHF <-
#  function(object, covariate = getCovariate(object), corr = TRUE)
#{
#  if (data.class(covariate) == "list") {
#    if (is.null(names(covariate))) {
#      names(covariate) <- 1:length(covariate)
#    }
#    corD <- Dim(object, rep(names(covariate), 
#			    unlist(lapply(covariate, length))))
#  } else {
#    corD <- Dim(object, rep(1, length(covariate)))
#  }
#  if (corr) {
#    val <- .C("HF_matList",
#	      as.double(as.vector(object)),
#	      as.integer(attr(object, "maxCov")),
#	      as.integer(unlist(covariate)),
#	      as.integer(unlist(corD)),
#	      mat = double(corD[["sumLenSq"]]))[["mat"]]
#    lD <- NULL
#  } else {
#    val <- .C("HF_factList",
#              as.double(as.vector(object)),
#              as.integer(attr(object, "maxCov")),
#              as.integer(unlist(covariate)),
#              as.integer(unlist(corD)),
#              factor = double(corD[["sumLenSq"]]),
#              logDet = double(1))[c("factor", "logDet")]
#    lD <- val[["logDet"]]
#    val <- val[["factor"]]
#  }
#  if (corD[["M"]] > 1) {
#    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
#    val <- lapply(val, function(el) {
#      nel <- round(sqrt(length(el)))
#      array(el, c(nel, nel))
#    })
#    names(val) <- names(corD[["len"]])
#  } else {
#    val <- array(val, c(corD[["N"]], corD[["N"]]))
#  }
#  attr(val, "logDet") <- lD
#  val
#}

####*# Methods for standard generics

#coef.corHF <-
#  function(object, unconstrained = TRUE)
#{
#  aux <- as.vector(object)
#  if (!unconstrained) {
#    aux <- 2 * (exp(aux) + attr(object, "inf")) + 1
#  }
#  aux
#}

#"coef<-.corHF" <-
#  function(object, value) 
#{
#  if (length(value) != length(object)) {
#    stop("Cannot change the length of the parameter of a corStruct object")
#  }
#  object[] <- value
#  corD <- attr(object, "Dim")
#  ## updating the factor list and logDet
#  aux <- .C("HF_factList",
#	    as.double(as.vector(object)),
#	    as.integer(attr(object, "maxCov")),
#	    as.integer(unlist(getCovariate(object))),
#	    as.integer(unlist(corD)),
#	    factor = double(corD[["sumLenSq"]]),
#	    logDet = double(1))[c("factor", "logDet")]
#  attr(object, "factor") <- aux[["factor"]]
#  attr(object, "logDet") <- -aux[["logDet"]]
#  object
#}

#initialize.corHF <-
#  function(object, data, ...)
#{
#  if (!is.null(attr(object, "inf"))) {   # initialized - nothing to do
#    return(object)
#  }
#  object <- NextMethod()
#  covar <- attr(object, "covariate")
#  if (data.class(covar) == "list") {
#    attr(object, "covariate") <- covar <- 
#      lapply(covar, function(el) el - 1)
#  } else {
#    attr(object, "covariate") <- covar <- covar - 1
#    covar <- list(covar)
#  }
#  if (any(unlist(lapply(covar, duplicated)))) {
#    stop(paste("Covariate must have unique values",
#               "within groups for corHF objects"))
#  }
#  maxCov <- max(uCov <- unique(unlist(covar))) + 1
#  if (length(uCov) != maxCov) {
#    stop(paste("Unique values of the covariate  for \"corHF\"",
#               "objects must be a sequence of consecutive integers"))
#  }
#  attr(object, "maxCov") <- maxCov
#  attr(object, "inf") <- -1/(2*maxCov)
#  natPar <- as.vector(object)
#  if (length(natPar) > 0) {
#    if (length(aux) != attr(object, "maxCov"))
#      stop("Initial value for Huyn-Feldt parameters of wrong dimension")
#    ## verifying if initial values satisfy constraints
#    if (any(natPar <= attr(object, "inf"))) {
#      stop(paste("Initial values for \"corHF\" parameters",
#		 "must be > than", attr(object, "inf")))
#    }
#    object[] <- log(natPar - attr(object, "inf"))
#  } else {				# initializing the parameters
#    oldAttr <- attributes(object)
#    object <- log(rep(-attr(object, "inf"), att(object, "maxCov")))
#    attributes(object) <- oldAttr
#  }
#  attr(object, "factor") <- corFactor(object)
#  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
#  object
#}

#print.corHF <-
#  function(x, ...)
#{
#  if (length(as.vector(x)) > 0 && !is.null(attr(object, "maxCov")))
#    NextMethod()
#  else cat("Unitialized correlation structure of class corHF\n")
#}

#recalc.corHF <- 
#  function(object, conLin)
#{
#  val <-
#    .C("HF_recalc", 
#       Xy = as.double(conLin[["Xy"]]),
#       as.integer(unlist(Dim(object))),
#       as.integer(ncol(conLin[["Xy"]])),
#       as.double(as.vector(object)),
#       as.integer(unlist(getCovariate(object))),
#       as.integer(attr(object, "maxCov")),
#       logLik = double(1))[c("Xy", "logLik")]
#  conLin[["Xy"]][] <- val[["Xy"]]
#  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
#  conLin
#}

#summary.corHF <- 
#  function(object, structName = "Huyn-Feldt")
#{
#  summary.corStruct(object, structName)
#}

###*# corSpatial - a virtual class of spatial correlation structures

###*# Constructor

corSpatial <-
  ## Constructor for the corSpatial class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   type = c("spherical", "exponential", "gaussian", "linear",
             "rational"),
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  type <- match.arg(type)
  spClass <- switch(type,
		    spherical = "corSpher",
		    exponential = "corExp",
		    gaussian = "corGaus",
		    linear = "corLin",
                    rational = "corRatio")
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c(spClass, "corSpatial", "corStruct")
  value
}

###*# Methods for local generics

corFactor.corSpatial <-
  function(object)
{
  corD <- Dim(object)
  val <- .C("spatial_factList",
	    as.double(as.vector(object)),
	    as.integer(attr(object, "nugget")),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    as.double(attr(object, "minD")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  lD <- val[["logDet"]]
  val <- val[["factor"]]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corSpatial <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) == "list") {
    if (is.null(names(covariate))) {
      names(covariate) <- 1:length(covariate)
    }
    corD <- Dim(object, rep(names(covariate), 
			    unlist(lapply(covariate, 
		  function(el) round((1 + sqrt(1 + 8 * length(el)))/2)))))
  } else {
    corD <- Dim(object, rep(1, round((1 + sqrt(1 + 8* length(covariate)))/2)))
  }
  if (corr) {
    val <- .C("spatial_matList",
	      as.double(as.vector(object)),
	      as.integer(attr(object, "nugget")),
	      as.double(unlist(covariate)),
	      as.integer(unlist(corD)),
	      as.double(attr(object, "minD")),
	      mat = double(corD[["sumLenSq"]]))[["mat"]]
    lD <- NULL
  } else {
    val <- .C("spatial_factList",
              as.double(as.vector(object)),
              as.integer(attr(object, "nugget")),
              as.double(unlist(getCovariate(object))),
              as.integer(unlist(corD)),
              as.double(attr(object, "minD")),
              factor = double(corD[["sumLenSq"]]),
              logDet = double(1))[c("factor", "logDet")]
    lD <- val[["logDet"]]
    val <- val[["factor"]]
  }
  if (corD[["M"]] > 1) {
    val <- split(val, rep(1:corD[["M"]], (corD[["len"]])^2))
    val <- lapply(val, function(el) {
      nel <- round(sqrt(length(el)))
      array(el, c(nel, nel))
    })
    names(val) <- names(corD[["len"]])
    val <- as.list(val)
  } else {
    val <- array(val, c(corD[["N"]], corD[["N"]]))
  }
  attr(val, "logDet") <- lD
  val
}

###*# Methods for standard generics

asNatural.corSpatial <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corSpatialNat(object)
  }
}
  
coef.corSpatial <-
  function(object, unconstrained = TRUE)
{
  if (attr(object, "fixed") && unconstrained) {
    return(numeric(0))
  }
  val <- as.vector(object)
  if (length(val) == 0) {               # uninitialized
    return(val)
  }
  if (!unconstrained) {
    val <- exp(val)
    if (attr(object, "nugget")) val[2] <- val[2]/(1+val[2])
  }
  if (attr(object, "nugget")) names(val) <- c("range", "nugget")
  else names(val) <- "range"
  val
}

"coef<-.corSpatial" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter after initialization")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  ## updating the factor list and logDet
  aux <- .C("spatial_factList",
	    as.double(as.vector(object)),
	    as.integer(attr(object, "nugget")),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    as.double(attr(object, "minD")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

Dim.corSpatial <-
  function(object, groups)
{
  if (missing(groups)) return(attr(object, "Dim"))
  val <- Dim.corStruct(object, groups)
  val[["start"]] <- 
    c(0, cumsum(val[["len"]] * (val[["len"]] - 1)/2)[-val[["M"]]])
  ## will use third component of Dim list for spClass
  names(val)[3] <- "spClass"
  val[[3]] <- 
    match(class(object)[1], c("corSpher", "corExp", "corGaus", "corLin",
                              "corRatio"), 0)
  val
}

getCovariate.corSpatial <- 
  function(object, form = formula(object), data) 
{
  if (is.null(covar <- attr(object, "covariate"))) { # need to calculate it
    if (missing(data)) {
      stop("Need data to calculate covariate")
    }
    covForm <- getCovariateFormula(form)
    if (length(all.vars(covForm)) > 0) { # covariate present
      if (attr(terms(covForm), "intercept") == 1) {
	covForm <-
          eval(parse(text = paste("~", deparse(covForm[[2]]),"-1",sep="")))
      }
      covar <- as.data.frame(unclass(model.matrix(covForm, 
					      model.frame(covForm, data))))
    } else {
      covar <- NULL
    }
      
    if (!is.null(getGroupsFormula(form))) { # by groups
      grps <- getGroups(object, data = data)
      if (is.null(covar)) {
	covar <- lapply(split(grps, grps),
                        function(x) as.vector(dist(1:length(x))))
      } else {
	covar <- lapply(split(covar, grps), 
			function(el, metric) {
                          el <- as.matrix(el)
                          if (nrow(el) > 1) {
                            as.vector(dist(el, metric))
                          } else {
                            numeric(0)
                          }
			}, metric = attr(object, "metric"))
      }
      covar <- covar[sapply(covar, length) > 0]  # no 1-obs groups
    } else {				# no groups
      if (is.null(covar)) {
	covar <- as.vector(dist(1:nrow(data)))
      } else {
	covar <- as.vector(dist(as.matrix(covar),
                                metric = attr(object, "metric")))
      }
    }
    if (any(unlist(covar) == 0)) {
      stop("Cannot have zero distances in \"corSpatial\"")
    }
  }
  covar
}

initialize.corSpatial <-
  function(object, data, ...)
{
  if (!is.null(attr(object, "minD"))) { #already initialized
    return(object)
  }
  object <- initialize.corStruct(object, data)
  nug <- attr(object, "nugget")

  val <- as.vector(object)
  if (length(val) > 0) {		# initialized
    if (val[1] <= 0) {
      stop("Range must be > 0 in \"corSpatial\" initial value")
    }
    if (nug) {				# with nugget effect
      if (length(val) == 1) {		# assuming nugget effect not given
	val <- c(val, 0.1)		# setting it to 0.1
      } else {
	if (length(val) != 2) {
	  stop("Initial value for corSpatial parameters of wrong dimension")
	}
      }
      if ((val[2] <= 0) || (val[2] >= 1)) {
	stop("Initial value of nugget ratio must be between 0 and 1")
      }
    } else {				# only range parameter
      if (length(val) != 1) {
	stop("Initial value for corSpatial parameters of wrong dimension")
      }
    }
  } else {
    val <- min(unlist(attr(object, "covariate"))) * 0.9
    if (nug) val <- c(val, 0.1)
  }
  val[1] <- log(val[1])
  if (nug) val[2] <- log(val[2]/(1 - val[2]))
  oldAttr <- attributes(object)
  object <- val
  attributes(object) <- oldAttr
  attr(object, "minD") <- min(unlist(attr(object, "covariate")))
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  object
}

"isInitialized<-.corSpatial" <-
  function(object, value)
{
  object <- NextMethod()
  if (!value) {
    attr(object, "minD") <- NULL
  }
  object
}

recalc.corSpatial <- 
  function(object, conLin)
{
  val <-
    .C("spatial_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(as.vector(object)),
       as.double(unlist(getCovariate(object))),
       as.double(attr(object, "minD")),
       as.integer(attr(object, "nugget")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

Variogram.corSpatial <-
  function(object, distance = NULL, sig2 = 1, length.out = 50, FUN)
{
  if (is.null(distance)) {
    rangeDist <- range(unlist(getCovariate(object)))
    distance <- seq(rangeDist[1], rangeDist[2], length = length.out)
  }
  params <- coef(object, unconstrained = FALSE)
  if (length(params) == 1) {            # no nugget effect
    rang <- params
    nugg <- 0
  } else {                              # nugget effect
    rang <- params[1]
    nugg <- params[2]
  }
  val <- data.frame(variog = sig2 * (nugg + (1 - nugg) * FUN(distance, rang)),
                    dist = distance)
  class(val) <- c("Variogram", "data.frame")
  val
}

###*# corSpatialNat - spatial structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corSpatialNat <-
  ## Constructor for the corSpatialNat class
  function(object)
{
  if (!inherits(object, "corSpatial")) {
    stop("Object must inherit from \"corSpatial\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corSpatialNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

coef.corSpatialNat <- 
  function(object, unconstrained = TRUE) 
{
  if (unconstrained && attr(object, "fixed")) {
    return(numeric(0))
  }
  val <- as.vector(object)
  if (attr(object, "nugget")) names(val) <- c("range", "nugget")
  else names(val) <- "range"
  val
}

"coef<-.corSpatialNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  val <- c(log(value), log(value/(1-value)))
  ## updating the factor list and logDet
  aux <- .C("spatial_factList",
	    as.double(val),
	    as.integer(attr(object, "nugget")),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    as.double(attr(object, "minD")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corSpatialNat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  aux <- c(log(aux), log(aux/(1-aux)))
  val <-
    .C("spatial_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       as.double(unlist(getCovariate(object))),
       as.double(attr(object, "minD")),
       as.integer(attr(object, "nugget")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}


###*# corExp - exponential spatial correlation structure

corExp <-
  ## Constructor for the corExp class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c("corExp", "corSpatial", "corStruct")
  value
}

###*# Methods for standard generics

summary.corExp <- 
  function(object, structName = "Exponential spatial correlation")
{
  summary.corStruct(object, structName)
}

Variogram.corExp <-
  function(object, distance = NULL, sig2 = 1, length.out = 50)
{
  Variogram.corSpatial(object, distance, sig2, length.out,
                       function(x, y) { 1 - exp(-x/y) })
}

###*# corGaus - Gaussian spatial correlation structure

corGaus <-
  ## Constructor for the corGaus class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c("corGaus", "corSpatial", "corStruct")
  value
}

###*# Methods for standard generics

summary.corGaus <- 
  function(object, structName = "Gaussian spatial correlation")
{
  summary.corStruct(object, structName)
}

Variogram.corGaus <-
  function(object, distance = NULL, sig2 = 1, length.out = 50)
{
  Variogram.corSpatial(object, distance, sig2, length.out,
                       function(x, y){ 1 - exp(-(x/y)^2) })
}

###*# corLin - Linear spatial correlation structure

corLin <-
  ## Constructor for the corLin class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c("corLin", "corSpatial", "corStruct")
  value
}

###*# Methods for standard generics

asNatural.corLin <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corLinNat(object)
  }
}

coef.corLin <-
  function(object, unconstrained = TRUE)
{
  val <- NextMethod()
  if (!unconstrained) val[1] <- val[1] + attr(object, "minD")
  val
}

initialize.corLin <-
  function(object, data, ...)
{
  if (!is.null(attr(object, "minD"))) { #already initialized
    return(object)
  }
  object <- initialize.corStruct(object, data)
  nug <- attr(object, "nugget")

  minD <- min(unlist(attr(object, "covariate")))
  val <- as.vector(object)
  if (length(val) > 0) {		# initialized
    if (val[1] <= 0) {
      stop("Range must be > 0 in \"corLin\" initial value")
    }
    if (val[1] <= minD) {
      warning(paste("Initial value for range less than minimum distance.",
		    "Setting it to 1.1 * min(distance)"))
      val[1] <- 1.1 * minD
    }
    if (nug) {				# with nugget effect
      if (length(val) == 1) {		# assuming nugget effect not given
	val <- c(val, 0.1)		# setting it to 0.1
      } else {
	if (length(val) != 2) {
	  stop("Initial value for corSpher parameters of wrong dimension")
	}
      }
      if ((val[2] <= 0) || (val[2] >= 1)) {
	stop("Initial value of nugget ratio must be in (0,1)")
      }
    } else {				# only range parameter
      if (length(val) != 1) {
	stop("Initial value for corLin parameters of wrong dimension")
      }
    }
  } else {
    val <- minD * 1.1
    if (nug) val <- c(val, 0.1)
  }
  val[1] <- log(val[1] - minD)
  if (nug) val[2] <- log(val[2]/(1 - val[2]))
  oldAttr <- attributes(object)
  object <- val
  attributes(object) <- oldAttr
  attr(object, "minD") <- minD
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  object
}

summary.corLin <- 
  function(object, structName = "Linear spatial correlation")
{
  summary.corStruct(object, structName)
}

Variogram.corLin <-
  function(object, distance = NULL, sig2 = 1, length.out = 50)
{
  Variogram.corSpatial(object, distance, sig2, length.out,
                       function(x, y) { pmin(x/y, 1) })
}

###*# corLinNat - spatial linear structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corLinNat <-
  ## Constructor for the corLinNat class
  function(object)
{
  if (!inherits(object, "corLin")) {
    stop("Object must inherit from \"corLin\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corLinNat", "corSpatialNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

"coef<-.corLinNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  val <- c(log(value - attr(object, "minD")), log(value/(1-value)))
  ## updating the factor list and logDet
  aux <- .C("spatial_factList",
	    as.double(val),
	    as.integer(attr(object, "nugget")),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    as.double(attr(object, "minD")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corLinNat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  aux <- c(log(aux - attr(object, "minD")), log(aux/(1-aux)))
  val <-
    .C("spatial_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       as.double(unlist(getCovariate(object))),
       as.double(attr(object, "minD")),
       as.integer(attr(object, "nugget")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

###*# corRatio - rational quadratic spatial correlation structure

corRatio <-
  ## Constructor for the corRational class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c("corRatio", "corSpatial", "corStruct")
  value
}

###*# Methods for standard generics

summary.corRatio <- 
  function(object, structName = "Rational quadratic spatial correlation")
{
  summary.corStruct(object, structName)
}

Variogram.corRatio <-
  function(object, distance = NULL, sig2 = 1, length.out = 50)
{
  Variogram.corSpatial(object, distance, sig2, length.out,
                       function(x, y) {
                         x <- (x/y)^2
                         x/(1+x)
                       })
}

###*# corSpher - spherical spatial correlation structure

corSpher <-
  ## Constructor for the corSpher class
  function(value = numeric(0), form = ~ 1, nugget = FALSE,
	   metric = c("euclidean", "maximum", "manhattan"), fixed = FALSE)
{
  attr(value, "formula") <- form
  attr(value, "nugget") <- nugget
  attr(value, "metric") <- match.arg(metric)
  attr(value, "fixed") <- fixed
  class(value) <- c("corSpher", "corSpatial", "corStruct")
  value
}

###*# Methods for standard generics

asNatural.corSpher <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    corSpherNat(object)
  }
}

coef.corSpher <-
  function(object, unconstrained = TRUE)
{
  val <- NextMethod()
  if (!unconstrained) val[1] <- val[1] + attr(object, "minD")
  val
}

initialize.corSpher <-
  function(object, data, ...)
{
  if (!is.null(attr(object, "minD"))) { #already initialized
    return(object)
  }
  object <- initialize.corStruct(object, data)
  nug <- attr(object, "nugget")

  minD <- min(unlist(attr(object, "covariate")))
  val <- as.vector(object)
  if (length(val) > 0) {		# initialized
    if (val[1] <= 0) {
      stop("Range must be > 0 in \"corSpher\" initial value")
    }
    if (val[1] <= minD) {
      warning(paste("Initial value for range less than minimum distance.",
		    "Setting it to 1.1 * min(distance)"))
      val[1] <- 1.1 * minD
    }
    if (nug) {				# with nugget effect
      if (length(val) == 1) {		# assuming nugget effect not given
	val <- c(val, 0.1)		# setting it to 0.1
      } else {
	if (length(val) != 2) {
	  stop("Initial value for corSpher parameters of wrong dimension")
	}
      }
      if ((val[2] <= 0) || (val[2] >= 1)) {
	stop("Initial value of nugget ratio must be between 0 and 1")
      }
    } else {				# only range parameter
      if (length(val) != 1) {
	stop("Initial value for corSpher parameters of wrong dimension")
      }
    }
  } else {
    val <- minD * 1.1
    if (nug) val <- c(val, 0.1)
  }
  val[1] <- log(val[1] - minD)
  if (nug) val[2] <- log(val[2]/(1 - val[2]))
  oldAttr <- attributes(object)
  object <- val
  attributes(object) <- oldAttr
  attr(object, "minD") <- minD
  attr(object, "factor") <- corFactor(object)
  attr(object, "logDet") <- -attr(attr(object, "factor"), "logDet")
  object
}

summary.corSpher <- 
  function(object, structName = "Spherical spatial correlation")
{
  summary.corStruct(object, structName)
}

Variogram.corSpher <-
  function(object, distance = NULL, sig2 = 1, length.out = 50)
{
  Variogram.corSpatial(object, distance, sig2, length.out,
                       function(x, y) {
                         x <- pmin(x/y, 1)
                         1.5 * x - 0.5 * x^3
                       })
}

###*# corSpherNat - spatial spherical structure in natural
###*# parameterization. This is NOT an unconstrained parameterization

####* Constructor

corSpherNat <-
  ## Constructor for the corSpherNat class
  function(object)
{
  if (!inherits(object, "corSpher")) {
    stop("Object must inherit from \"corSpher\"")
  }
  val <- coef(object, unconstrained = FALSE)
  class(object) <- c("corSpherNat", "corSpatialNat", "corStruct")
  coef(object) <- val
  object
}

###*# Methods for standard generics

"coef<-.corSpherNat" <-
  function(object, value) 
{
  if (length(value) != length(object)) {
    stop("Cannot change the length of the parameter of a corStruct object")
  }
  object[] <- value
  corD <- attr(object, "Dim")
  val <- c(log(value - attr(object, "minD")), log(value/(1-value)))
  ## updating the factor list and logDet
  aux <- .C("spatial_factList",
	    as.double(val),
	    as.integer(attr(object, "nugget")),
	    as.double(unlist(getCovariate(object))),
	    as.integer(unlist(corD)),
	    as.double(attr(object, "minD")),
	    factor = double(corD[["sumLenSq"]]),
	    logDet = double(1))[c("factor", "logDet")]
  attr(object, "factor") <- aux[["factor"]]
  attr(object, "logDet") <- -aux[["logDet"]]
  object
}

recalc.corSpherNat <- 
  function(object, conLin)
{
  aux <- as.vector(object)
  aux <- c(log(aux - attr(object, "minD")), log(aux/(1-aux)))
  val <-
    .C("spatial_recalc", 
       Xy = as.double(conLin[["Xy"]]),
       as.integer(unlist(Dim(object))),
       as.integer(ncol(conLin[["Xy"]])),
       as.double(aux),
       as.double(unlist(getCovariate(object))),
       as.double(attr(object, "minD")),
       as.integer(attr(object, "nugget")),
       logLik = double(1))[c("Xy", "logLik")]
  conLin[["Xy"]][] <- val[["Xy"]]
  conLin[["logLik"]] <- conLin[["logLik"]] + val[["logLik"]]
  conLin
}

####*# corWave - Wave spatial correlation structure

#corWave <-
#  ## Constructor for the corWave class
#  function(value = numeric(0), form = ~ 1, nugget = FALSE,
#	   metric = c("euclidean", "maximum", "manhattan"))
#{
#  attr(value, "formula") <- form
#  attr(value, "nugget") <- nugget
#  attr(value, "metric") <- match.arg(metric)
#  class(value) <- c("corWave", "corSpatial", "corStruct")
#  value
#}

####*# Methods for standard generics

#summary.corWave <- 
#  function(object, structName = "Wave spatial correlation")
#{
#  summary.corStruct(object, structName)
#}

###*# corStrat - a class representing different corStruct objects per stratum

####* Constructor

corStrat <-
  ## constructor for the corStrat class
  function(value, form, strata)
{
  if (data.class(value) == "list") {
    ## must be a list of corStruct objects
    if (!all(sapply(value, inherits, "corStruct"))) {
      stop("All elements in value must inherit from corStruct")
    }
    if (length(unique(sapply(value,
               function(el) deparse(as.vector(formula(el)[[2]]))))) > 1) {
      stop("All corStruct elements must have the same formula")
    }
  } else {
    if (!inherits(value, "corStruct")) {
      stop(paste("Value can only be a corStruct object",
                 "or a list of corStruct objects"))
    }
    value <- list(value)
  }
  attr(value, "stratForm") <- asOneSidedFormula(strata)
  class(value) <- c("corStrat", "corStruct")
  if (!missing(form)) {
    if (!inherits(form, "formula") || length(form) != 2) {
      stop("Form must be a two-sided formula")
    }
    formula(value) <- form
  }
  value
}

###*# Methods for local generics

corFactor.corStrat <-
  function(object)
{
  val <- lapply(object, corFactor)
  lD <- sum(sapply(val, function(el) attr(el, "logDet")))
  val <- unlist(lapply(val, as.vector))[attr(object, "factOrder")]
  attr(val, "logDet") <- lD
  val
}

corMatrix.corStrat <-
  function(object, covariate = getCovariate(object), corr = TRUE)
{
  if (data.class(covariate) != "list") {
    stop("Covariate must be a list of length equal to number of groups")
  }
  if (is.null(strat <- attr(object, "strata"))) {
    stop("Need strata information")
  }
  namCov <- names(covariate)
  namGrp <- names(strat)
  if (is.null(namCov) ||
      any(is.na(match(namCov, namGrp)))) {
    stop("Names for covariate must be a subset of group names")
  }
  nStrata <- attr(object, "nStrata")
  val <- vector("list", nStrata)
  for(i in 1:nStrata) {
    wchGrp <- !is.na(match(namCov, namGrp[strat == (i-1)]))
    val[[i]] <- corMatrix(object[[i]], covariate[wchGrp], corr = corr)
  }
  do.call("c", val)[namCov]
}

###*# Methods for standard generics

asNatural.corStrat <-
  function(object, unconstrained = TRUE)
{
  object[] <- lapply(object, asNatural, unconstrained = unconstrained)
  object
}

coef.corStrat <-
  ## Accessor for constrained or unconstrained parameters for
  ## corStrat objects 
  function(object, unconstrained = TRUE) 
{
  if (is.null(attr(object, "nStrata"))) return(numeric(0))
  val <- lapply(object, coef, unconstrained = unconstrained)
  if (unconstrained) return(unlist(val))
  if (any(sapply(val, is.matrix))) return(val)
  strLev <- attr(object, "stratLevs")
  namCoef <- unlist(lapply(val, names))
  lenCoef <- sapply(val, length)
  nams <- paste(namCoef, rep(strLev, lenCoef), sep = ":")
  val <- unlist(val)
  names(val) <- nams
  val
}

"coef<-.corStrat" <-
  function(object, value)
{
  ## Assignment of the unconstrained parameter of corStrat objects
  if (is.null(attr(object, "nStrata"))) {
    stop("Cannot assign coefficients before defining number of strata")
  }
  if (is.null(parMap <- attr(object, "parMap"))) {
    stop("Need parameter map")
  }
  if (length(value) != parMap[nrow(parMap), ncol(parMap)]) {
    stop("Value of incorrect length")
  }
  for (i in names(object)) {
    coef(object[[i]]) <- value[parMap[i,1]:parMap[i,2]]
  }
  ## updating logDet
  attr(object, "logDet") <- sum(sapply(object, logDet))
  object
}

Dim.corStrat <-
  function(object, groups) 
{
  if (missing(groups)) return(lapply(object, Dim))
  ugrp <- unique(groups)
  groups <- factor(groups, levels = ugrp)
  stratFull <- attr(object, "strataFull")
  if (length(groups) != length(stratFull)) {
    stop("Groups must have the same length as full strata vector")
  }
  groups <- split(groups, stratFull)
  nStrata <- attr(object, "nStrata")
  val <- vector("list", nStrata)
  names(val) <- names(object)
  for(i in 1:nStrata) {
    val[[i]] <- Dims(object[[i]], groups[[i]]) # order should be consistent
  }
  val
}

formula.corStrat <-
  ## Accessor for the covariate formula
  function(object) formula(object[[1]])

"formula<-.corStrat" <-
  function(object, value)
{
  for(i in seq(along = object)) {
    formula(object[[i]]) <- value
  }
  object
}

getCovariate.corStrat <- 
  function(object, form = formula(object), data) 
{
  if (!missing(form)) {
    form <- formula(object)
    warning("Cannot change \"form\".")
  }
  covar <- lapply(object, getCovariate)
  if (is.null(covar[[1]])) { # need to calculate it
    if (missing(data)) {
      stop("Need data to calculate covariate of corStrat object")
    }
    if (is.null(stratFull <- attr(object, "strataFull"))) {
      stop("Need full strata information to calculate covariate")
    }
    data <- as.data.frame(data)
    if (length(stratFull) != nrow(data)) {
      stop("Data must have as many rows as the strata vector length")
    }
    data <- split(as.data.frame(data), stratFull)
    for(i in 1:length(object)) {
      covar[[i]] <- getCovariate(object[[i]], data = data[[i]])
    }
    if (any(sapply(covar, data.class) != "list")) {
      stop("All corStruct objects must have groups")
    }
  }
  names(covar) <- NULL
  covar <- do.call("c", covar)
  namGrp <- names(attr(object, "strata"))
  if (any(is.na(match(names(covar), namGrp)))) {
    stop("Mismatch between group names from covariate and strata")
  }
  covar[namGrp]
}

getGroups.corStrat <-
  function(object, form = formula(object), level, data, sep)
{
  if (!is.null(val <- attr(object, "groups"))) return(val)
  val <- lapply(object, getGroups)
  if (is.null(val[[1]])) { # need to calculate
    if (!missing(data)) {
      if ((grpLev <- length(getGroupsFormula(form, asList = TRUE))) > 0) {
        ## use innermost grouping level
        val <- getGroups(data, form, level = grpLev)
        factor(val, levels = unique(as.character(val)))
      } else {
        stop("corStrat can only be used with grouped data")
      }
    } else {
      NULL
    }
  } else {
    unlist(val)[attr(object, "origOrder")]
  }
}

initialize.corStrat <-
  ## Initializes some attributes of corStrat objects
  function(object, data, ...)
{
  if (length(coef(object)) > 0) {
    ## initialized, nothing to do
    return(object)
  }
  stratForm <- attr(object, "stratForm")
  strataF <- eval(stratForm[[2]], data) # full strata
  if (inherits(strataF, "factor")) {
    ## converting to integers
    stratLevs <- levels(strataF)
    if (!is.null(namObj <- names(object))) {
      ## checking consistency
      if (any(sort(namObj) != sort(stratLevs))) {
        stop("Object names inconsistent with strata")
      }
      object <- object[stratLevs]
      attr(object, "stratForm") <- stratForm
      class(object) <- c("corStrat", "corStruct")
    } 
    strataF <- as.integer(strataF)
  } else {
    if (!is.integer(strataF)) {
      stop("Strata must evaluate to either a factor or an integer")
    }
    ustrata <- sort(unique(strataF))
    if (any(diff(ustrata) != 1)) {
      stop("Strata must be a sequence of consecutive integers")
    }
    stratLevs <- as.character(ustrata)
  }
  strataF <- strataF - min(strataF)
  nStrata <- length(stratLevs)
  if (length(object) == 1) {
    ## need to repeat same structure for each stratum
    obj <- vector("list", nStrata)
    names(obj) <- stratLevs
    for(i in 1:nStrata) {
      obj[[i]] <- object[[1]]
    }
    attr(obj, "stratForm") <- stratForm
    class(obj) <- c("corStrat", "corStruct")
    object <- obj
  }
  if (is.null(names(object))) {
    names(object) <- stratLevs
  }
  grps <- getGroups(object, data = data)
  ## checking if strata is outer to groups
  strata <- tapply(strataF, grps, unique)
  if (data.class(strata) == "list") {
    stop("Groups must contain only one stratum")
  }
  data <- split(data, strataF)
  ## initializing individual objects
  for(i in 1:nStrata) {
    object[[i]] <- initialize(object[[i]], data[[i]])
  }
  if (length(object) == 1) {
    ## single stratum, return base corStruct object
    return(object[[1]])
  }
  ## assigning attributes
  attr(object, "groups") <- grps
  attr(object, "origOrder") <-
    as.vector(unlist(split(1:length(grps), strataF)))
  attr(object, "nStrata") <- nStrata
  attr(object, "strata") <- strata
  attr(object, "strataFull") <- strataF
  attr(object, "stratForm") <- stratForm
  attr(object, "stratLevs") <- stratLevs
  attr(object, "logDet") <- logDet(object)
  ## parameter map
  parLen <- sapply(object, function(el) length(coef(el)))
  inf <- cumsum(c(1, parLen))[1:length(parLen)]
  sup <- cumsum(parLen)
  attr(object, "parMap") <-
    array(c(inf, sup), c(length(parLen), 2),
          list(names(object), c("inf","sup")))
  ## index for factors
  lenGrps <- table(grps)
  nGrps <- length(lenGrps)
  fO <- vector("list", nGrps)
  names(fO) <- names(lenGrps)
  base <- 0
  for(i in 1:nGrps) {
    ni2 <- lenGrps[i]^2
    fO[[i]] <- base + (1:ni2)
    base <- base + ni2
  }
  fO <- unlist(fO[unlist(lapply(split(lenGrps, strata), names))])
  attr(object, "factOrder") <- fO
  object
}

"isInitialized<-.corStrat" <-
  function(object, value)
{
  if (!is.logical(value)) {
    stop("Value must be of mode logical")
  }
  if (value) {
    if (is.null(getCovariate(object))) {
      stop("Don't know how to change initialization status")
    }
  } else {
    attr(object, "groups") <- NULL
    for(i in names(object)) {
      isInitialized(object) <- FALSE
    }
  }
  object
}

logDet.corStrat <- 
  function(object, covariate = getCovariate(object))
{
  if (!is.null(val <- attr(object, "logDet"))) {
    return(val)
  }
  if(is.null(strata <- attr(object, "strata"))) {
    stop("Cannot calculate logDet for uninitialized objects")
  }
  val <- 0
  for(i in 1:length(object)) {
    val <- val + logDet(object[[i]],
                        covariate[names(strata)[strata==(i-1)]])
  }
  val
}

print.corStrat <-
  function(x, ...)
{
  if (!is.null(attr(x, "nStrata"))) {
    cat("Correlation structure of class corStrat representing\n")
    for(i in names(x)) {
      cat(" Stratum: ", i, ", structure: ", class(x[[i]])[1],"\n", sep = "")
      print(invisible(coef(x[[i]], unconstrained = FALSE)))
    }
  } else {
    cat("Uninitialized correlation structure of class", class(x)[1], "\n")
  }
}

print.summary.corStrat <-
  function(x, ...)
{
  stNam <- unique(sapply(x, function(el) attr(el, "structName")))
  if (length(stNam) == 1) {
    cat(paste("Correlation Structure: ", stNam,
              " stratified by ", deparse(as.vector(attr(x, "stratForm")[[2]])),
              "\n", sep = ""))
  } else {
    cat(paste("Correlation Structure stratified by ",
              deparse(as.vector(attr(x, "stratForm")[[2]])),
              "\n", sep = ""))
  }
  cat(paste(" Formula:", deparse(as.vector(formula(x[[1]]))),"\n"))
  if (length(stNam) == 1) {
    if (inherits(x[[1]], "corSymm") || inherits(x[[1]], "corBand")) {
      for(i in names(x)) {
        cat("Stratum:", i, "\n")
        formula(x[[i]]) <- NULL
        attr(x[[i]], "structName") <- NULL
        print(invisible(x[[i]]))
      }
    } else {
      cat(" Parameter estimate(s):\n")
      val <- as.vector(unlist(lapply(x, coef, unconstrained = FALSE)))
      val <- array(val, c(length(coef(x[[1]])), length(x)),
                   list(names(coef(x[[1]], unc = FALSE)), names(x)))
      print(invisible(val))
    }
  } else {
    for(i in names(x)) {
      cat("Stratum:", i, "\n")
      formula(x[[i]]) <- NULL
#      cat(attr(x, "structName"), "\n")
#      cat("Parameter estimate(s):\n")
      print(invisible(x[[i]]))
    }
  }
}

recalc.corStrat <-
  function(object, conLin)
{
  nStrata <- attr(object, "nStrata")
  val <- vector("list", nStrata)
  stratF <- attr(object, "strataFull")
  auxCL <- conLin[c("Xy", "logLik")]
  for(i in 1:nStrata) {
    auxCL[["Xy"]] <- conLin[["Xy"]][stratF == (i-1), , drop = FALSE]
    val[[i]] <- recalc(object[[i]], auxCL)[["Xy"]]
  }
  val <- do.call("rbind", val)
  val <- val[attr(object, "origOrder"),, drop = FALSE]
  conLin[["Xy"]][] <- val
  conLin[["logLik"]] <- conLin[["logLik"]] + logLik(object)
  conLin
}
	 
summary.corStrat <-
  function(object, structName = class(object)[1])
{
  val <- lapply(as.list(object), summary)
  attr(val, "stratForm") <- attr(object, "stratForm")
  class(val) <- "summary.corStrat"
  val
}

##*## Beginning of epilogue
### This file is automatically placed in Outline minor mode.
### The file is structured as follows:
### Chapters:     ^L # 
### Sections:    ##*##
### Subsections: ###*###
### Components:  non-comment lines flushed left
###              Random code beginning with a ####* comment

### Local variables:
### mode: S
### mode: outline-minor
### outline-regexp: "\^L\\|\\`#\\|##\\*\\|###\\*\\|[a-zA-Z]\\|\\\"[a-zA-Z]\\|####\\*"
### End:


