### $Id: pdMat.q,v 1.1 2001/09/24 21:35:31 pinheiro Exp $
###
###              Classes of positive-definite matrices
###
### Copyright 1997, 1999 Jose C. Pinheiro <jcp@research.bell-labs.com>,
###                      Douglas M. Bates <bates@stat.wisc.edu>
###
### This file is part of the nlme library for S and related languages.
### It is made available under the terms of the GNU General Public
### License, version 2, or at your option, any later version,
### incorporated herein by reference.
### 
### This program is distributed in the hope that it will be
### useful, but WITHOUT ANY WARRANTY; without even the implied
### warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
### PURPOSE.  See the GNU General Public License for more
### details.
### 
### You should have received a copy of the GNU General Public
### License along with this program; if not, write to the Free
### Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
### MA 02111-1307, USA

##*## Generics that should be implemented for any pdMat class

pdConstruct <-
  ## a virtual constructor for these objects
  function(object, value, ...) UseMethod("pdConstruct")

pdFactor <-
  function(object) UseMethod("pdFactor")

pdMatrix <-
  ## extractor for the pd, correlation, or square-root factor matrix
  function(object, factor = FALSE) UseMethod("pdMatrix")

##*## pdMat - a virtual class of positive definite matrices

###*#  constructor for the virtual class

pdMat <- 
  function(value = numeric(0), form = NULL, nam = NULL, 
	   data = sys.parent(), pdClass = "pdSymm")
{
  if (inherits(value, "pdMat")) {	# nothing to construct
    pdClass <- class(value)
  }
  object <- numeric(0)
  class(object) <- unique(c(pdClass, "pdMat"))
  pdConstruct(object, value, form, nam, data)
}

###*# Methods for local generics

corMatrix.pdMat <-
  function(object)
{
  if (!isInitialized(object)) {
    stop("Cannot access the matrix of uninitialized objects")
  }
  Var <- pdMatrix(object)
  if (length(unlist(dimnames(Var))) == 0) {
    aux <- paste("V", 1:(Dim(Var)[2]), sep = "")
    dimnames(Var) <- list(aux, aux)
  }
  dd <- dim(Var)
  dn <- dimnames(Var)
  stdDev <- sqrt(diag(Var))
  names(stdDev) <- dimnames(Var)[[2]]
  value <- array(t(Var/stdDev)/stdDev, dd, dn)
  attr(value, "stdDev") <- stdDev
  value
}

pdConstruct.pdMat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  if (inherits(value, "pdMat")) {	# constructing from another pdMat
    if (length(form) == 0) {
      form <- formula(value)
    }
    if (length(nam) == 0) {
      nam <- Names(value)
    }
    if (isInitialized(value)) {
      return(pdConstruct(object, as.matrix(value), form, nam, data))
    } else {
      return(pdConstruct(object, form = form, nam = nam, data = data))
    }
  }
  if (length(value) > 0) {
    if (inherits(value, "formula") || data.class(value) == "call") {
      ## constructing from a formula
      if (!is.null(form)) {
	warning("Ignoring argument \"form\"")
      }
      form <- formula(value)
      if (length(form) == 3) {          #two-sided case - nlme
        form <- list(form)
      }
    } else if (is.character(value)) {	# constructing from character array
      if (length(nam) > 0) {
	warning("Ignoring argument \"nam\"")
      }
      nam <- value
    } else if (is.matrix(value)) {	# constructing from a pd matrix
      vdim <- dim(value)
      if (length(vdim) != 2 || diff(vdim) != 0) {
        stop("\"value\" must be a square matrix")
      }
      if (length(unlist(vnam <- dimnames(value))) > 0) {
        vnam <- unique(unlist(vnam))
        if (length(vnam) != vdim[1]) {
          stop("dimnames of value must match or be NULL")
        }
        dimnames(value) <- list(vnam, vnam)
        if (length(nam) > 0) {          # check consistency
	  if (any(is.na(match(nam, vnam))) || any(is.na(match(vnam, nam)))) {
	    stop(paste("Names of \"value\" are not consistent",
		       "with \"nam\" argument"))
	  }
	  value <- value[nam, nam, drop = FALSE]
	} else {
	  nam <- vnam
	}
      }
      form <- form                      # avoid problems with lazy evaluation
      nam <- nam
      object <- chol((value + t(value))/2) # ensure it is positive-definite
      attr(object, "dimnames") <- NULL
      attr(object, "rank") <- NULL
    } else if (is.numeric(value)) {	# constructing from the parameter
      value <- as.numeric(value)
      attributes(value) <- attributes(object)
      object <- value
    } else if (data.class(value) == "list") {
      ## constructing from a list of two-sided formulae - nlme case
      if (!is.null(form)) {
	warning("Ignoring argument \"form\"")
      }
      form <- value
    } else {
      stop(paste(deparse(object), "is not a valid object for \"pdMat\""))
    }
  }

  if (!is.null(form)) {
    if (inherits(form, "formula") && length(form) == 3) {#two-sided case - nlme
      form <- list(form)
    }
    if (is.list(form)) {   # list of formulae
      if (any(!unlist(lapply(form,
                             function(el) {
                               inherits(el, "formula") && length(el) == 3
                             })))) {
        stop("All elements of \"form\" list must be two-sided formulas")
      }
      val <- list()
      for(i in seq(along = form)) {
        if (is.name(form[[i]][[2]])) {
          val <- c(val, list(form[[i]]))
        } else {
          val <- c(val, eval(parse(text = paste("list(",
            paste(paste(all.vars(form[[i]][[2]]), deparse(form[[i]][[3]]),
                        sep = "~"), collapse=","),")"))))
        }
      }
      form <- val
      class(form) <- "listForm"
      namesForm <- Names(form, data)
    } else {
      if (inherits(form, "formula")) {
        namesForm <- Names(asOneSidedFormula(form), data)
        namesForm1 <- NULL
      } else {
        stop("\"form\" can only be a formula or a list of formulae")
      }
    }
    if (length(namesForm) > 0) {
      if (length(nam) == 0) {             # getting names from formula
        nam <- namesForm
      } else {				# checking consistency with names
        if (any(noMatch <- is.na(match(nam, namesForm)))) {
          err <- TRUE
          namCopy <- nam
          indNoMatch <- (1:length(nam))[noMatch]
          if (any(wch1 <- (nchar(nam) > 12))) {
            ## possibly names with .(Intercept) in value
            wch1 <- substring(nam, nchar(nam)-10) == "(Intercept)"
            if (any(wch1)) {
              namCopy[indNoMatch[wch1]] <-
                substring(nam[wch1], 1, nchar(nam[wch1]) - 12)
              noMatch[wch1] <- FALSE
              indNoMatch <- indNoMatch[!wch1]  # possibly not matched
            }
          }
          if (sum(noMatch) > 0) {
            ## still no matches - try adding .(Intercept)
            namCopy[indNoMatch] <-
              paste(namCopy[indNoMatch], "(Intercept)", sep = ".")
          }
          ## try matching modified value
          if (!any(is.na(match(namCopy, namesForm)))) {
            err <- FALSE
          }
          if (err) stop("\"form\" not consistent with \"nam\"")
        }
      }
    }
  }

  if (is.matrix(object)) {	# initialized as matrix, check consistency
    if (length(nam) > 0 && (length(nam) != dim(object)[2])) {
      stop(paste("Length of nam not consistent with dimensions",
		 "of initial value"))
    }
  }
  attr(object, "formula") <- form    
  attr(object, "Dimnames") <- list(nam, nam)
  object
}

pdFactor.pdMat <-
  function(object) 
{
  c(qr.R(qr(pdMatrix(object))))
}

pdMatrix.pdMat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot access the matrix of uninitialized objects")
  }
  if (factor) {
    stop(paste("No default method for extracting the square",
               "root of a pdMat object"))
  } else {
    crossprod(pdMatrix(object, factor = TRUE))
  }
}

###*# Methods for standard generics

as.matrix.pdMat <-
  function(x) pdMatrix(x)

asNatural.pdMat <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    return(object)
  }
  stop(paste("Don't know how to convert pdMat object to natural,",
             "constrained parameterization."))
}

coef.pdMat <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained || !isInitialized(object)) {
    as.vector(object)
  } else {
    stop("Don't know how to obtain constrained coefficients")
  }
}

"coef<-.pdMat" <-
  function(object, value)
{
  value <- as.numeric(value)
  if (isInitialized(object)) {
    if (length(value) != length(object)) {
      stop("Cannot change the length of the parameter after initialization")
    }
  } else {
    return(pdConstruct(object, value))
  }
  class(value) <- class(object)
  attributes(value) <- attributes(object)
  value
}

Dim.pdMat <-
  function(object)
{
  if ((val <- length(Names(object))) > 0) {
    return(c(val, val))
  } else if (isInitialized(object)) {
    return(dim(as.matrix(object)))
  } 
  stop(paste("Cannot access the number of columns of",
	     "uninitialized objects without names."))
}

formula.pdMat <-
  function(object, asList) eval(attr(object, "formula"))

getStrata.pdMat <-
  function(object) attr(object, "strata")

getStrataFormula.pdMat <-
  function(object) attr(object, "stratForm")

isInitialized.pdMat <-
  function(object)
{
  length(object) > 0
}

logDet.pdMat <-
  function(object)
{
  if (!isInitialized(object)) {
    stop(paste("Cannot extract the log of the determinant",
	       "from an uninitialized object"))
  }
  sum(log(svd(pdMatrix(object, factor = TRUE))$d))
}

"matrix<-.pdMat" <-
  function(object, value)
{
  value <- as.matrix(value)
  ## check for consistency of dimensions when object is initialized
  if (isInitialized(object) && any(dim(value) != Dim(object))) {
    stop("Cannot change dimensions on an initialized pdMat object")
  }
  isInv <- attr(object, "inv")
  if (!is.null(isInv) && isInv) {
    value <- solve(value)
  }
  pdConstruct(object, value)
}

Names.pdMat <-
  function(object)
{
  as.character(attr(object, "Dimnames")[[2]])
}

"Names<-.pdMat" <-
  function(object, ..., value)
{
  if (is.null(value)) {
    attr(object, "Dimnames") <- NULL
    return(object)
  } else {
    value <- as.character(value)
    if (length(dn <- Names(object)) == 0) {
      if (isInitialized(object)) {	# object is initialized without names
	if (length(value) != (aux <- Dim(object)[2])) {
	  stop(paste("Length of names should be", aux))
	}
      }
      attr(object, "Dimnames") <- list(value, value)
      return(object)
    }
    if (length(dn) != length(value)) {
      stop(paste("Length of names should be", length(dn)))
    }
    err <- FALSE
    if (any(noMatch <- is.na(match(value, dn)))) {
      err <- TRUE
      ## checking nlme case
      valueCopy <- value
      indNoMatch <- (1:length(value))[noMatch]
      nam1 <- value[noMatch]            # no matching names
      if (any(wch1 <- (nchar(nam1) > 12))) {
        ## possibly names with .(Intercept) in value
        wch1 <- substring(nam1, nchar(nam1)-10) == "(Intercept)"
        if (any(wch1)) {
          valueCopy[indNoMatch[wch1]] <-
            substring(nam1[wch1], 1, nchar(nam1[wch1]) - 12)
          noMatch[wch1] <- FALSE
          indNoMatch <- indNoMatch[!wch1]  # possibly not matched
        }
      }
      if (sum(noMatch) > 0) {
        ## still no matches - try adding .(Intercept)
        valueCopy[indNoMatch] <-
          paste(valueCopy[indNoMatch], "(Intercept)", sep = ".")
      }
      ## try matching modified value
      indMatch <- match(valueCopy, dn)
      if (!any(is.na(indMatch))) {      # all match
        attr(object, "Dimnames") <- list(value, value)
        if ((length(indMatch)) > 1 && any(diff(indMatch) != 1) &&
            isInitialized(object)) { # permutation
          auxMat <- as.matrix(object)[indMatch, indMatch, drop = FALSE]
          dimnames(auxMat) <- list(value, value)
          return(pdConstruct(object, auxMat))
        }
        return(object)
      }
    }
    if (err) {
      stop(paste("Names being assigned do not correspond to a permutation",
                 "of previous names", sep = "\n"))
    }
    indMatch <- match(value, dn)
    if ((length(indMatch) == 1) || all(diff(indMatch) == 1)) {
      return(object)
    }
    ## must be a permutation of names
    attr(object, "Dimnames") <- list(value, value)
    if (isInitialized(object)) {
      auxMat <- as.matrix(object)[indMatch, indMatch, drop = FALSE]
      dimnames(auxMat) <- list(value, value)
      return(pdConstruct(object, auxMat))
    }
    object
  }
}    

"plot.pdMat"<-
  function(x, nseg = 50, levels = 1, center = rep(0, length(stdDev)),
	   additional, ...)
{
  corr <- corMatrix(x)
  stdDev <- attr(corr, "stdDev")
  attr(corr, "stdDev") <- NULL
  assign(".corr", corr, frame = 1)
  assign(".angles", seq(-pi, pi, length = nseg + 1), frame = 1)
  assign(".cosines", cos(.angles), frame = 1)
  nlev <- length(levels)
  dataMat <- array(aperm(outer(rbind(-stdDev, stdDev), levels), c(1, 3, 2)),
		   dim = c(nlev * 2, length(stdDev)),
		   dimnames = list(NULL, names(stdDev)))
  groups <- rep(1:nlev, rep(2, nlev))
  dataMat <- t(t(dataMat) + center)
  if (!missing(additional)) {
    additional <- as.matrix(additional)
    dataMat <- rbind(dataMat, additional)
    groups <- c(groups, rep(0, nrow(additional)))
  }
  splom(~ dataMat, panel = function(x, y, subscripts, groups, ...) {
    groups <- groups[subscripts]	# should be a no-op but
    if (any(g0 <- groups == 0)) {	# plot as points
      panel.xyplot(x[g0], y[g0], ..., type = "p")
    }
    g1 <- groups == 1			# plot the center points
    panel.xyplot(mean(x[g1]), mean(y[g1]), ..., type = "p", pch = 3)
    p <- ncol(.corr)
    laggedCos <- cos(.angles + acos(.corr[round(mean(x[g1])*p + 0.5), 
					  round(mean(y[g1])*p + 0.5)]))
    xylist <- lapply(split(data.frame(x = x[!g0], y = y[!g0]), groups[!g0]),
		     function(el, lagged) {
		       if (nrow(el) != 2) {
			 stop("x-y data to splom got botched somehow")
		       }
		       sumDif <- array(c(1,1,1,-1)/2, c(2,2)) %*% as.matrix(el)
		       list(x = sumDif[1,1] + .cosines * sumDif[2,1],
			    y = sumDif[1,2] + lagged * sumDif[2,2])
		     }, lagged = laggedCos)
    gg <- rep(seq(along = xylist), rep(length(.angles), length(xylist)))
    panel.superpose(unlist(lapply(xylist, "[[", "x")),
		    unlist(lapply(xylist, "[[", "y")),
		    subscripts = seq(along = gg), groups = gg, ..., type = "l")
  }, subscripts = TRUE, groups = groups)
}

print.pdMat <-
  function(x, ...)
{
  if (isInitialized(x)) {
    cat("Positive definite matrix structure of class", class(x)[1], "representing\n")
    print(invisible(as.matrix(x)), ...)
  } else {
    cat("Uninitialized positive definite matrix structure of class ", class(x)[1], 
	".\n", sep = "")
  }
}

print.summary.pdMat <-
  function(x, sigma = 1, rdig = 3, Level = NULL, resid = FALSE, ...)
  ## resid = TRUE causes an extra row to be added
{
  if (!is.list(x)) {
    if (!(is.null(form <- attr(x, "formula")))) {
      cat(paste(" Formula: "))
      if (inherits(form, "formula")) {
        cat(deparse(as.vector(form)))
        if (!is.null(Level)) { cat( paste( " |", Level ) ) }
      } else {
        if (length(form) == 1) {
          cat(deparse(as.vector(form[[1]])))
          if (!is.null(Level)) { cat( paste( " |", Level ) ) }
        } else {
          cat(deparse(lapply(form,
                             function(el) as.name(deparse(as.vector(el))))))
          cat("\n Level:", Level)
        }
      }
      cat( "\n" )
    }
    if (ncol(x) == 1) {
      if (resid) {
        print(array(sigma * c(attr(x, "stdDev"), 1), c(1, 2),
                    list("StdDev:",
                         c(names(attr(x, "stdDev")), "Residual"))), ... )
      } else {
        print(array(sigma * attr(x, "stdDev"), c(1,1),
                    list("StdDev:", names(attr(x, "stdDev")))), ... )
      }
    } else {
      if (!is.null(attr(x, "structName"))) {
        cat(paste(" Structure: ", attr(x, "structName"), "\n", sep = ""))
      }
      if (attr(x, "noCorrelation") | (1 >= (p <- dim(x)[2]))) {
        if (resid) {
          print(array(sigma * c(attr(x, "stdDev"), 1), c(1, p + 1),
                      list("StdDev:",
                           c(names(attr(x, "stdDev")), "Residual"))), ...)
        } else {
          print(array(sigma * attr(x, "stdDev"), c(1, p),
                      list("StdDev:", names(attr(x, "stdDev")))), ...)
        }
      } else {                          # we essentially do print.correlation here
        ll <- lower.tri(x)
        stdDev <- attr(x, "stdDev")
        x[ll] <- format(round(x[ll], digits = rdig), ...)
        x[!ll] <- ""
        xx <- array("", dim(x),
                    list(names(attr(x, "stdDev")),
                         c("StdDev", "Corr", rep("", p - 2))))
        xx[, 1] <- format(sigma * attr(x, "stdDev"))
        xx[-1, -1] <- x[ -1, -p ]
        if (!is.null(dimnames(x)[[2]])) {
          xx[1, -1] <- abbreviate(dimnames(x)[[2]][ -p ], minlength = rdig + 3)
        }
        if (resid) {
          x <- array("", dim(xx) + c(1, 0),
                     list(c(dimnames(xx)[[1]], "Residual"), dimnames(xx)[[2]]))
          x[ 1:p, ] <- xx
          x[ , 1 ] <- format(sigma * c(stdDev, 1))
          xx <- x
        }
        print( xx, ..., quote = FALSE )
      }
    }
  } else {				# composite structure
    cat(paste(" Composite Structure: ", attr(x, "structName"), "\n", sep =""))
    elName <- attr(x, "elementName")
    compNames <- names(x)
    for (i in seq(along = x)) {
      cat(paste("\n ", elName, i))
      if (!is.null(compNames[i]) && nchar(compNames[i]) > 0) {
        cat(paste(": ", compNames[i], "\n", sep = ""))
      } else {
        cat("\n")
      }
      print(x[[i]], sigma = sigma, Level = Level,
            resid = resid && (i == length(x)), ...)
    }
  }
  invisible(x)
}

solve.pdMat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot get the inverse of an uninitialized object")
  }
  matrix(a) <- solve(as.matrix(a))
  a
}

summary.pdMat <-
  function(object, structName = class(object)[1], noCorrelation = FALSE)
{
  if (isInitialized(object)) {
    value <- corMatrix(object)
    attr(value, "structName") <- structName
    attr(value, "noCorrelation") <- noCorrelation
    attr(value, "formula") <- formula(object)
    class(value) <- "summary.pdMat"
    value
  } else {
    object
  }
}

"[.pdMat" <- 
  function(x, i, j, drop = TRUE)
{
  xx <- x
  x <- as.matrix(x)
  if (missing(i)) li <- 0
  else li <- length(i)
  if (missing(j)) lj <- 0
  else lj <- length(j)
  
  if ((li + lj == 0) ||
      (li == lj) && ((mode(i) == mode(j)) && all(i == j))) {
    drop <- F				# even for a 1 by 1 submatrix,
					# you want it to be a matrix
    pdConstruct(xx, NextMethod())
  } else {
    NextMethod()
  }
}

"[<-.pdMat" <- 
  function(x, i, j, value)
{
  xx <- x
  x <- as.matrix(x)
  pdConstruct(xx, NextMethod())
}

##*## Classes that substitute for (i.e. inherit from) pdMat

###*# pdSymm - a class of general pd matrices

####* Constructor

pdSymm <-
  ## Constructor for the pdSymm class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdSymm", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

pdConstruct.pdSymm <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {               # uninitialized object
    class(val) <- c("pdSymm", "pdMat")
    return(val)
  }
  if (is.matrix(val)) {			
    vald <- svd(val, nu = 0)
    object <- vald$v %*% (log(vald$d) * t(vald$v))
    value <- object[row(object) <= col(object)]
    attributes(value) <- attributes(val)[names(attributes(val)) !=  "dim"]
    class(value) <- c("pdSymm", "pdMat")
    return(value)
  }
  Ncol <- round((sqrt(8*length(val) + 1) - 1)/2)
  if (length(val) != round((Ncol * (Ncol + 1))/2)) {
    stop(paste("An object of length", length(val),
	       "does not match the required parameter size"))
  }
  class(val) <- c("pdSymm", "pdMat")
  val
}

pdFactor.pdSymm <-
  function(object)
{
  Ncol <- round((-1 + sqrt(1 + 8 * length(object))) / 2)
  .C("matrixLog_pd", 
     Factor = double(Ncol * Ncol),
     as.integer(Ncol), 
     as.double(object))$Factor
}

pdMatrix.pdSymm <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract matrix from an uninitialized object")
  }
  if (factor) {
    Ncol <- Dim(object)[2]
    value <- array(pdFactor(object), c(Ncol, Ncol), attr(object, "Dimnames"))
    attr(value, "logDet") <- sum(log(abs(svd(value)$d)))
    value
  } else {
    NextMethod()
  }
}

####* Methods for standard generics

asNatural.pdSymm <-
  function(object, unconstrained = TRUE)
{
  pdSymmNat(object, unconstrained = unconstrained)
}

coef.pdSymm <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained || !isInitialized(object)) NextMethod()
  else {				# upper triangular elements
    val <- as.matrix(object)
    aN <- Names(object)
    aN1 <- paste("cov(", aN, sep ="")
    aN2 <- paste(aN, ")", sep ="")
    aNmat <- t(outer(aN1, aN2, paste, sep = ","))
    aNmat[row(aNmat) == col(aNmat)] <- paste("var(",aN,")",sep="")
    val <- val[row(val) <= col(val)]
    names(val) <- aNmat[row(aNmat) <= col(aNmat)]
    val
  }
}

Dim.pdSymm <-
  function(object)
{
  if (isInitialized(object)) {
    val <- round((sqrt(8*length(object) + 1) - 1)/2)
    c(val, val)
  } else {
    NextMethod()
  }
}

logDet.pdSymm <-
  function(object)
{
  if (!isInitialized(object)) {
    stop(paste("Cannot extract the log of the determinant",
	       "from an uninitialized object"))
  }
  attr(pdMatrix(object, factor = TRUE), "logDet")
}

solve.pdSymm <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  coef(a) <- -coef(a, TRUE)
  a
}

summary.pdSymm <-
  function(object,
	   structName = "General positive-definite")
{
  summary.pdMat(object, structName)
}

### No need to implement other methods as the methods for pdMat
### are sufficient.

####*# pdSymmNat - a general positive definite structure parameterized
####   by the log of the square root of the diagonal elements and the
####   generalized logit of the correlations (unconstrained), or the
####   variances and covariances (constrained). This is NOT an
####   unrestricted parametrization. Not all methods are available.

####* Constructor

pdSymmNat <- 
  ## Constructor for the pdSymmNat class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
           unconstrained = TRUE)
{
  object <- numeric(0)
  attr(object, "uncons") <- unconstrained
  class(object) <- c("pdSymmNat", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

pdConstruct.pdSymmNat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- pdConstruct.pdMat(object, value, form, nam, data)
  uncons <- attr(val, "uncons") <- attr(object, "uncons")
  if (length(val) == 0) {               # uninitiliazed object
#indef SP5    
    class(val) <- c("pdSymmNat", "pdMat")
    attr(val, "uncons") <- uncons
    return(val)
  }
  if (is.matrix(val)) {			
    q <- ncol(val)
    if (q > 1) {
      aux <- crossprod(val)
      if (uncons) {
        stdDev <- sqrt(diag(aux))
        aux <- t(aux/stdDev)/stdDev
        aux <- aux[row(aux) > col(aux)]
        value <- c(log(stdDev), log((aux + 1)/(1 - aux)))
      } else {                          # constrained parameterization
        value <- c(diag(aux), aux[row(aux) > col(aux)])
      }
    } else {
      if (uncons) {
        value <- log(val)
      } else {
        value <- val^2
      }
    }
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "uncons") <- uncons
    class(value) <- c("pdSymmNat", "pdMat")
    return(value)
  }
  cfs <- as.vector(val)
  Ncol <- round((sqrt(8*length(cfs) + 1) - 1)/2)
  if (length(cfs) != round((Ncol * (Ncol + 1))/2)) {
    stop(paste("An object of length", length(cfs),
	       "does not match the required parameter size"))
  }
  if (!uncons) {
    if (Ncol > 1) {
      aux <- diag(cfs[1:Ncol])
      aux[row(aux) > col(aux)] <- cfs[-(1:Ncol)]
      aux <- (t(aux) + aux)/2
    } else {
      aux <- as.matrix(cfs)
    }
    if (any(eigen(aux)$value <= 0)) {
      stop("Parameters do not defined a positive-definite matrix")
    }
  }
#indef SP5    
  class(val) <- c("pdSymmNat", "pdMat")
  val
}

pdFactor.pdSymmNat <-
  function(object)
{
  Ncol <- round((-1 + sqrt(1 + 8 * length(object))) / 2)
  .C("symmNat_pd", 
     Factor = double(Ncol * Ncol),
     as.integer(Ncol), 
     as.double(object),
     as.integer(attr(object, "uncons")))$Factor
}

pdMatrix.pdSymmNat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract matrix from an uninitialized object")
  }
  if (factor) {
    Ncol <- Dim(object)[2]
    value <- array(pdFactor(object), c(Ncol, Ncol), attr(object, "Dimnames"))
    attr(value, "logDet") <- sum(log(diag(value)))
    value
  } else {
    NextMethod()
  }
}

####* Methods for standard generics

coef.pdSymmNat <-
  function(object, unconstrained = TRUE)
{
  if (attr(object, "uncons")) {
    if (unconstrained || !isInitialized(object)) return(as.vector(object))
    else {				# standard deviations and correlations
      Ncol <- round((-1 + sqrt(1 + 8 * length(object))) / 2)
      val <- exp(as.vector(object))
      aux <- val[-(1:Ncol)]
      val[-(1:Ncol)] <- (aux - 1) / (aux + 1)
      aN <- Names(object)
      aNmat <- t(outer(aN, aN, paste, sep = ","))
      names(val) <- c(paste("sd(",aN,")", sep = ""), 
                   if (Ncol > 1) {
                     paste("cor(", aNmat[row(aNmat) > col(aNmat)],")",sep="")
                   })
    }
  } else {
    Ncol <- round((-1 + sqrt(1 + 8 * length(object))) / 2)
    val <- as.vector(object)
    aN <- Names(object)
    aNmat <- t(outer(aN, aN, paste, sep = ","))
    names(val) <- c(paste("var(",aN,")", sep = ""), 
                    if (Ncol > 1) {
                      paste("cov(", aNmat[row(aNmat) > col(aNmat)],")",sep="")
                    })
  }
  val
}

Dim.pdSymmNat <-
  function(object)
{
  if (isInitialized(object)) {
    val <- round((sqrt(8*length(object) + 1) - 1)/2)
    c(val, val)
  } else {
    NextMethod()
  }
}

solve.pdSymmNat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot get the inverse of an uninitialized object")
  }
  Ncol <- round((-1 + sqrt(1 + 8 * length(a))) / 2)
  if (Ncol > 1) {
    val <- array(.Fortran("dbksl",
			  as.double(pdFactor(a)),
			  as.integer(Ncol),
			  as.integer(Ncol),
			  val = as.double(diag(Ncol)),
			  as.integer(Ncol),
			  integer(1))[["val"]], c(Ncol, Ncol))
    val <- val %*% t(val)
    if (attr(a, "uncons")) {
      stdDev <- sqrt(diag(val))
      val <- t(val/stdDev)/stdDev
      val <- val[row(val) > col(val)]
      coef(a) <- c(log(stdDev), log((val + 1)/(1 - val)))
    } else {
      coef(a) <- c(diag(val), val[row(val) > col(val)])
    }
  } else {
    if (attr(a, "uncons")) {
      coef(a) <- -coef(a)
    } else {
      coef(a) <- 1/a
    }
  }
  a
}

### No need to implement other methods as the methods for pdMat
### are sufficient.

###*# pdDiag - diagonal structure parameterized by the logarithm of
###   the square root of the diagonal terms.

####* Constructor

pdDiag <-
  ## Constructor for the pdDiag class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdDiag", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

corMatrix.pdDiag <-
  function(object)
{
  val <- diag(length(as.vector(object)))
  attr(val, "stdDev") <- exp(as.vector(object))
  len <- length(as.vector(object))
  if (length(nm <- Names(object)) == 0) {
    nm <- paste("V", 1:len, sep = "")
    dimnames(val) <- list(nm, nm)
  }
  names(attr(val, "stdDev")) <- nm
  val
}
  
pdConstruct.pdDiag <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {               # uninitiliazed object
    return(val)
  }
  if (is.matrix(val)) {			# initialize from a positive definite
    value <- log(diag(crossprod(val)))/2
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    class(value) <- c("pdDiag", "pdMat")
    return(value)
  }
  if ((aux <- length(Names(val))) > 0) {
    if (aux && (aux != length(val))) {
      stop(paste("An object of length", length(val),
		 "does not match the required parameter size"))
    }
  }
  val
}

pdFactor.pdDiag <-
  function(object)
{
  diag(exp(as.vector(object)), length(object))
}

pdMatrix.pdDiag <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized object")
  }
  len <- length(as.vector(object))
  if (factor) {
    value <- diag(exp(as.vector(object)), len)
    attr(value, "logDet") <- sum(as.vector(object))
  } else {
    value <- diag(exp(2 * as.vector(object)), len)
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

asNatural.pdDiag <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    pdDiagNat(object)
  }
}

coef.pdDiag <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) NextMethod()
  else {
    val <- exp(as.vector(object))
    names(val) <- paste("sd(",Names(object),")", sep ="")
    val
  }
}

Dim.pdDiag <-
  function(object)
{
  if (isInitialized(object)) {
    val <- length(object)
    c(val, val)
  } else {
    NextMethod()
  }
}

logDet.pdDiag <-
  function(object)
{
  if (!isInitialized(object)) {
    stop(paste("Cannot extract the log of the determinant",
	       "from an uninitialized object"))
  }
  sum(as.vector(object))
}

solve.pdDiag <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  coef(a) <- -coef(a, TRUE)
  a
}

summary.pdDiag <-
  function(object, structName = "Diagonal")
{
  summary.pdMat(object, structName, noCorrelation = TRUE)
}

### No need to implement other methods as the "pdMat" methods suffice.

###*# pdDiagNat - diagonal structure parameterized by the diagonal terms.
###*# This is NOT an unrestricted parameterization. Not all methods are
###*# available

####* Constructor

pdDiagNat <-
  ## Constructor for the pdDiag class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdDiagNat", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

pdConstruct.pdDiagNat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {
    ## uninitiliazed object 
    return(val)
  }
  if (is.matrix(val)) {			# initialize from a positive definite
    value <- diag(crossprod(val))
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    class(value) <- c("pdDiagNat", "pdMat")
    return(value)
  }
  if ((aux <- length(Names(val))) > 0) {
    if (aux && (aux != length(val))) {
      stop(paste("An object of length", length(val),
		 "does not match the required parameter size"))
    }
  }
  if (any(as.vector(val) <= 0)) {
    stop("Initial variances must be >=0")
  }
  val
}

pdFactor.pdDiagNat <-
  function(object)
{
  diag(sqrt(as.vector(object)), length(object))
}

pdMatrix.pdDiagNat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized object")
  }
  len <- length(as.vector(object))
  if (factor) {
    value <- diag(sqrt(as.vector(object)), len)
    attr(value, "logDet") <- sum(log(as.vector(object)))/2
  } else {
    value <- diag(as.vector(object), len)
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

coef.pdDiagNat <-
  function(object, unconstrained = TRUE)
{
  val <- as.vector(object)
  names(val) <- paste("var(",Names(object),")", sep ="")
  val
}

Dim.pdDiagNat <-
  function(object)
{
  if (isInitialized(object)) {
    val <- length(object)
    c(val, val)
  } else {
    NextMethod()
  }
}

solve.pdDiagNat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  coef(a) <- 1/coef(a)
  a
}

### No need to implement other methods as the "pdMat" methods suffice.

###*# pdIdent: multiple of the identity matrix - the parameter is
###   the log of the multiple. 

####* Constructor

pdIdent <-
  ## Constructor for the pdIdent class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdIdent", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

asNatural.pdIdent <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    pdIdentNat(object)
  }
}

corMatrix.pdIdent <-
  function(object)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }
  val <- diag(Ncol)
  attr(val, "stdDev") <- rep(exp(as.vector(object)), Ncol)
  if (length(nm <- Names(object)) == 0) {
    nm <- paste("V", 1:len, sep = "")
    dimnames(val) <- list(nm, nm)
  }
  names(attr(val, "stdDev")) <- nm
  val
}

pdConstruct.pdIdent <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {			# uninitialized object
    if ((ncol <- length(Names(val))) > 0) {
      attr(val, "ncol") <- ncol
    }
    return(val)
  }
  if (is.matrix(val)) {
    value <- log(mean(diag(crossprod(val))))/2
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "ncol") <- dim(val)[2]
    class(value) <- c("pdIdent", "pdMat")
    return(value)
  }
  if (length(val) > 1) {
    stop(paste("An object of length", length(val),
	       "does not match the required parameter size"))
  }
  if ((aux <- length(Names(val))) == 0) {
    if (is.null(formula(val))) {
      stop(paste("Must give names when initializing pdIdent from parameter.",
                 "without a formula"))
    }
  } else {
    attr(val, "ncol") <- aux
  }
  val
}

pdFactor.pdIdent <-
  function(object)
{
  exp(as.vector(object)) * diag(attr(object, "ncol"))
}


pdMatrix.pdIdent <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }
  value <- diag(Ncol)
  if (factor) {
    value <- exp(as.vector(object)) * value
    attr(value, "logDet") <- Ncol * as.vector(object)
  } else {
    value <- exp(2 * as.vector(object)) * value
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

coef.pdIdent <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) NextMethod()
  else {
    val <- exp(as.vector(object))
    names(val) <- paste("sd(", deparse(formula(object)[[2]]),")",sep = "")
    val
  }
}

Dim.pdIdent <- 
  function(object)
{
  if (!is.null(val <- attr(object, "ncol"))) {
    c(val, val)
  } else {
    stop("Cannot extract the dimensions")
  }
}

logDet.pdIdent <-
  function(object)
{
  attr(object, "ncol") * as.vector(object)
}

solve.pdIdent <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  coef(a) <- -coef(a, TRUE)
  a
}

summary.pdIdent <-
  function(object, structName = "Multiple of an Identity")
{
  summary.pdMat(object, structName, noCorrelation = TRUE)
}

###*# pdIdentNat: multiple of the identity matrix - the parameter is
###*# the multiplier. This is NOT an unrestricted parameterization.
###*# Not all methods are available.

####* Constructor

pdIdentNat <-
  ## Constructor for the pdIdent class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdIdentNat", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

pdConstruct.pdIdentNat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {			# uninitialized object
    if ((ncol <- length(Names(val))) > 0) {
      attr(val, "ncol") <- ncol
    }
    return(val)
  }
  if (is.matrix(val)) {
    value <- mean(diag(crossprod(val)))
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "ncol") <- dim(val)[2]
    class(value) <- c("pdIdentNat", "pdMat")
    return(value)
  }
  if (length(val) > 1) {
    stop(paste("An object of length", length(val),
	       "does not match the required parameter size"))
  }
  if ((aux <- length(Names(val))) == 0) {
    if (is.null(formula(val))) {
      stop(paste("Must give names when initializing pdIdentNat from parameter.",
                 "without a formula"))
    }
  } else {
    attr(val, "ncol") <- aux
  }
  if (val <= 0) {
    stop("Initial value must be positive")
  }
  val
}

pdFactor.pdIdentNat <-
  function(object)
{
  sqrt(as.vector(object)) * diag(attr(object, "ncol"))
}

pdMatrix.pdIdentNat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }
  value <- diag(Ncol)
  if (factor) {
    value <- sqrt(as.vector(object)) * value
    attr(value, "logDet") <- Ncol * log(as.vector(object))/2
  } else {
    value <- as.vector(object) * value
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

coef.pdIdentNat <-
  function(object, unconstrained = TRUE)
{
  val <- as.vector(object)
  names(val) <- paste("var(", deparse(formula(object)[[2]]),")",sep = "")
  val
}

Dim.pdIdentNat <- 
  function(object)
{
  if (!is.null(val <- attr(object, "ncol"))) {
    c(val, val)
  } else {
    stop("Cannot extract the dimensions")
  }
}

solve.pdIdentNat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  coef(a) <- 1/coef(a)
  a
}

###*# pdCompSymm: Compound symmetry structure

####* Constructor

pdCompSymm <-
  ## Constructor for the pdCompSymm class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdCompSymm", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

corMatrix.pdCompSymm <-
  function(object)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }
  obj <- as.vector(object)
  aux <- exp(obj[2])
  aux <- c(exp(2 * obj[1]), (aux - 1/(Ncol - 1))/(aux + 1))
  value <- array(aux[2], c(Ncol, Ncol))
  value[row(value) == col(value)] <- 1
  attr(value, "stdDev") <- rep(exp(obj[1]), Ncol)
  if (length(nm <- Names(object)) == 0) {
    nm <- paste("V", 1:Ncol, sep = "")
    dimnames(value) <- list(nm, nm)
  }
  names(attr(value, "stdDev")) <- nm
  value
}

pdConstruct.pdCompSymm <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {                # uninitialized object
    if ((nc <- length(Names(val))) > 0) {
      attr(val, "ncol") <- nc
    }
    return(val)
  }
  if (is.matrix(val)) {
    value <- crossprod(val)
#    if (length(unique(value[row(value) != col(value)])) > 1) {
#      warning("Initializing pdCompSymm object from non-compound symmetry matrix")
#    }
#    if (any(diag(value) != value[1,1])) {
#      warning("Diagonal of initializing matrix is not constant")
#    }
    nc <- dim(value)[2]
    aux <- 1/sqrt(diag(value))
    aux <- aux * t(value * aux)
    if ((aux <- mean(aux[row(aux) != col(aux)])) <= -1/(nc - 1)) {
      aux <- -1/nc
      warning("Initializing pdCompSymm object is not positive definite")
    }
    value <- c(log(mean(diag(value)))/2, log((aux + 1/(nc - 1))/(1 - aux)))
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "ncol") <- nc
    class(value) <- c("pdCompSymm", "pdMat")
    return(value)
  }
  if (length(val) != 2) {
    stop(paste("An object of length", length(val),
	       "does not match the required parameter size"))
  }
  if ((aux <- length(Names(val))) == 0) {
    if (is.null(formula(val))) {
      stop(paste("Must give names when initializing pdCompSymm from parameter.",
                 "without a formula"))
    }
  } else {
    attr(val, "ncol") <- aux
  }
  val
}

pdFactor.pdCompSymm <-
  function(object)
{
  Ncol <- attr(object, "ncol")
  .C("compSymm_pd", 
     Factor = double(Ncol * Ncol),
     as.integer(Ncol),
     as.double(object))$Factor
}

pdMatrix.pdCompSymm <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }

  obj <- as.vector(object)
  aux <- exp(obj[2])
  aux <- c(exp(2 * obj[1]), (aux - 1/(Ncol - 1))/(aux + 1))
  if (factor) {
    value <- array(pdFactor(object), c(Ncol, Ncol))
    attr(value, "logDet") <-  Ncol * obj[1] + 
      ((Ncol - 1) * log(1 - aux[2]) + log(1 + (Ncol - 1) * aux[2]))/2
  } else {
    value <- array(aux[2], c(Ncol, Ncol))
    value[row(value) == col(value)] <- 1
    value <- aux[1] * value
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

asNatural.pdCompSymm <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained) {
    object
  } else {
    pdCompSymmNat(object)
  }
}

coef.pdCompSymm <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained || !isInitialized(object)) NextMethod()
  else {
    if (is.null(Ncol <- attr(object, "ncol"))) {
      stop(paste("Cannot obtain constrained coefficients with",
		 "uninitialized dimensions"))
    }
    val <- as.vector(object)
    aux <- exp(val[2])
    val <- c(exp(val[1]), (aux - 1 / (Ncol - 1)) / (aux + 1))
    names(val) <- c("std. dev", "corr.")
    val
  }
}

Dim.pdCompSymm <- 
  function(object)
{
  if (!is.null(val <- attr(object, "ncol"))) {
    c(val, val)
  } else {
    stop("Cannot extract the dimensions")
  }
}

logDet.pdCompSymm <-
  function(object)
{
  attr(pdMatrix(object, factor = TRUE), "logDet")
}

solve.pdCompSymm <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  val <- coef(a, unconstrained = FALSE)
  nc <- attr(a, "ncol")
  val1 <- c((1+(nc - 2)*val[2])/(val[1]*val[1]*(1-val[2])*
                  (1+(nc-1)*val[2])), -val[2]/(1 + (nc - 2) * val[2]))
  coef(a) <- c(log(val1[1])/2, log((val1[2] + 1/(nc - 1))/(1 - val1[2])))
  a
}

summary.pdCompSymm <-
  function(object, structName = "Compound Symmetry")
{
  summary.pdMat(object, structName)
}

###*# pdCompSymmNat: Compound symmetry structure with constrained
###*# parameterization. This is NOT an unrestricted parameterization.
###*# Not all methods are available.

####* Constructor

pdCompSymmNat <-
  ## Constructor for the pdCompSymmNat class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent())
{
  object <- numeric(0)
  class(object) <- c("pdCompSymmNat", "pdMat")
  pdConstruct(object, value, form, nam, data)
}

####* Methods for local generics

pdConstruct.pdCompSymmNat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  val <- NextMethod()
  if (length(val) == 0) {                # uninitialized object
    if ((nc <- length(Names(val))) > 0) {
      attr(val, "ncol") <- nc
    }
    return(val)
  }
  if (is.matrix(val)) {
    value <- crossprod(val)
    nc <- dim(value)[2]
    aux <- 1/sqrt(diag(value))
    aux <- aux * t(value * aux)
    if ((aux <- mean(aux[row(aux) != col(aux)])) <= -1/(nc - 1)) {
      aux <- -1/nc
      warning("Initializing pdCompSymmNat object is not positive definite")
    }
    value <- mean(diag(value)) * c(1, aux)
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "ncol") <- nc
    class(value) <- c("pdCompSymmNat", "pdMat")
    return(value)
  }
  if (length(val) != 2) {
    stop(paste("An object of length", length(val),
	       "does not match the required parameter size"))
  }
  if ((aux <- length(Names(val))) == 0) {
    if (is.null(formula(val))) {
      stop(paste("Must give names when initializing pdCompSymmNat from parameter.",
                 "without a formula"))
    }
  } else {
    attr(val, "ncol") <- aux
  }
  if ((val[1] <= 0) || (abs(val[2]) >= val[1])) {
    stop("Initial values do not define a positive-definite matrix")
  }
  val
}

pdFactor.pdCompSymmNat <-
  function(object)
{
  Ncol <- attr(object, "ncol")
  val <- as.vector(object)
  val[2] <- val[2]/val[1]
  val[1] <- log(val[1])/2
  val[2] <- log((val[2] + 1/(nc - 1))/(1 - val[2]))
  .C("compSymm_pd", 
     Factor = double(Ncol * Ncol),
     as.integer(Ncol),
     as.double(val))$Factor
}

pdMatrix.pdCompSymmNat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract the matrix from an uninitialized pdMat object")
  }
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot extract the matrix with uninitialized dimensions"))
  }

  obj <- as.vector(object)
  if (factor) {
    value <- array(pdFactor(object), c(Ncol, Ncol))
    attr(value, "logDet") <-  (Ncol - 1) * log(obj[1] - obj[2])/2 +
      log(obj[1] + (Ncol - 1) * obj[2])/2
  } else {
    value <- array(obj[2], c(Ncol, Ncol))
    value[row(value) == col(value)] <- obj[1]
  }
  dimnames(value) <- attr(object, "Dimnames")
  value
}

####* Methods for standard generics

coef.pdCompSymmNat <-
  function(object, unconstrained = TRUE)
{
  if (is.null(Ncol <- attr(object, "ncol"))) {
    stop(paste("Cannot obtain coefficients with uninitialized dimensions"))
  }
  val <- as.vector(object)
  names(val) <- c("var", "cov")
  val
}

Dim.pdCompSymmNat <- 
  function(object)
{
  if (!is.null(val <- attr(object, "ncol"))) {
    c(val, val)
  } else {
    stop("Cannot extract the dimensions")
  }
}

solve.pdCompSymmNat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  val <- coef(a)
  val[2] <- val[2]/val[1]
  nc <- attr(a, "ncol")
  val1 <- c((1+(nc - 2)*val[2])/(val[1]*(1-val[2])*
                  (1+(nc-1)*val[2])), -val[2]/(1 + (nc - 2) * val[2]))
  coef(a) <- c(val1[1], val1[1] * val1[2])
  a
}

###*# pdBand - a class of banded pd matrices

####* Constructor

pdBand <-
  ## Constructor for the pdBand class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
           ord = 2)
{
  ord <- round(ord)
  if (ord <= 0) {
    stop("Order must be a positive integer")
  }
  object <- numeric(0)
  if (ord == 1) {
    class(object) <- c("pdDiag", "pdMat")
    pdConstruct(object, value, form, nam, data)
  } else {
    class(object) <- c("pdBand", "pdMat")
    attr(object, "order") <- ord
    val <- pdConstruct(object, value, form, nam, data)
    attr(val, "order") <- ord
    val
  }
}

####* Methods for local generics

pdConstruct.pdBand <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  ord <- attr(object, "order")
  if (is.null(ord) && inherits(value, "pdBand")) {
    ord <- attr(value, "order")
  }
  inv <- attr(object, "inv")
  if (is.null(inv) && inherits(value, "pdBand")) {
    inv <- attr(value, "inv")
  }
  attr(object, "order") <- ord
  val <- pdConstruct.pdMat(object, value, form, nam, data)
  if (length(val) == 0) {               # uninitialized object
    attr(val, "order") <- ord
    attr(val, "inv") <- inv
    class(val) <- c("pdBand", "pdMat")
    return(val)
  }
  if (is.matrix(val)) {
    if (ncol(val) < ord) {
      stop("Cannot have order larger than dimension of matrix")
    }
    val[row(val) < (col(val) - ord)] <- 0  # making sure
    value <- c(log(diag(val)), val[(row(val) < col(val)) &
                                   (row(val) > (col(val) - ord))])
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "order") <- ord
    attr(value, "inv") <- inv
    class(value) <- c("pdBand", "pdMat")
    return(value)
  }
  Ncol <- round((2 * length(val) + ord^2 - ord)/(2*ord))
  if (length(val) != round((2*Ncol*ord - ord^2 + ord)/2)) {
    stop(paste("An object of length", length(val),
	       "does not match a pdBand factor of order", ord))
  }
  attr(val, "order") <- ord
  attr(val, "inv") <- inv
  val
}

pdFactor.pdBand <-
  function(object)
{
  ord <- attr(object, "order")
  Ncol <- round((2 * length(object) + ord^2 - ord)/(2*ord))
  val <- diag(Ncol)
  aux <- as.vector(object)
  aux <- c(exp(aux[1:Ncol]), aux[-(1:Ncol)])
  val[(row(val) > (col(val) - ord)) & (row(val) < col(val))] <-
    aux[-(1:Ncol)]
  diag(val) <- aux[1:Ncol]
  isInv <- attr(object, "inv")
  if (!is.null(isInv) && isInv) {
    val <- t(array(.Fortran("dbksl",
                            as.double(val),
                            as.integer(Ncol),
                            as.integer(Ncol),
                            val = as.double(diag(Ncol)),
                            as.integer(Ncol),
                            integer(1))[["val"]], c(Ncol, Ncol)))
  }
  val
}

pdMatrix.pdBand <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract matrix from an uninitialized object")
  }
  if (factor) {
    Ncol <- Dim(object)[2]
    value <- array(pdFactor(object), c(Ncol, Ncol), attr(object, "Dimnames"))
    attr(value, "logDet") <- sum(as.vector(object)[1:Ncol])
    if (!is.null(attr(object, "inv")) && attr(object, "inv")) {
      attr(value, "logDet") <- -attr(value, "logDet")
    }
    value
  } else {
    NextMethod()
  }
}

####* Methods for standard generics

asNatural.pdBand <-
  function(object, unconstrained = TRUE)
{
  pdBandNat(object, ord = attr(object, "order"),
            unconstrained = unconstrained)
}

coef.pdBand <-
  function(object, unconstrained = TRUE)
{
  if (unconstrained || !isInitialized(object)) NextMethod()
  else {				# upper triangular elements
    val <- as.matrix(object)
    aN <- Names(object)
    if (length(aN) == 0) {
      aN <- paste("V", 1:Dim(object)[1], sep = "")
    }
    aN1 <- paste("cov(", aN, sep ="")
    aN2 <- paste(aN, ")", sep ="")
    aNmat <- t(outer(aN1, aN2, paste, sep = ","))
    aNmat[row(aNmat) == col(aNmat)] <- paste("var(",aN,")",sep="")
    indx <-
      (row(aNmat) <= col(aNmat)) & (row(aNmat) >
            (col(aNmat) - attr(object, "order")))
    val <- val[indx]
    names(val) <- aNmat[indx]
    val
  }
}

Dim.pdBand <-
  function(object)
{
  if (isInitialized(object)) {
    ord <- attr(object, "order")
    val <- round((2 * length(object) + ord^2 - ord)/(2*ord))
    c(val, val)
  } else {
    NextMethod()
  }
}

logDet.pdBand <-
  function(object)
{
  if (!isInitialized(object)) {
    stop(paste("Cannot extract the log of the determinant",
	       "from an uninitialized object"))
  }
  val <- sum(as.vector(object)[1:Dim(object)[1]])
  isInv <- attr(object, "inv")
  if (!is.null(isInv) && isInv) {
    val <- -val
  }
  val
}

"matrix<-.pdBand" <-
  function(object, value)
{
  value <- as.matrix(value)
  ## check for consistency of dimensions when object is initialized
  if (isInitialized(object) && any(dim(value) != Dim(object))) {
    stop("Cannot change dimensions on an initialized pdMat object")
  }
  if (!is.null(attr(object, "inv")) && attr(object, "inv")) {
    value <- solve(value)
  }
  pdConstruct(object, value)
}

solve.pdBand <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot extract the inverse from an uninitialized object")
  }
  Ncol <- Dim(a)[1]
  if (Ncol == 1) {
    coef(a) <- -coef(a)
  } else {
    if (is.null(attr(a, "inv"))) {
      attr(a, "inv") <- TRUE
    } else {
      attr(a, "inv") <- !attr(a, "inv")
    }
  }
  a 
}

summary.pdBand <-
  function(object,
	   structName = paste("Banded positive-definite of order",
             attr(object, "order")))
{
  summary.pdMat(object, structName)
}

###*# pdBandNat - a class of banded pd matrices in constrained
###*# parameterization. This is NOT an unrestricted parameterization
###*# Not all methods are available for this class.

####* Constructor

pdBandNat <-
  ## Constructor for the pdBandNat class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
           ord = 2, unconstrained = TRUE)
{
  ord <- round(ord)
  if (ord <= 0) {
    stop("Order must be a positive integer")
  }
  object <- numeric(0)
  attr(object, "uncons") <- unconstrained
  if (ord == 1) {
    class(object) <- c("pdDiagNat", "pdMat")
    pdConstruct(object, value, form, nam, data)
  } else {
    class(object) <- c("pdBandNat", "pdMat")
    attr(object, "order") <- ord
    attr(object, "uncons") <- unconstrained
    val <- pdConstruct(object, value, form, nam, data)
    attr(val, "uncons") <- unconstrained
    attr(val, "order") <- ord
    val
  }
}

####* Methods for local generics

pdConstruct.pdBandNat <-
  function(object, value = numeric(0), form = formula(object), 
	   nam = Names(object), data = sys.parent())
{
  ord <- attr(object, "order")
  if (is.null(ord) && inherits(value, "pdBandNat")) {
    ord <- attr(object, "order") <- attr(value, "order")
    attr(object, "order") <- ord
  }
  inv <- attr(object, "inv")
  if (is.null(inv) && (inherits(value, "pdBandNat") ||
                       inherits(value, "pdBand"))) {
    inv <- attr(value, "inv")
  }
  uncons <- attr(object, "uncons")
  if (is.null(uncons) && inherits(value, "pdBandNat")) {
    uncons <- attr(object, "uncons") <- attr(value, "uncons")
  }
  
  val <- pdConstruct.pdMat(object, value, form, nam, data)
  if (length(val) == 0) {               # uninitialized object
    attr(val, "order") <- ord
    attr(val, "uncons") <- uncons
    attr(val, "inv") <- inv
    class(val) <- c("pdBandNat", "pdMat")
    return(val)
  }
  if (is.matrix(val)) {
    if (ncol(val) < ord) {
      stop("Cannot have order larger than dimension of matrix")
    }
    q <- ncol(val)
    if (q > 1) {
      val[row(val) < (col(val) - ord)] <- 0  # making sure
      aux <- crossprod(val)
      if (uncons) {
        stdDev <- sqrt(diag(aux))
        aux <- t(aux/stdDev)/stdDev
        aux <- aux[(row(aux) < col(aux)) &
                   (row(aux) > (col(aux) - ord))]
        value <- c(log(stdDev), log((aux + 1)/(1 - aux)))
      } else {
        value <- c(diag(aux), aux[(row(aux) < col(aux)) &
                                  (row(aux) > (col(aux) - ord))])
      }
    } else {
      if (uncons) {
        value <- log(val)
      } else {
        value <- val^2
      }
    }
    attributes(value) <- attributes(val)[names(attributes(val)) != "dim"]
    attr(value, "order") <- ord
    attr(value, "uncons") <- uncons
    attr(value, "inv") <- inv
    class(value) <- c("pdBandNat", "pdMat")
    return(value)
  }
  cfs <- as.vector(val)
  Ncol <- round((2 * length(cfs) + ord^2 - ord)/(2*ord))
  if (length(cfs) != round((2*Ncol*ord - ord^2 + ord)/2)) {
    stop(paste("An object of length", length(cfs),
	       "does not match a pdBandNat factor of order", ord))
  }
  if (!uncons) {
    if (Ncol > 1) {
      aux <- diag(cfs[1:Ncol])
      aux[(row(aux) < col(aux)) & (row(aux) > (col(aux) - ord))] <-
        cfs[-(1:Ncol)]
      aux <- (t(aux) + aux)/2
    } else {
      aux <- as.matrix(cfs)
    }
    if (any(eigen(aux)$value <= 0)) {
      stop("Parameters do not defined a positive-definite matrix")
    }
  }
  attr(val, "order") <- ord
  attr(val, "uncons") <- uncons
  attr(val, "inv") <- inv
  class(val) <- c("pdBandNat", "pdMat")
  val
}

pdFactor.pdBandNat <-
  function(object)
{
  ord <- attr(object, "order")
  Ncol <- round((2 * length(object) + ord^2 - ord)/(2*ord))
  if (Ncol > 1) {
    val <- double(Ncol * (Ncol - 1)/2)
    cfs <- as.vector(object)
    aux <- diag(Ncol)
    val[(row(aux) > (col(aux) - ord))[row(aux) < col(aux)]] <- cfs[-(1:Ncol)]
    val <- c(cfs[1:Ncol], val)
  } else {
    val <- as.vector(object)
  }
  val <- .C("symmNat_pd", 
            Factor = double(Ncol * Ncol),
            as.integer(Ncol), 
            as.double(val),
            as.integer(attr(object, "uncons")))$Factor
  isInv <- attr(object, "inv")
  if (!is.null(isInv) && isInv) {
    val <- t(array(.Fortran("dbksl",
                            as.double(val),
                            as.integer(Ncol),
                            as.integer(Ncol),
                            val = as.double(diag(Ncol)),
                            as.integer(Ncol),
                            integer(1))[["val"]], c(Ncol, Ncol)))
  }
  val
}

pdMatrix.pdBandNat <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot extract matrix from an uninitialized object")
  }
  if (factor) {
    Ncol <- Dim(object)[2]
    value <- array(pdFactor(object), c(Ncol, Ncol), attr(object, "Dimnames"))
    if (attr(object, "uncons")) {
      attr(value, "logDet") <- sum(as.vector(object)[1:Ncol])
    } else {
      attr(value, "logDet") <- sum(log(as.vector(object)[1:Ncol]))/2
    }
    if (!is.null(attr(object, "inv")) && attr(object, "inv")) {
      attr(value, "logDet") <- - attr(value, "logDet")
    }
    value
  } else {
    NextMethod()
  }
}

####* Methods for standard generics

coef.pdBandNat <-
  function(object, unconstrained = TRUE)
{
  ord <- attr(object, "order")
  if (attr(object, "uncons")) {
    if (unconstrained || !isInitialized(object)) return(as.vector(object))
    else {				# upper triangular elements
      Ncol <- Dim(object)[1]
      val <- exp(as.vector(object))
      if (Ncol > 1) {
        aux <- val[-(1:Ncol)]
        val[-(1:Ncol)] <- (aux - 1) / (aux + 1)
      }
      aN <- Names(object)
      aNmat <- t(outer(aN, aN, paste, sep = ","))
      namVal <- paste("sd(",aN,")", sep = "")
      if (Ncol > 1) {
        namVal <- c(namVal, 
          paste("cor(", aNmat[(row(aNmat) < col(aNmat)) &
                              (row(aNmat) > (col(aNmat) - ord))],")",sep=""))
      }
      names(val) <- namVal
    }
  } else {
    val <- as.matrix(object)
    aN <- Names(object)
    if (length(aN) == 0) {
      aN <- paste("V", 1:Dim(object)[1], sep = "")
    }
    aN1 <- paste("cov(", aN, sep ="")
    aN2 <- paste(aN, ")", sep ="")
    aNmat <- t(outer(aN1, aN2, paste, sep = ","))
    aNmat[row(aNmat) == col(aNmat)] <- paste("var(",aN,")",sep="")
    indx <-
      (row(aNmat) <= col(aNmat)) & (row(aNmat) >
            (col(aNmat) - ord))
    val <- val[indx]
    names(val) <- aNmat[indx]
  }
  val
}

Dim.pdBandNat <-
  function(object)
{
  if (isInitialized(object)) {
    ord <- attr(object, "order")
    val <- round((2 * length(object) + ord^2 - ord)/(2*ord))
    c(val, val)
  } else {
    NextMethod()
  }
}

solve.pdBandNat <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot get the inverse of an uninitialized object")
  }
  Ncol <- Dim(a)[1]
  if (Ncol == 1) {
    if (attr(a, "uncons")) {
      coef(a) <- -coef(a)
    } else {
      coef(a) <- 1/coef(a)
    }
  } else {
    if (is.null(attr(a, "inv"))) {
      attr(a, "inv") <- TRUE
    } else {
      attr(a, "inv") <- !attr(a, "inv")
    }
  }
  a
}

####*# pdBlocked: A blocked variance structure

#####* Constructor

pdBlocked <-
  ## Constructor for the pdBlocked class
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
	   pdClass = "pdSymm")
{
  object <- numeric(0)
  class(object) <- c("pdBlocked", "pdMat")
  pdConstruct(object, value, form, nam, data, pdClass)
}

####* Methods for local generics

corMatrix.pdBlocked <-
  function(object)
{
  if (!isInitialized(object)) {
    stop("Cannot access the matrix of uninitialized objects")
  }
  if (length(Names(object)) == 0) {
    stop("Cannot access the matrix of object without names")
  }
  namesList <- Names(object, TRUE)
  Ncol <- Dim(object)[2]
  value <- array(0, c(Ncol, Ncol), attr(object, "Dimnames"))
  stdDev <- double(Ncol)
  names(stdDev) <- dimnames(value)[[2]]
  for (i in seq(along = object)) {
    aux <- corMatrix(object[[i]])
    value[namesList[[i]], namesList[[i]]] <- as.vector(aux)
    stdDev[namesList[[i]]] <- attr(aux, "stdDev")
  }
  attr(value, "stdDev") <- stdDev
  value
}
  

pdConstruct.pdBlocked <-
  function(object, value = numeric(0), form = formula(object, TRUE), 
	   nam = Names(object, TRUE), data = sys.parent(), 
	   pdClass = "pdSymm")
{
  if (inherits(value, "pdMat")) {	# constructing from another pdMat
    if (inherits(value, "pdBlocked")) {
      if (length(form) == 0) form <- formula(value, TRUE)
      if (length(nam) == 0) nam <- Names(value, TRUE)
      if (missing(pdClass)) pdClass <- unlist(lapply(value, data.class))
      if (length(object) == 0) object <- value
    }
    if (isInitialized(value)) {
      return(pdConstruct(object, as.matrix(value), form, nam, data, pdClass))
    } else {
      return(pdConstruct(object, form = form, nam = nam, data = data,
                         pdClass = pdClass))
    }
  }
  ## checking validity and consistency of form, nam, and pdClass
  if (!is.null(form)) {
    if (data.class(form) != "list") {
      stop("\"form\" must be a list")
    }
    nF <- length(form)
  } else {
    nF <- 0
  }

  if (!is.null(nam)) {
    if (data.class(nam) != "list") {
      stop("\"nam\" must be a list")
    }
    nN <- length(nam)
    if ((nF > 0) && (nN != nF)) {
      stop("\"form\" and \"nam\" have incompatible lengths")
    }
  } else {
    nN <- 0
  }

  if (!missing(pdClass)) {
    if (!is.character(pdClass)) {
      stop("\"pdClass\" must be a character vector")
    }
    nP <- length(pdClass)
    if ((nP > 1)) {
      if ((nF > 0) && (nF != nP)) {
	stop("\"form\" and \"pdClass\" have incompatible lengths")
      }
      if ((nN > 0) && (nN != nP)) {
	stop("\"nam\" and \"pdClass\" have incompatible lengths")
      }
    }
  } else {
    nP <- 1
  }
  
  nB <- max(c(nF, nN, nP))

  oVal <- value
  if (length(value) == 0 || is.matrix(value) || is.numeric(value)) {
    if (nB == 1) {
      stop("None of the arguments specify more than one block")
    }
    ## will first do a null initialization when value is a matrix or numeric
    value <- lapply(vector("list", nB), function(el) numeric(0))
    if (inherits(object, "pdBlocked") && length(object) == nB) {
      ## will preserve attributes in value
      for (i in 1:nB) attributes(value[[i]]) <- attributes(object[[i]])
    }
  } else {
    if (data.class(value) != "list") {
      stop(paste("\"object\" must be a list, when not missing,",
		 "not a matrix, and not numeric"))
    }
    nO <- length(value)
    if ((nB > 1) && (nB != nO)) {
      stop("Arguments imply different number of blocks")
    }
    nB <- nO
  }
  if (nP == 1) {
    pdClass <- rep(pdClass, nB)
  }

  if (length(object) != nB) {
    object <- vector("list", nB)
  }
  namInterc <- rep(FALSE, nB)
  namCoef <- vector("list", nB)
  for(i in 1:nB) {
    if (is.null(nm <- nam[[i]])) {
      if (is.null(frm <- form[[i]])) {
        if (inherits(value[[i]], "formula")) {
          nm <- Names(getCovariateFormula(value[[i]]))
          if ((length(nm) == 1) && (nm == "(Intercept)") &&
              length(value[[i]]) == 3) {
            ## nlme case with single intercept terms
            nm <-  sapply(splitFormula(getResponseFormula(value[[i]])[[2]],
                                       sep = "+"),
                          function(el) deparse(as.vector(el[[2]])))
          }
          if (length(value[[i]]) == 3) { # nlme case
            namCoef[[i]] <-
              sapply(splitFormula(getResponseFormula(value[[i]])[[2]],
                                  sep = "+"),
                     function(el) deparse(as.vector(el[[2]])))
          }
        }
      } else {
        if (inherits(frm, "formula")) {
          nm <- Names(getCovariateFormula(frm))
          if ((length(nm) == 1) && (nm == "(Intercept)") &&
              length(frm) == 3) {
            ## nlme case with single intercept terms
            nm <-  sapply(splitFormula(getResponseFormula(frm)[[2]],
                                       sep = "+"),
                          function(el) deparse(as.vector(el[[2]])))
          }
          if (length(value[[i]]) == 3) { # nlme case
            namCoef[[i]] <-
              sapply(splitFormula(getResponseFormula(value[[i]])[[2]],
                                  sep = "+"),
                     function(el) deparse(as.vector(el[[2]])))
          }
        } else {                        # listForm
          nm <- unique(unlist(lapply(frm,
                                     function(el) {
                                       Names(getCovariateFormula(el))
                                     })))
          if ((length(nm) == 1) && (nm == "(Intercept)") &&
              length(frm[[1]]) == 3) {
            ## nlme case with single intercept terms
            nm <-  sapply(frm, function(el) {
              sapply(splitFormula(getResponseFormula(el)[[2]],
                                  sep = "+"),
                     function(el1) deparse(as.vector(el1[[2]])))
            })
          }
          namCoef[[i]] <- sapply(frm, function(el) {
            sapply(splitFormula(getResponseFormula(el)[[2]],
                                  sep = "+"),
                   function(el1) deparse(as.vector(el1[[2]])))
          })
        }
      }
    }
    if (!is.null(nm)) {
      namInterc[i] <- (length(nm) == 1) && (nm == "(Intercept)")
    }
    object[[i]] <- pdMat(value[[i]], form[[i]], nam[[i]], data, pdClass[i])
  }
  if (!all(unlist(lapply(object, inherits, "pdMat")))) {
    stop("all elements in the argument must generate pdMat objects")
  }
  namesList <- lapply(object, Names)
  lNam <- unlist(lapply(namesList, length))
#  namInterc <- unlist(lapply(namesList,
#                             function(el) {
#                               (length(el) == 1) && (el == "(Intercept)")
#                             }))
  if (!is.null(namCoef[[1]])) {         # nlme case
    namCoef <- unlist(namCoef)
    duplCoef <- unique(namCoef[duplicated(namCoef)])
    if (length(duplCoef) > 0) {
      for(i in 1:nB) {
        wchDupl <- !is.na(match(namesList[[i]], duplCoef))
        if (any(wchDupl)) {
          namesList[[i]][wchDupl] <-
            paste(namesList[[i]][wchDupl], "(Intercept)", sep = ".")
          Names(object[[i]]) <- namesList[[i]]
        }
      }
    }
  }
  if (sum(namInterc) > 1 && (length(unique(lNam[namInterc])) == 1)) {
    stop("Cannot have duplicated column names in a pdMat object")
  }
  if ((sum(namInterc) == length(lNam)) ||
      !any(lNam[!namInterc])) {			# no names
    class(object) <- c("pdBlocked", "pdMat")
    if (is.null(formula(object))) {
      stop("Must have formula, when no names are given")
    }
    if (length(oVal) && (is.matrix(oVal) || is.numeric(oVal))) {
      stop("Must give names when initializing from matrix or parameter")
    }
    return(object)
  } else {
    if (!all(lNam)) {
      stop("All elements must have names, when any has names.")
    }
    attr(object, "namesList") <- namesList
    allNames <- unlist(namesList)
    if (any(duplicated(allNames))) {
      stop("Cannot have duplicated column names in a pdMat object")
    }
    plen <- unlist(lapply(object, function(el)
			  {
			    if (isInitialized(el)) {
			      length(coef(el, TRUE))
			    } else {
			      matrix(el) <- diag(length(Names(el)))
			      length(coef(el, TRUE))
			    }
			  }))
    if (!all(plen)) {
      stop("All elements must have a non-zero size")
    }
    attr(object, "plen") <- plen
    attr(object, "Dimnames") <- list(allNames, allNames)
    class(object) <- c("pdBlocked", "pdMat")

    if (length(oVal) > 0) {
      if (is.matrix(oVal)) {		# initializing from matrix
	matrix(object) <- oVal
      } else if (is.numeric(oVal)){		# initializing from a vector
	coef(object) <- oVal
      }
    }
    return(object)
  }
}

pdMatrix.pdBlocked <-
  function(object, factor = FALSE)
{
  if (!isInitialized(object)) {
    stop("Cannot access the matrix of uninitialized objects")
  }
  if (length(Names(object)) == 0) {
    stop("Cannot access the matrix of object without names")
  }
  namesList <- Names(object, TRUE)
  Ncol <- Dim(object)[2]
  value <- array(0, c(Ncol, Ncol), attr(object, "Dimnames"))
  if (factor) {
    lD <- 0
  }
  for (i in seq(along = object)) {
    aux <- pdMatrix(object[[i]], factor)
    value[namesList[[i]], namesList[[i]]] <- as.vector(aux)
    if (factor) lD <- lD + attr(aux, "logDet")
  }
  if (factor) attr(value, "logDet") <- lD
  value
}

####* Methods for standard generics

asNatural.pdBlocked <-
  function(object, unconstrained = TRUE)
{
  for (i in 1:length(object)) {
    object[[i]] <- asNatural(object[[i]], unconstrained = unconstrained)
  }
  object
}

coef.pdBlocked <-
  function(object, unconstrained = TRUE)
{
  unlist(lapply(object, coef, unconstrained))
}

"coef<-.pdBlocked" <-
  function(object, value)
{
  if (is.null(plen <- attr(object, "plen"))) {
    stop(paste("Cannot change the parameter when",
	       "length of parameters is undefined"))
  }
  if (length(value) != sum(plen)) {
    stop("Cannot change parameter length of initialized pdMat object")
  }
  ends <- cumsum(plen)
  starts <- 1 + c(0, ends[-length(ends)])
  for (i in seq(along = object)) {
    coef(object[[i]]) <- value[(starts[i]):(ends[i])]
  }
  object
}

formula.pdBlocked <-
  function(object, asList = TRUE)
{
  val <- lapply(object, formula)
  isNULL <- unlist(lapply(val, is.null))
  if (all(isNULL)) return(NULL)
  if (any(isNULL)) {
    stop("All elements must have formulas, when any has a formula.")
  }
  if (asList) return(val)
  isTwoSided <- unlist(lapply(val,
                              function(el) {
                                inherits(el, "listForm")
                              }))
  if (all(isTwoSided)) {
    ## list of two-sided formulas
    val <- do.call("c", val)
#    for(i in seq(along = object)) {
#      val <- if (inherits(object[[i]], "formula")) list(object[[i]])
#               else object[[i]]
#    }
    class(val) <- "listForm"
    return(val)
  }
  if (any(isTwoSided)) {
    stop(paste("All elements of formula must be list of two-sided formulae",
               "or two-sided formulae"))
  }
  val <- lapply(val, terms)
  aux <- paste(unlist(lapply(val, function(el) attr(el, "term.labels"))),
	       collapse = "+")
  if (!any(unlist(lapply(val, function(el) attr(el, "intercept"))))) {
    ## no intercept
    aux <- paste(aux, " - 1")
  }
  eval(parse(text = paste("~", aux)))
}

isInitialized.pdBlocked <-
  function(object)
{
  all(unlist(lapply(object, isInitialized)))
}

logDet.pdBlocked <-
  function(object)
{
  sum(unlist(lapply(object, logDet)))
}

"matrix<-.pdBlocked" <-
  function(object, value)
{
  value <- as.matrix(value)
  namesList <- Names(object, TRUE)
  Ncol <- Dim(object)[2]
  dims <- dim(value)
  if (!((dims[1] == dims[2]) && (dims[1] == Ncol))) {
    stop("Cannot change the number of columns on an initialized object")
  }
  if (is.null(vNames <- dimnames(value)[[1]])) {
    vNames <- unlist(namesList)
    dimnames(value) <- list(vNames, vNames)
  } else {
    if (!(all(match(unlist(namesList), vNames, nomatch = 0)))) {
      stop("Names of object and value must match.")
    }
    attr(object, "Dimnames") <- list(vNames, vNames)
  }
  for (i in seq(along = object)) {
    matrix(object[[i]]) <- value[namesList[[i]], namesList[[i]]]
  }
  object
}

Names.pdBlocked <-
  function(object, asList = FALSE)
{
  if (asList) attr(object, "namesList")
  else attr(object, "Dimnames")[[2]]
}

"Names<-.pdBlocked" <-
  function(object, ..., value)
{
  if (!is.null(Names(object))) NextMethod()
  else {
    ## cannot do anything before initialization of names
    object
  }
}

pdFactor.pdBlocked <-
  function(object)
{
  pdMatrix(object, factor = TRUE)
}

solve.pdBlocked <-
  function(a, b)
{
  if (!isInitialized(a)) {
    stop("Cannot get the inverse of an uninitialized object")
  }
  oAttr <- attributes(a)
  a <- lapply(a, solve)
  attributes(a) <- oAttr
  a
}

summary.pdBlocked <-
  function(object, structName = "Blocked")
{
  value <- lapply(object, summary)
  names(value) <- unlist(lapply(object, function(el) paste(Names(el),
							   collapse = ", ")))
  ## too many names for pdKron; will NULL it
  if (any(wchKron <- sapply(object, inherits, "pdKron"))) {
    names(value)[wchKron] <- rep("", sum(wchKron))
  }
  attr(value, "structName") <- structName
  attr(value, "elementName") <- "Block"
  class(value) <- "summary.pdMat"
  value
}

"[.pdBlocked" <- 
  function(x, i, j, drop = TRUE)
{
  xx <- x
  x <- as.matrix(x)
  mCall <- match.call()
  mCall[[1]] <- get("[")
  mCall[["x"]] <- x
  mCall[["drop"]] <- drop
  if (length(i) == length(j) && mode(i) == mode(j) && all(i == j)) {
    mCall[["drop"]] <- F		# even for a 1 by 1 submatrix,
					# you want it to be a matrix
    val <- eval(mCall)
    vNames <- dimnames(val)[[2]]
    auxNames <- lapply(Names(xx, TRUE), 
		       function(el, vN) {
			 aux <- match(vN, el)
			 if (any(aux1 <- !is.na(aux))) {
			   el[aux[aux1]]
			 }
		       }, vN = vNames)
    auxWhich <- !unlist(lapply(auxNames, is.null))
    if (sum(auxWhich) == 1) {
      return(pdConstruct(as.list(xx)[auxWhich][[1]], val))
    }
    auxNames <- auxNames[auxWhich]
    auxClass <- unlist(lapply(xx, function(el) class(el)[1]))[auxWhich]
    return(pdConstruct(xx, val, nam = auxNames, form = NULL, 
		       pdClass = auxClass))
  } else {
    eval(mCall)
  }
}

###*# pdKron - a class of Kronecker product pd matrices
###*#   structure for fitting crossed random effects models uses a
###*#   base pdMat object and separate groups to define a Kronecker
###*#   product matrix for all random effects 

pdKron <-
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
           pdClass = "pdSymm", groups)
{
  
  object <- numeric(0)
  if (inherits(value, "pdMat")) {       # will overwrite pdClass
    if (inherits(value, "pdKron")) {
      ## use only the atomic part of the class
      class(value) <- class(value)[-1]
    }
    pdClass <- class(value)[1]
  }
  class(object) <- c("pdKron", pdClass, "pdMat")
  attr(object, "groups") <- asOneSidedFormula(groups)
  pdConstruct(object, value, form, nam, data)
}

pdConstruct.pdKron <-
  function(object, value = numeric(0), form = NULL, nam = NULL,
           data = sys.parent())
{
  if (inherits(value, "pdKron")) {
    ## possibly use attributes from value in object
    if (is.null(attr(object, "groups"))) {
      attr(object, "groups") <- attr(value, "groups")
    }
    if (is.null(attr(object, "index"))) {
      attr(object, "index") <- attr(value, "index")
    }
    if (is.null(attr(object, "groupLevels"))) {
      attr(object, "groupLevels") <- attr(value, "groupLevels")
    }
    ## use only atomic part of the class
    class(value) <- class(value)[-1]
    if (!is.null(valForm <- formula(value))) {
      if (!is.null(form)) {
        form <- valForm
      }
    }
    if (!is.null(valNam <- Names(value))) {
      if (!is.null(nam)) {
        nam <- valNam
      }
    }
  }
  class(object) <- class(object)[-1]
  if (!missing(data) && is.null(attr(object, "groupLevels")) &&
      (length(Names(object)) > 0)) {
    ## first access to data, may need to correct names
    nam <- Names(formula(object), data)
    attr(object, "Dimnames") <- list(nam, nam)
  }
  val <- NextMethod()
  attr(val, "groups") <- attr(object, "groups")
  if (is.null(attr(object, "index"))) {
    if (length(Names(val)) > 0) {
      ## had access to data and has a formula
      grps <-
        pruneLevels(as.factor(eval(attr(object, "groups")[[2]], data)))
      numLevels <- length(attr(val, "groupLevels") <- levels(grps))
      nCol <- length(Names(val))
      attr(val, "index") <- rep((0:(numLevels-1)) * nCol, nCol) +
        rep(1:nCol, rep(numLevels, nCol))
    }
  } else {
    attr(val, "index") <- attr(object, "index")
    attr(val, "groupLevels") <- attr(object, "groupLevels")
  }
  class(val) <- c("pdKron", class(val))
  val
}

pdFactor.pdKron <-
  function(object)
{
  class(object) <- class(object)[-1]
  val <- pdFactor(object)
  ind <- attr(object, "index")
  nL <- length(attr(object, "groupLevels"))
  c(kronecker(diag(nL), array(val, Dim(object)))[ind, ind])
}
  
pdMatrix.pdKron <-
  function(object, factor = FALSE)
{
  nam <- Names(object)
  if (is.null(gL <- attr(object, "groupLevels"))) {
    ## names uninitialized, will call next method
    class(object) <- class(object)[-1]
    return(pdMatrix(object, factor))
  }
  nL <- length(gL)
  if (factor) {
    Ncol <- Dim(object)[2]
    value <- array(pdFactor(object), Dim(object), list(nam, nam))
    attr(value, "logDet") <- logDet(object)
  } else {
    class(object) <- class(object)[-1]
    value <- pdMatrix(object, factor)
    ind <- attr(object, "index")
    value <- kronecker(diag(nL), array(value, Dim(object)))[ind, ind]
    dimnames(value) <- list(nam, nam)
  }
  value
}


asNatural.pdKron <-
  function(object, unconstrained = TRUE)
{
  class(object) <- class(object)[-1]
  val <- asNatural(object, unconstrained)
  attr(val, "groups") <- attr(object, "groups")
  attr(val, "index") <- attr(object, "index")
  attr(val, "groupLevels") <- attr(object, "groupLevels")
  class(val) <- c("pdKron", class(val))
  val
}

coef.pdKron <-
  function(object, unconstrained = TRUE)
{
  class(object) <- class(object)[-1]
  val <- coef(object, unconstrained)
  if (length(namsVal <- names(val)) > 0) {
    grps <- deparse(as.vector(attr(object, "groups")[[2]]))
    for(i in namsVal) {
      names(val)[i] <- paste(i, grps, sep = "|")
    }
  }
  val
}

Dim.pdKron <-
  function(object)
{
  class(object) <- class(object)[-1]
  val <- Dim(object)
  if (!is.null(gL <- attr(object, "groupLevels"))) {
    val * length(gL)
  } else {
    stop("Cannot get dimensions of uninitialized object")
  }
}

formula.pdKron <-
  function(object, asList)
{
  fixForm <- function(form, groups) {
    RHS <- form[[length(form)]]
    if (length(all.vars(RHS)) == 0) {
      ## single intercept case
      val <- paste(groups, "1", sep = "-")
    } else {
      aux <- terms(form)
      if (attr(aux, "intercept") == 1) {    # intercept
        val <- paste(groups,
             paste(groups, ":(",deparse(as.vector(RHS)), ") -1", sep = ""),
                     sep = "+")
      } else {
        RHS <- attr(terms(asOneSidedFormula(RHS)),"term.labels")
        RHS <- paste(groups, RHS, sep = ":")
        val <- paste(paste(RHS, collapse = "+"), "-1", sep = "")
      }
    }
    val <- paste("~", val)
    if (length(form) == 3) {
      val <- paste(deparse(as.vector(form[[2]])), val)
    }
    eval(parse(text = val))
  }
  groups <- deparse(as.vector(attr(object, "groups")[[2]]))
  class(object) <- class(object)[-1]
  val <- formula(object, asList = FALSE)
  if (is.null(val)) {
    stop("Cannot extract formula from uninitialized pdKron object")
  }
  if (inherits(val, "listForm")) { # nlme case
    for(i in 1:length(val)) {
      val[[i]] <- fixForm(val[[i]], groups)
    }
  } else {
    val <- fixForm(val, groups)
  }
  val
}

getGroupsFormula.pdKron <-
  function(object) attr(object, "groups")

logDet.pdKron <-
  function(object)
{
  if (is.null(gL <- attr(object, "groupLevels"))) {
    stop(paste("Cannot extract the log of the determinant",
	       "from an uninitialized pdKron object"))
  }
  class(object) <- class(object)[-1]
  length(gL) * attr(pdMatrix(object, factor = TRUE), "logDet")
}

"matrix<-.pdKron" <-
  function(object, value)
{
  if (is.null(gL <- attr(object, "groupLevels"))) {
    stop("Cannot assign matrix to uninitialized pdKron object")
  }
  attr(object, "index") <- attr(object, "index")
  value <- as.matrix(value)
  ## two cases: either value for the atomic pdClass, or
  ## for the full pdKron object
  nC <- length(attr(object, "Dimnames")[[2]])
  dimObj <- rep(length(attr(object, "index")), 2)
  nL <- length(gL)
  if (all(dim(value) == dimObj)) {
    ## full matrix case; will reduce it by taking average of
    ## block diagonal terms
    ind <- order(attr(object, "index"))
    value <- value[ind, ind]            # putting in block diagonal order
    aux <- array(0, c(nL, nC * nC))
    auxInd <- 1:nC
    for(i in 1:nL) {
      aux[i,] <- c(value[auxInd, auxInd])
      auxInd <- auxInd + nC
    }
    value <- array(apply(aux, 2, mean, trim = 0.1, na.rm = T), c(nC, nC))
    value <- (value + t(value))/2
    dimnames(value) <- NULL             # making sure
  }
  pdConstruct(object, value)
}

Names.pdKron <-
  function(object)
{
  val <- as.character(attr(object, "Dimnames")[[2]])
  gL <- attr(object, "groupLevels")
  nV <- length(val)
  if (is.null(gL) || nV == 0) {
    return(character(0))
  }
  ## if nL is not null, index must also be defined
  if (length(attr(object, "index")) == nV) {
    ## names defined in full matrix format
    return(val)
  }
  nL <- length(gL)
  ## will exapand names to match dimensions of full matrix
  val <- rep(val, rep(nL, nV))
  ## adding groups to make names unique
  grps <- deparse(as.vector(attr(object, "groups")[[2]]))
  paste(paste(grps, rep(gL, nV), sep = ":"), val, sep = ".")
}

print.summary.pdKron <-
  function(x, sigma = 1, rdig = 3, Level = NULL, resid = FALSE, ...)
  ## resid = TRUE causes an extra row to be added
{
  mCall <- as.list(match.call())[-1]
  for(i in names(mCall)) {              # forcing evaluation
    mCall[[i]] <- eval(as.name(i))
  }
  groups <- deparse(as.vector(attr(x, "groups")[[2]]))
  if (is.null(Level) || Level == "1") {
    mCall$Level <- groups
  } else {
    mCall$Level <- paste(groups, mCall$Level, sep = " %in% ")
  }
  class(x) <- class(x)[-1]
  mCall$x <- x
  do.call("print", mCall)
}

solve.pdKron <-
  function(a, b)
{
  class(a) <- class(a)[-1]
  coef(a) <- coef(solve(a))
  class(a) <- c("pdKron", class(a))
  a
}

summary.pdKron <-
  function(object, structName)
{
  class(object) <- class(object)[-1]
  val <- NextMethod()
  attr(val, "groups") <- attr(object, "groups")
  class(val) <- c("summary.pdKron", class(val))
  val
}

###*# pdStrat - a class representing different pd matrices per stratum

pdStrat <-
  function(value = numeric(0), form = NULL, nam = NULL, data = sys.parent(),
           strata)
{
  object <- list(numeric(0))
  class(object) <- c("pdStrat", "pdMat")
  attr(object, "stratForm") <- asOneSidedFormula(strata)
  pdConstruct(object, value, form, nam, data)
}

###*# Methods for local generics

corMatrix.pdStrat <-
  function(object)
{
  lapply(object, corMatrix)
}

pdConstruct.pdStrat <-
  function(object, value = numeric(0), form = formula(object),
           nam = Names(object), data = sys.parent())
{
  if (!is.list(object)) {
    ## must be numeric(0)
    stratForm <- attr(object, "stratForm")
    object <- list(numeric(0))
    attr(object, "stratForm") <- stratForm
    class(object) <- c("pdStrat", "pdMat")
  }
  if (inherits(value, "pdStrat")) {
    ## constructing from another pdStrat object
    if (is.null(stratForm <- attr(object, "stratForm"))) {
      stratForm <- attr(value, "stratForm")
    }
    if (length(object) > 1) {
      ## object has been initialized with strata information
      if (length(object) != length(value)) {
        stop("Value must have the same length as object")
      }
      for(i in seq(along = value)) {
        object[[i]] <- pdConstruct(object[[i]], value[[i]], form, nam, data)
      }
    } else {
      ## object has not had access to data
      obj <- vector("list", length(value))
      if (inherits(object[[1]], "pdMat")) {
        for(i in seq(along = value)) {
          obj[[i]] <- pdConstruct(object[[1]], value[[i]], form, nam, data)
        }
      } else {
        for(i in seq(along = value)) {
          obj[[i]] <- pdConstruct(value[[i]], value[[i]], form, nam, data)
        }
      }
      object <- obj
    }
    if (is.null(attr(object, "nStrata"))) {
      attr(object, "stratForm") <- stratForm
      attr(object, "strata") <- attr(value, "strata")
      attr(object, "nStrata") <- attr(value, "nStrata")
      attr(object, "stratLevs") <- attr(value, "stratLevs")
    }
    if (is.null(names(object))) {
      names(object) <- names(value)
    }
    class(object) <- c("pdStrat", "pdMat")
  }
  stratForm <- attr(object, "stratForm")
  if (!missing(data) && is.list(data)) {
    ## access to data: need to get strata
    strata <- eval(stratForm[[2]], data)
    if (inherits(strata, "factor")) {
      ## converting to integers
      stratLevs <- levels(strata)
      strata <- as.integer(strata)
    } else {
      if (!is.integer(strata)) {
        stop("Strata must evaluate to either a factor or an integer")
      }
      ustrata <- sort(unique(strata))
      if (any(diff(ustrata) != 1)) {
        stop("Strata must be a sequence of consecutive integers")
      }
      stratLevs <- as.character(ustrata)
    }
    nStrata <- length(stratLevs)
    if (length(object) > 1) {
      ## has already had access to data
      if (!is.null(namObj <- names(object))) {
        ## names should be consistent
        if (any(is.na(match(stratLevs, namObj)))) {
          stop("Stratum levels do not match object names")
        }
        object <- as.list(object)[stratLevs]
      } else {
        if (length(object) < nStrata) {
          stop(paste("Number of strata must be less than or equal to",
                     "the length of object"))
        }
        object <- as.list(object)[1:nStrata]
        names(object) <- stratLevs
      }
    }
    attr(object, "nStrata") <- nStrata
    attr(object, "strata") <- strata - min(strata)   # zero origin
    attr(object, "stratLevs") <- stratLevs
    attr(object, "stratForm") <- stratForm
  }
  nStrata <- attr(object, "nStrata")
  stratLevs <- attr(object, "stratLevs")
  strata <- attr(object, "strata")
  if (inherits(value, "pdStrat")) {
    if (!is.null(nStrata)) {
      ## check if right length
      if (length(object) == 1) {
        if (nStrata > 1) {
          ## repeat same pdMat structure nStrata times
          obj <- vector("list", nStrata)
          for(i in 1:nStrata) {
            obj[[i]] <- object[[1]]
          }
          attributes(obj) <- attributes(object)
          names(obj) <- stratLevs
          class(obj) <- c("pdStrat", "pdMat")
          object <- obj
        } else {
          ## single stratum
          return(object[[1]])
        }
      }
    }
  } else{
    ## checking what type of "value" is used
    if (data.class(value) == "list") {
      if (all(sapply(value, is.matrix)) ||
          all(sapply(value, is.numeric))||
          all(sapply(value, inherits, "pdMat"))) {
        ## list of matrices, numeric vectors, or pdMat objects
        if (!is.null(nStrata) && (length(value) != nStrata)) {
          stop(paste("When given as a list, value must have the same length",
                     "as the number of strata"))
        }
        ## checking consistency with stratum levels, if given
        if (!is.null(stratLevs)) {
          if (!is.null(namVal <- names(value))) {
            if (any(sort(stratLevs) != sort(namVal))) {
              stop("Names of value should be consistent with stratum levels")
            }
            value <- value[stratLevs]
          } else {
            ## asumed in the same order as stratum levels
            names(value) <- stratLevs
          }
        }
        if (all(sapply(as.list(object), length) == 0)) {
          ## uninitialized
          object <- vector("list", length(value))
          names(object) <- names(value)
          for(i in seq(along = object)) {
            object[[i]] <- pdMat(value[[i]], form, nam, data)
          }
        } else {
          for(i in seq(along = obj)) {
            object[[i]] <- pdConstruct(object[[i]], value[[i]], form, nam,
                                       data)
          }
        }
      } else {
        stop(paste("When given as a list, value must have as elements",
                   "matrices, numeric vectors, or pdMat objects"))
      }
    } else {
      ## repeat same value for all strata
      if (is.null(nStrata)) {
        if (inherits(object[[1]], "pdMat")) {
          object[[1]] <- pdConstruct(object[[1]], value, form, nam, data)
        } else {
          ## uninitialized
          object[[1]] <- pdMat(value, form, nam, data)
        }
      } else {
        if (length(object) == 1) {
          obj <- vector("list", nStrata)
          names(obj) <- stratLevs
          if (inherits(object[[1]], "pdMat")) {
            for (i in 1:nStrata) {
              obj[[i]] <- pdConstruct(object[[1]], value, form, nam, data)
            }
          } else {
            for (i in 1:nStrata) {
              obj[[i]] <- pdMat(value, form, nam, data)
            }
          }
          object <- obj
        } else {
          for(i in 1:nStrata) {
            object[[i]] <- pdConstruct(object[[i]], value, form, nam, data)
          }
        }
      }
    }
    if (!is.null(nStrata) && nStrata == 1) {
      ## single stratum, return pdMat component
      return(object[[1]])
    }
    attr(object, "stratForm") <- stratForm
    attr(object, "strata") <- strata
    attr(object, "stratLevs") <- stratLevs
    attr(object, "nStrata") <- nStrata
  }
  class(object) <- c("pdStrat", "pdMat")
  ## checking if parMap can be constructed
  if (isInitialized(object) && length(object) > 1) {
    parLen <- sapply(object, function(el) length(coef(el)))
    inf <- cumsum(c(1, parLen))[1:length(parLen)]
    sup <- cumsum(parLen)
    attr(object, "parMap") <-
      array(c(inf, sup), c(length(parLen), 2),
            list(names(object), c("inf","sup")))
  }
  ## checking consistency of formulas and names
  formList <- unlist(lapply(object,
                            function(el) {
                              deparse(as.vector(formula(el)[[2]]))
                            }))
  if (length(unique(formList)) > 1) {
    stop("All pdMat elements must have same formula")
  }
  namList <- unlist(lapply(object,
                           function(el) {
                             paste(Names, collapse = "")
                           }))
  if (length(unique(namList)) > 1) {
    stop("All pdMat elements must have same names")
  }
  object
}

pdFactor.pdStrat <-
  function(object)
{
  unlist(lapply(as.list(object), pdFactor))
}
  
pdMatrix.pdStrat <-
  function(object, factor = FALSE)
{
  lapply(as.list(object), pdMatrix, factor = factor)
}

###*# Methods for standard generics

asNatural.pdStrat <-
  function(object, unconstrained = TRUE)
{
  val <- lapply(as.list(object), asNatural)
  attributes(val) <- attributes(object)
  class(object) <- c("pdStrat", "pdMat")
  val
}

coef.pdStrat <-
  function(object, unconstrained = TRUE)
{
  if (!isInitialized(object)) return(numeric(0))
  val <- lapply(as.list(object), coef,  unconstrained = unconstrained)
  if (unconstrained) return(unlist(val))
  if (all(sapply(val, function(el) length(el) == 1 && !is.matrix(el)))) {
    return(unlist(val))
  }
  if (all(sapply(val, is.matrix))) {
    return(val)
  }
  strLev <- attr(object, "stratLevs")
  namCoef <- unlist(lapply(val, names))
  lenCoef <- sapply(val, length)
  nams <- paste(namCoef, rep(strLev, lenCoef), sep = ":")
  val <- unlist(val)
  names(val) <- nams
  val
}

"coef<-.pdStrat" <-
  function(object, value)
{
  if (is.null(attr(object, "nStrata"))) {
    stop("Cannot assign coefficients before defining number of strata")
  }
  if (is.null(parMap <- attr(object, "parMap"))) {
    stop("Need parameter map")
  }
  if (length(value) != parMap[nrow(parMap), ncol(parMap)]) {
    stop("Value of incorrect length")
  }
  for (i in names(object)) {
    coef(object[[i]]) <- value[parMap[i,1]:parMap[i,2]]
  }
  object
}

Dim.pdStrat <-
  function(object)
{
  aux <- as.list(object)[[1]]
  if (inherits(aux, "pdMat")) {
    Dim(aux)
  } else {
    NULL
  }
}

formula.pdStrat <-
  function(object, asList)
{
  aux <- as.list(object)[[1]]
  if (inherits(aux, "pdMat")) {
    formula(aux)
  } else {
    NULL
  }
}

isInitialized.pdStrat <-
  function(object)
{
  aux <- as.list(object)[[1]]
  if (inherits(aux, "pdMat")) {
    isInitialized(aux)
  } else {
    NULL
  }
}

logDet.pdStrat <-
  function(object)
{
  sapply(as.list(object), logDet)
}

"matrix<-.pdStrat" <-
  function(object, value)
{
  if (data.class(value) == "list") {
    ## list of matrices
    if (length(object) != length(value)) {
      stop("Must have the same number of matrices as strata")
    }
    if (!is.null(namVal <- names(value))) {
      if (any(is.na(match(namVal, names(object))))) {
        stop("Nonexistent stratum names in value")
      }
      for(i in namVal) {
        matrix(object[[i]]) <- value[[i]]
      }
    } else {                            # assume in order
      for(i in seq(along = value)) {
        matrix(object[[i]]) <- value[[i]]
      }
    }
  } else {
    ## single matrix for all pd structures
    for(i in seq(along = object)) {
      matrix(object[[i]]) <- value
    }
  }
  if (is.null(attr(object, "parMap"))) {
    parLen <- sapply(object, function(el) length(coef(el)))
    inf <- cumsum(c(1, parLen))[1:length(parLen)]
    sup <- cumsum(parLen)
    attr(object, "parMap") <-
      array(c(inf, sup), c(length(parLen), 2),
            list(names(object), c("inf","sup")))
  }
  object
}

Names.pdStrat <-
  function(object)
{
  aux <- as.list(object)[[1]]
  if (!is.null(attr(object, "nStrata")) && inherits(aux, "pdMat")) {
    Names(aux)
  } else {
    NULL
  }
}

"Names<-.pdStrat" <-
  function(object, ..., value)
{
  for(i in seq(along = object)) {
    Names(object[[i]]) <- value
  }
  object
}

print.summary.pdStrat <-
  function(x, sigma = 1, rdig = 3, Level = NULL, resid = FALSE, ...)
  ## resid = TRUE causes an extra row to be added
{
  allSame <- length(unique(sapply(x,
                                  function(el) attr(el, "structName")))) == 1
  if (!any(sapply(x, is.list))) {
    if (!(is.null(form <- attr(x, "formula")))) {
      cat(paste(" Formula: "))
      if (inherits(form, "formula")) {
        cat(deparse(as.vector(form)))
        if (!is.null(Level)) { cat( paste( " |", Level ) ) }
      } else {
        if (length(form) == 1) {
          cat(deparse(as.vector(form[[1]])))
          if (!is.null(Level)) { cat( paste( " |", Level ) ) }
        } else {
          cat(deparse(lapply(form,
                             function(el) as.name(deparse(as.vector(el))))))
          cat("\n Level:", Level)
        }
      }
      cat( "\n" )
    }
    if (ncol(x[[1]]) > 1 && allSame) {
      cat(paste(" Structure: ", attr(x[[1]], "structName"), " stratified by ",
                deparse(as.vector(attr(x, "stratForm")[[2]])), "\n", sep = ""))
    } else {
      cat(paste(" Strata: ", deparse(as.vector(attr(x, "stratForm"))),
                "\n", sep = ""))
    }
    for(i in names(x)) {
      cat("Stratum:", i, "\n")
      attr(x[[i]], "formula") <- NULL
      if (allSame) {
        attr(x[[i]], "structName") <- NULL
      }
      print(x[[i]], sigma = sigma)
    }
  } else {			
    if (allSame) {                      # composite structure
      cat(paste(" Composite Structure: ", attr(x[[1]], "structName"),
                " stratified by  ",
                deparse(as.vector(attr(x, "stratForm")[[2]])), "\n", sep =""))
      elName <- attr(x[[1]], "elementName")
      compNames <- names(x[[1]])
      for(i in names(x)) {
        cat("Stratum:", i, "\n")
        for (j in seq(along = x[[i]])) {
          cat(paste("\n ", elName, " ", j, ": ", compNames[j], "\n", sep = ""))
          print(x[[i]][[j]], sigma = sigma, Level = Level,
                resid = FALSE, ...)
        }
      }
    } else {
      cat(paste(" Strata: ~", deparse(as.vector(attr(x, "strataForm")[[2]])),
                "\n", sep = ""))
      for(i in names(x)) {
        cat("Stratum:", i, "\n")
        print(x[[i]], sigma = sigma)
      }
    }
  }
  if (resid) {
    cat(paste(" Within-group standard deviation:", format(sigma)))
    cat("\n")
  }
  invisible(x)
}

solve.pdStrat <-
  function(a, b)
{
  for(i in seq(along = a)) {
    a[[i]] <- solve(a[[i]])
  }
  a
}

summary.pdStrat <-
  function(object, structName)
{
  val <- lapply(as.list(object), summary)
  attr(val, "stratForm") <- attr(object, "stratForm")
  attr(val, "formula") <- formula(object)
  class(val) <- "summary.pdStrat"
  val
}

### Local variables:
### mode: S
### End:


