#!/opt/exp/bin/expectk -f
set VS_PATH /home/god/linux/verisoft/bin

# NOTE: update the previous lines if necessary!


# Copyright (c) 1996-2006 by Lucent Technologies. All Rights Reserved.
# This software is for evaluation purposes and educational use only. 
# No guarantee is expressed or implied by the distribution of this code. 
# Software written by Patrice Godefroid.                                 
# Send bug-reports, comments and questions to: god@bell-labs.com        

#
# simul.tcl: Execute a.out in current dir, capture the number of processes
#   in the system and their pids, create a debugger proc for each proc,
#   initialize debugger in the appropriate way (set breakpoint etc.),
#   get the list of proc that have an enabled trans.
#
#   Includes a graphical interface.
#
#   an error.path file can be passed as arg; the simulator is then
#   executed in guided simulation mode.
#   (it is possible to escape from error.path at any time.)
#

#
# Notes on expect programming:
#
# - for debugging expect, use "exp_internal 1" (chap 7)
# - if too slow, there is probably an expiration of timeout and
#   error in pattern matching
# - use "log_user 0" to hide internal conversations
# - use \n, not \r, when interacting with user
# - expect may be interrupted by commands triggered by X events!!!
# - when timeout expires, the expect command *always* terminates!!!

#
# Warning: new version of gdb reacts to only the first ^C !
#  So, one should not rely on this feature... Fixed on Jan 27, 1997.
#

#
# Functions and procedures.
#

#
# quit:  cleans the system and then exit
#

proc quit {} {
	global number_of_pids
	global array_of_spawn_ids
	global debugger_prompt
        global verisoft_spawn_id

	# destroy all widgets first
	# (such a way, no interruptions by other commands)

	destroy .

    # then detach the gdb processes (necessary for sigusr2 option)

    for {set i 1} {$i <= $number_of_pids} {incr i} {
		
	exp_send -i $array_of_spawn_ids($i) "detach\r"
	expect {
	    -i $array_of_spawn_ids($i)
	    $debugger_prompt { }
	    timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after detach! (timeout 99)\nDiagnosis:\n$expect_out(buffer)" ;}
	    eof { send_user "VeriSoft_exp Error: can't get debugger prompt after detach! (eof 09)\nDiagnosis:\n$expect_out(buffer)" ; }
	}
	
    }

    # terminate the debugger processes 

    for {set i 1} {$i <= $number_of_pids} {incr i} {
	
	# kill debugger process (hard kill)
	catch {exec kill -9 [exp_pid -i $array_of_spawn_ids($i)]}
	
    }

	# terminate verisoft by sending INT signal
	# (such a way, divergences are handled correctly as well)

	exec kill -INT [exp_pid -i $verisoft_spawn_id]

	# wait for terminaison of verisoft
	# (otherwise, the system may not be cleaned)

	expect -i $verisoft_spawn_id "VeriSoft" ;# work faster this way !?!
	wait 

	exit 1
}


# get_file_and_line:  gets the current state of the process pid
#  by using gdb.
# (the current file is in variable array_of_files($pid),
# the current line is in variable array_of_lines($pid) and
# the stack frame where to evaluate variables is in 
# array_of_stack_frames($pid).)
# this is done by using the command "backtrace".

proc get_file_and_line {pid} {
	global array_of_files
	global array_of_lines
	global array_of_stack_frames
	global array_of_spawn_ids
	global debugger_prompt

        set get_file_and_line_error 0

	# do not use the "info line" command of gdb:
	# its output is not very reliable...

	exp_send -i $array_of_spawn_ids($pid) "bt\r"

	expect {
		-i $array_of_spawn_ids($pid)
		-re "#(\[0-9]*) \[^#]* at (\[^\r]*):(\[0-9]*)\r" {
			set array_of_stack_frames($pid) $expect_out(1,string)
			set array_of_files($pid) $expect_out(2,string)
			set array_of_lines($pid) $expect_out(3,string)
		}
	        "corrupt stack" {
		        set get_file_and_line_error 1
	        }
		timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get file and line! (timeout 1)\nThe last process might have terminated abnormally without executing 'exit'. \nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get file and line! (eof 1)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}
	expect {
		-i $array_of_spawn_ids($pid)
		$debugger_prompt { }
		timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt! (timeout 2)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get debugger prompt! (eof 2)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

    if {$get_file_and_line_error == 1} {
	exp_send -i $array_of_spawn_ids($pid) "ni 3\r"
	get_file_and_line $pid
    }
}

#
# get_toss_choice_from_user pops up a window with the different
# values that can be returned by a VS_toss function call, and 
# returns the value selected by the user.
#

proc get_toss_choice_from_user {pid} {
	global nbr_of_enabled_trans
	global reply ;# reply has to be a global variable!
	global simul_pid
	global simul_toss_choice

	set w .proc_$pid.scroll.toss_$pid
	catch {destroy $w}
	toplevel $w
	wm title $w "VeriSoft Simulator"
	wm iconname $w "VeriSoft Simulator"

	label $w.label -text "Toss choice for Process $pid"
	pack $w.label -side top -fill x -pady 2m

	for {set i 0} {$i < $nbr_of_enabled_trans($pid)} {incr i} {
		button $w.$i -text "$i" -command "set reply $i; destroy $w"
		pack $w.$i -side top -fill x
	}

	if {$simul_pid != 0} {
		$w.$simul_toss_choice configure -background cyan
	}

	grab set .proc_$pid.scroll  ;# focus the attention to $w and .proc_$pid.scroll
	tkwait window $w ;# wait until $w is destroyed
	grab release .proc_$pid.scroll

	if {$reply != $simul_toss_choice} {set simul_pid 0} ;# quit guided simulation

	return $reply ;# return choice of the user

}


#
# get_next_simul_pid: sets simul_pid and simul_toss_choice 
#   to the next value read from current_trace
#

proc get_next_simul_pid {} {
	global simul_pid
	global simul_toss_choice
	global number_of_pids
	global nbr_of_enabled_trans
	global current_trace
	global PC

	set next [expr $PC+1]
	set simul_pid [lindex $current_trace($next) 0]
	if {($simul_pid < 1) || ($simul_pid > $number_of_pids)} {
		set simul_pid 0
	} else {
		set simul_toss_choice [lindex $current_trace($next) 1]
		if {($simul_toss_choice < 0) || ($simul_toss_choice >= $nbr_of_enabled_trans($simul_pid))} {
			signal_error "Error: cannot follow this scenario !"
			set simul_pid 0
		}
	}

}


#
# execute: exectues gdb command cmd on process pid,
#  and waits for next "global state"
#  sss: toss_choice added as argument

proc gdb_execute {pid cmd {toss_choice -1} } {
	global nbr_of_enabled_trans
	global choice_of_verisoft
	global array_of_spawn_ids
	global debugger_prompt
	global number_of_pids
	global stable_state
	global array_of_pids
	global simul_pid
	global simul_toss_choice
	global next_event
	global PC
	global current_trace
	global sss_vs_exists
        global VS_divergence_limit

	if {$nbr_of_enabled_trans($pid) <= 0} return

	if {$stable_state == 1} {

		if {$pid != $simul_pid} {set simul_pid 0} ;# quit guided simulation

	        if {$toss_choice < 0} {
		    set toss_choice 0 ;# not guided by sss view
		    if {$nbr_of_enabled_trans($pid) > 1} {
			if {$next_event == 0} {
			    set toss_choice [get_toss_choice_from_user $pid]
			} else {
			    set toss_choice $simul_toss_choice
			}
		    }
		} else {
		    # check whether the transition requested by sss view is reasonable
		    if {($toss_choice < 0) || ($toss_choice >= $nbr_of_enabled_trans($pid)) || ($pid < 1) || ($pid > $number_of_pids)} {
			signal_error "Error: cannot follow this scenario in state space view!"
			return
		    }
		}

		if {$sss_vs_exists == 1} {
			.sss.v.transition do $pid $toss_choice
		}

		# if non-guided simulation, save choice of user in current_trace

		if {$simul_pid == 0} {
			set current_trace([expr $PC+1]) "$pid $toss_choice"
			set current_trace([expr $PC+2]) "0" ;# mark end of trace
		}

		# send the choice to verisoft first!

		exp_send "[expr $choice_of_verisoft($pid)+$toss_choice]\r"

	}

	# send command to gdb
	exp_send -i $array_of_spawn_ids($pid) "$cmd\r" 

	# VERY IMPORTANT WARNING: expect may be interrupted by new X events!!!
	# to prevent the following expect to be interrupted,
	# we grab on the quit button of the current process window.
	# consequently, the user can only press this button while this
	# script is executing the following expect command

	grab set .proc_$pid.buttons.quit ;# prevent interruptions

	set timeout $VS_divergence_limit

	expect {
		-i $array_of_spawn_ids($pid)
		"Breakpoint 1" {
			expect {
				-i $array_of_spawn_ids($pid)
				$debugger_prompt { }
				timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after breakpoint! (timeout 3)\nDiagnosis:\n$expect_out(buffer)" ; quit }
				eof { send_user "VeriSoft_exp Error: can't get debugger prompt after breakpoint! (eof 3)\nDiagnosis:\n$expect_out(buffer)" ; quit }
				}
			# debbugger prompt is consumed by the previous expect command

			set stable_state 1 ;# we are in a global state again

			expect {
				-re "Next transition is (\[^\r]*)\r" {
					# $expect_out(1,string) contains the label of next event
					# $pid contains the pid of the next process to be executed

					app_comm "Process_$pid" $expect_out(1,string) "text"
				}
				timeout { expect *; send_user "VeriSoft_exp Error: can't get next transition! (timeout 4)\nDiagnosis:\n$expect_out(buffer)" ; quit }
				eof { send_user "VeriSoft_exp Error: can't get next transition! (eof 4)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			}

			get_next_enabled_trans ;# computes the new set of enabled trans

			if {$simul_pid != 0} { get_next_simul_pid }

			if {$next_event != 2} {

				get_file_and_line $pid ;# update the file and line of last active process
				for {set i 1} {$i <= $number_of_pids} {incr i} {
					update_window $i
				}
			}
		}
		$debugger_prompt { 
			# we are in a non stable state
			# refresh the window of current process
			get_file_and_line $pid
			update_window $pid
			if {($stable_state == 1) && (($cmd == "step") || ($cmd == "next"))} {
				set stable_state 0 ;# we are in the first intermediate state

				for {set i 1} {$i <= $number_of_pids} {incr i} {
					if {$i != $pid} {
						set nbr_of_enabled_trans($i) -1
						update_window $i
					}
				} ;# this paints all the other processes in yellow
			}
		}
		timeout {
			# send ^C to the gdb process
			exec kill -INT [exp_pid -i $array_of_spawn_ids($pid)]
			signal_error "Divergence!\n"
			set simul_pid 0 ;# quit guided simulation
			set nbr_of_enabled_trans($pid) 1 ;# in case last trans was a toss
			set cmd "next" ;# necessary when eating debugger_prompt after next command
			exp_continue
		}
		eof {
			# send ^C to the gdb process
			exec kill -INT [exp_pid -i $array_of_spawn_ids($pid)]
			signal_error "Divergence!\n"
			set simul_pid 0 ;# quit guided simulation
			set nbr_of_enabled_trans($pid) 1 ;# in case last trans was a toss
			set cmd "next" ;# necessary when eating debugger_prompt after next command
			exp_continue
		}
	}
	grab release .proc_$pid.buttons.quit ;# release interruptions
}


#
# gdb_print: prints the value of a var in a separate window.
#

proc gdb_print {pid} {
	global array_of_spawn_ids
	global debugger_prompt
	global print_counter
	global stable_state
	global array_of_stack_frames

	# return immediately if there is no selection;
	# otherwise, the selection is stored in var_name

	if {[catch {selection get} var_name]} return

	grab set .proc_$pid.buttons.quit ;# prevent interruptions

	set w .proc_$pid.print_$print_counter
	catch {destroy $w}
	toplevel $w
	wm title $w "VeriSoft Simulator"
	wm iconname $w "VeriSoft Simulator"

	incr print_counter

	# if in a stable state, select first the right stack frame

	if {$stable_state == 1} {
		exp_send -i $array_of_spawn_ids($pid) "frame $array_of_stack_frames($pid)\r"
		expect {
			-i $array_of_spawn_ids($pid)
			$debugger_prompt { }
			timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after frame! (timeout 5)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			eof { send_user "VeriSoft_exp Error: can't get debugger prompt after frame! (eof 5)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			}
	}

	exp_send -i $array_of_spawn_ids($pid) "print $var_name\r"

	expect {
		-i $array_of_spawn_ids($pid)
		-re "= (\[^\r]*)\r" { }
		-re "\n(\[^\r]*)\r" { }
		timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get value of variable! (timeout 6)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get value of variable! (eof 6)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

	label $w.label -text "Value of \"$var_name\" in Process $pid:"
	label $w.value -text "$expect_out(1,string)"
	pack $w.label $w.value -side top -fill x -pady 2m

	expect {
		-i $array_of_spawn_ids($pid)
		$debugger_prompt { }
		timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after print! (timeout 7)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get debugger prompt after print! (eof 7)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

	button $w.button -text "Dismiss" -command "destroy $w"
	pack $w.button -side bottom -pady 2m

	grab release .proc_$pid.buttons.quit ;# release interruptions

}


# textloadfile --
# This procedure below loads a file into a text widget, discarding
# the previous contents of the widget. Tags for the old widget are
# not affected, however.
#
# Arguments:
# w -		The window into which to load the file.  Must be a
#		text widget.
# file -	The name of the file to load.  Must be readable.

proc textloadfile {w file} {
    if {[file exists $file] == 0} {
	puts "VeriSoft Warning: can't find source file $file !"
	$w delete 1.0 end
	$w insert end "VeriSoft Warning: can't find source file $file !"
	return
    }
    set f [open $file]
    $w delete 1.0 end
    while {![eof $f]} {
	$w insert end [read $f 10000]
    }
    close $f
}

#
# create_window creates a window .proc_$pid
#

proc create_window {pid} {
	global array_of_files
	global array_of_lines
	global nbr_of_enabled_trans

	set w .proc_$pid
	catch {destroy $w}
	toplevel $w
	wm title $w "VeriSoft Simulator - Process $pid"
	wm iconname $w "VeriSoft Simulator - Process $pid"

	frame $w.buttons
	pack  $w.buttons -side top -fill x -pady 2m

	button $w.buttons.step -text "Step" -command "gdb_execute $pid step"
	button $w.buttons.next -text "Next" -command "gdb_execute $pid next"
	button $w.buttons.continue -text "Continue" -command "gdb_execute $pid continue"
	button $w.buttons.print -text "Print" -command "gdb_print $pid"
	button $w.buttons.quit -text "Quit" -command quit

	pack $w.buttons.step $w.buttons.next $w.buttons.continue \
	$w.buttons.print $w.buttons.quit -side left 

	text $w.text -relief sunken -bd 2 -yscrollcommand "$w.scroll set" -setgrid 1 \
		-height 15 -width 60 -background white
	scrollbar $w.scroll -command "$w.text yview"
	update_window $pid
	pack $w.scroll -side right -fill y
	pack $w.text -expand 1 -fill both

}

#
# update_window updates the content of the text part of the
#  corresponding window
#

proc update_window {pid} {
	global array_of_files
	global array_of_lines
	global nbr_of_enabled_trans
	global simul_pid

	set w .proc_$pid

	textloadfile $w.text "$array_of_files($pid)"

	# create a tag for the line corresponding to the current state
	# and highlight it 

	$w.text tag add current_line $array_of_lines($pid).0 $array_of_lines($pid).end

	if {$nbr_of_enabled_trans($pid) > 0} {
		$w.text tag configure current_line -background green
	} elseif {$nbr_of_enabled_trans($pid) == 0} {
		$w.text tag configure current_line -background red
	} else {
		$w.text tag configure current_line -background yellow
	}

	if {$pid == $simul_pid} {
		$w.text tag configure current_line -background cyan
	}

	# get the nbr of lines (height) h of the current window
	scan [wm geometry $w] "%dx%d+%d+%d" width h x y

	# On sunOS 5.x, h is sometimes 0 just after the creation of
	# the window; fix this with the following
	if {$h == 0} {
	    set h 15
	}

	# display the current line in middle of window
	# (note: no need to test whether the arg of yview is positive
	# or not too large)

	$w.text yview [expr $array_of_lines($pid)-($h/2)-1]
}


#
# get_next_enabled_trans gets the next set of enabled transitions 
# from verisoft, and updates the vars nbr_of_enabled_trans($pid).
# (has to be executed before update_window)
#

proc get_next_enabled_trans {} {
	global number_of_pids
	global choice_of_verisoft
	global nbr_of_enabled_trans


	# the value of spawn_id is always $verisoft_spawn_id

	for {set i 1} {$i <= $number_of_pids} {incr i} {
		set nbr_of_enabled_trans($i) 0
		set choice_of_verisoft($i) 0
		}

	set choice 1

	# get the next set of enabled transitions from verisoft

	expect {
		-re "Process_(\[0-9]*):" { 
			if {$choice_of_verisoft($expect_out(1,string)) == 0} { 
				set choice_of_verisoft($expect_out(1,string)) $choice
				}
			incr nbr_of_enabled_trans($expect_out(1,string))
			incr choice
			exp_continue
			}
		"assertion violation" {
			# an assertion violation has been detected by VeriSoft.
			# the simulator is then in a state where no transition 
			# is executable, but where the value of variables may
			# be examined.
			# signal the error to the user.

			signal_error "Assertion violation!"

			exp_continue ;# because "Next transition?" has to be eaten
		}
		"abort" {
			# an abort has been detected by VeriSoft.
			# the simulator is then in a state where no transition 
			# is executable, but where the value of variables may
			# be examined.
			# signal the error to the user.

			signal_error "Abort!"

			exp_continue ;# because "Next transition?" has to be eaten
		}
		"deadlock" {
			signal_error "Deadlock!"
			exp_continue ;# because "Next transition?" has to be eaten
		}
		"Next transition? " { }
		-re "Error(\[^\r]*)\r" { send_user "$expect_out(buffer)" ; quit }
		-re "Warning(\[^\r]*)\r" { send_user "$expect_out(buffer)" ; quit }
		timeout { expect *; send_user "VeriSoft_exp Error: can't communicate with VS_scheduler! (timeout 8)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't communicate with VS_scheduler! (eof 8)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

}

proc signal_error {message} {
	global error_counter

	set w .error_$error_counter
	catch {destroy $w}
	toplevel $w
	wm title $w "VeriSoft Simulator"
	wm iconname $w "VeriSoft Simulator"

	label $w.bitmap -bitmap error -foreground red -width 8c
	message $w.message -text "$message" -foreground red -width 6c
	pack $w.bitmap $w.message -side top -fill x -pady 2m
	button $w.button -text "Dismiss" -command "destroy $w"
	pack $w.button -side bottom -pady 2m

	incr error_counter
}


#
# load_trace: load trace from error_file to current_trace;
#  the last line of current_trace has 0 for first element
#  (event_cnt is the number of events displayed in trace view)
#

proc load_trace {filename} {
	global current_trace

	set error_file [open $filename r]

	for {set i 1} {$i > 0} {incr i} {
		if {[gets $error_file current_trace($i)] < 0} {
			set current_trace($i) "0" ;# mark the last line
			break
		}
	}
	close $error_file
}

#
# save_trace: save trace from current_trace to filename;
#  the last line of current_trace has 0 for first element
#

proc save_trace {filename} {
    global current_trace

    set error_file [open $filename w]

    set i 1
    while {[lindex $current_trace($i) 0] > 0} {
	puts $error_file $current_trace($i)
	incr i
    }

    close $error_file
}



#
# reset_trace: prevent interruptions by the user,
#  detach the gdb processes, kill the system processes,
#  recreate the system processes, reattach the gdb processes,
#  update the process windows,
#  wait for next command from user

proc reset_trace {} {
	global number_of_pids
	global executable_file
	global array_of_spawn_ids
	global array_of_pids
	global debugger_prompt
	global verisoft_spawn_id
	global spawn_id
	global stable_state
	global simul_pid
	global simul_toss_choice
	global PC
	global event_cnt
	global sss_vs_exists

	grab set .menu.quit ;# prevent interruptions

	# then detach the gdb processes 

	for {set i 1} {$i <= $number_of_pids} {incr i} {

		exp_send -i $array_of_spawn_ids($i) "detach\r"
		expect {
			-i $array_of_spawn_ids($i)
			$debugger_prompt { }
			timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after detach! (timeout 9)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			eof { send_user "VeriSoft_exp Error: can't get debugger prompt after detach! (eof 9)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			}

		}

	# if stable_state==0, the scheduler is not ready to receive orders;
	# therefore, terminate verisoft and reinitialize it
	# (such a way, divergences are handled correctly as well)

	if {$stable_state == 0} {
		exec kill -INT [exp_pid]
		# wait for terminaison of verisoft
		# (otherwise, the system may not be cleaned)
		expect "VeriSoft" ;# work faster this way !?!
		wait 
		# recreate verisoft (i.e., scheduler process)
		spawn $executable_file
		set verisoft_spawn_id $spawn_id
		# WARNING! spawn_id and verisoft_spawn_id must be global variables!!!
	} else {
		# tell verisoft (scheduler) to backtrack
		exp_send "0\r" 
	}

	# reinitialize variables

	set number_of_pids 0
	set stable_state 1 
	set simul_pid 0 
	set simul_toss_choice 0

	if {$sss_vs_exists == 1} {
		.sss.v.transition clear
	}

	# get the pids of the processes of the system from verisoft

	set timeout 10

	expect {
		-re "pid: (\[0-9]*)\r" { 
			incr number_of_pids
			set array_of_pids($number_of_pids) $expect_out(1,string)
			exp_continue
			}
		"Enabled" { }
		timeout { expect *; send_user "VeriSoft_exp Error: can't get pid's of system processes! (timeout 10)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get pid's of system processes! (eof 10)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

	# re-initialize gdb processes

	for {set i 1} {$i <= $number_of_pids} {incr i} {

		exp_send -i $array_of_spawn_ids($i) "attach $array_of_pids($i)\r"
		expect {
			-i $array_of_spawn_ids($i)
			$debugger_prompt { }
			timeout { expect -i $array_of_spawn_ids($pid) *; send_user "VeriSoft_exp Error: can't get debugger prompt after attach! (timeout 11)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			eof { send_user "VeriSoft_exp Error: can't get debugger prompt after attach! (eof 11)\nDiagnosis:\n$expect_out(buffer)" ; quit }
			}

		# when gdb attaches to the running proc, it sends a ^C to it;
		# the proc is now controled by gdb

		get_file_and_line $i
		}

	get_next_enabled_trans

	set PC 0 ;# needed for get_next_simul_pid (next line)
	get_next_simul_pid


	for {set i 1} {$i <= $number_of_pids} {incr i} {
		update_window $i
	}
	grab release .menu.quit ;# release interruptions

}

#
# Initializations of variables.
#

log_user 0 ;# when 0, hide internal conversations

exp_internal 0 ;# when 0, hide expect pattern matching

set number_of_pids 0
set stable_state 1 
set print_counter 0
set error_counter 0

set executable_file "./a.out"

if {[file executable $executable_file] == 0} {
	send_user "$executable_file is not executable!\n"
	exit
}

# debugger is gdb

set debugger_prompt "(gdb) "

# simul_pid contains the pid of the next transition to be executed
# given in an error trace, in guided simulation mode.

set current_trace(1) "0"
set simul_pid 0 ;# by default, no guided simulation mode
set simul_toss_choice 0
set next_event 0 ;# if !=0, then execute next event of guided simulation
		  # if !=2, then update display in process windows
set VS_divergence_limit 0

# create trace view
source $VS_PATH/trace-view.tcl

grab set .menu.quit ;# prevent interruptions

#
# Start VeriSoft Simulator.
#


# run verisoft IN SIMULATION MODE AND WITH SYSTEM COMPILED WITH -g !!!
# (use -DSIMUL with verisoft.c)

spawn $executable_file
set verisoft_spawn_id $spawn_id

# get VS_divergence_limit and the pids of the processes of the system from verisoft

set timeout 10

expect {
        -re "VS_divergence_limit is (\[0-9]*)\r" {
	        set VS_divergence_limit $expect_out(1,string)
		exp_continue
	        }
	-re "pid: (\[0-9]*)\r" { 
		incr number_of_pids
		set array_of_pids($number_of_pids) $expect_out(1,string)
		exp_continue
		}
	"Enabled" { }
	timeout { expect *; send_user "VeriSoft_exp Error: can't communicate with VS_scheduler! (timeout 12)\nDiagnosis:\n$expect_out(buffer)\n(For more diagnosis info, try running VeriSoft without '-simul'.)\n" ; exit 1 }
	eof { send_user "VeriSoft_exp Error: can't communicate with VS_scheduler! (eof 12)\nDiagnosis:\n$expect_out(buffer)\n(For more diagnosis info, try running VeriSoft without '-simul'.)\n" ; exit 1 }
	}

if {$VS_divergence_limit == 0} {
	send_user "divergence_limit is 0: it should be set to a positive value in system_file.VS!\n"
	exit
}

# create and initialize gdb processes

for {set i 1} {$i <= $number_of_pids} {incr i} {

	spawn gdb -quiet $executable_file $array_of_pids($i)
	set array_of_spawn_ids($i) $spawn_id
	expect {
		$debugger_prompt { }
		timeout { expect *; send_user "VeriSoft_exp Error: can't get debugger prompt after initialization! (timeout 13)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get debugger prompt after initialization! (eof 13)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}
	# force gdb to unbuffer its output (thanks to Angelica Verstraten!)
	exp_send "set height 0\r"
	expect {
		$debugger_prompt { }
		timeout { expect *; send_user "VeriSoft_exp Error: can't get debugger prompt after 'set height 0'! (timeout 113)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get debugger prompt after 'set height 0'! (eof 113)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}

	# when gdb attaches to the running proc, it sends a ^C to it;
	# the proc is now controled by gdb

	exp_send "b VS_pause_semwait\r"
	expect {
		$debugger_prompt { }
		timeout { expect *; send_user "VeriSoft_exp Error: can't get debugger prompt after setting breakpoint! (timeout 14)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		eof { send_user "VeriSoft_exp Error: can't get debugger prompt after setting breakpoint! (eof 14)\nDiagnosis:\n$expect_out(buffer)" ; quit }
		}
	get_file_and_line $i
	}

# after the creation of the gdb processes, restore spawn_id

set spawn_id $verisoft_spawn_id

# create one window per process and initialize it
# WARNING: executes get_next_enabled_trans before update_window

get_next_enabled_trans

set PC 0 ;# value of current position in trace view 
	 # (always included between 0 and event_cnt)

if {$argc == 1} {
	# guided simulation mode  and
	# $argv contains the error.path file to follow

	if {[catch {load_trace $argv} error_msg] == 1} {
		signal_error "$error_msg"
		set simul_pid 0
	        set current_trace(1) "0"
	} else {
		get_next_simul_pid
	}
}

for {set i 1} {$i <= $number_of_pids} {incr i} {
	create_window $i
	# trace view
	add_proc "Process_$i"
}

# intilialization of trace view

set event_cnt 0 ;# number of events in trace view

set row [expr $Begin+$Stepsz]
set PC_obj \
	[.pad.c create line \
	[expr $Colwidth-(2*$W2)] [expr $row - $H1] \
	[expr $nrproc*$Colwidth+(2*$W2)] [expr $row - $H1] \
	-tags "PC" -width 3 -fill red ]
.pad.c lower $PC_obj

# create state-space view only if sss.VS file exists

if {[ file exists sss.VS ]} {
    if {[info sharedlibextension]!= ".so"} {
	puts "Tcl dynamic loading is disabled -- can't load .so files!"
	puts "No state space view."
	set sss_vs_exists 0
    } else {
	source $VS_PATH/sss.tcl
	set sss_vs_exists 1
    }
} else {
	set sss_vs_exists 0
}

grab release .menu.quit ;# release interruptions

#
# The script stops here. The simulator reacts to X events.
#
